package ru.yandex.chemodan.app.djfs.core.album;

import org.bson.types.ObjectId;
import org.joda.time.Instant;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.chemodan.app.djfs.core.db.pg.TransactionUtils;
import ru.yandex.chemodan.app.djfs.core.filesystem.AlbumRemoveActivity;
import ru.yandex.chemodan.app.djfs.core.filesystem.AlbumSetAttrActivity;
import ru.yandex.chemodan.app.djfs.core.filesystem.DjfsPrincipal;
import ru.yandex.chemodan.app.djfs.core.filesystem.DjfsResourceDao;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.DjfsResource;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.DjfsResourceArea;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.DjfsResourceId;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.FileDjfsResource;
import ru.yandex.chemodan.app.djfs.core.filesystem.operation.albumremove.AlbumRemoveCallbacks;
import ru.yandex.chemodan.app.djfs.core.filesystem.operation.albumsetattr.AlbumSetAttrCallbacks;
import ru.yandex.chemodan.app.djfs.core.notification.XivaPushGenerator;
import ru.yandex.chemodan.app.djfs.core.share.ShareInfoManager;
import ru.yandex.chemodan.app.djfs.core.user.DjfsUid;
import ru.yandex.chemodan.app.djfs.core.util.InstantUtils;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

public class PersonalAlbumManager extends AbstractAlbumManager {
    private static final Logger logger = LoggerFactory.getLogger(GeoAlbumManager.class);

    private final ListF<AlbumType> cleanTypes = Cf.list(AlbumType.PERSONAL, AlbumType.FAVORITES);

    public PersonalAlbumManager(
            DjfsResourceDao djfsResourceDao,
            ShareInfoManager shareInfoManager,
            AlbumItemDao albumItemDao,
            AlbumDao albumDao,
            AlbumDeltaDao albumDeltaDao,
            XivaPushGenerator xivaPushGenerator,
            TransactionUtils transactionUtils
    ) {
        super(
                albumDao,
                albumItemDao,
                albumDeltaDao,
                transactionUtils,
                xivaPushGenerator,
                shareInfoManager,
                djfsResourceDao
        );
    }

    @Override
    public AlbumType getAlbumType() {
        return AlbumType.PERSONAL;
    }

    public void removeAlbum(DjfsUid uid, Album album, AlbumRemoveCallbacks callbacks) {
        DjfsPrincipal principal = DjfsPrincipal.cons(uid);
        Tuple2<Boolean, Long> statusWithRevision = transactionUtils.executeInNewOrCurrentTransaction(uid, () -> {
            long currentRevision = getAndLockCurrentRevision(uid);

            AlbumRemoveActivity albumRemoveActivity = new AlbumRemoveActivity(principal, album);
            callbacks.callAfterAlbumRemoveOutsideTransaction(albumRemoveActivity);

            albumItemDao.removeAllItemsFromAlbum(uid, album.getId());

            currentRevision += 1;
            AlbumDelta removeAlbumDelta = AlbumUtils.createRemoveAlbumDelta(album, currentRevision);
            albumDeltaDao.insert(removeAlbumDelta);

            albumDao.delete(album);

            albumDeltaDao.updateCurrentRevision(uid, currentRevision);

            return Tuple2.tuple(true, currentRevision);
        });

        if (statusWithRevision._1) {
            xivaPushGenerator.sendAlbumsDatabaseChangedPush(uid, statusWithRevision._2);
            xivaPushGenerator.sendAlbumRemovedPush(uid, album);
        }
    }

    public void changePublicity(DjfsUid uid, Album album, boolean newPublicity, AlbumSetAttrCallbacks callbacks) {
        DjfsPrincipal principal = DjfsPrincipal.cons(uid);
        if (album.isPublic() == newPublicity) {
            return;
            // TODO А может тут выкидывать ошибку?
        }
        Tuple2<Boolean, Long> statusWithRevision = transactionUtils.executeInNewOrCurrentTransaction(uid, () -> {
            long currentRevision = getAndLockCurrentRevision(uid);
            currentRevision += 1;
            ListF<AlbumDeltaFieldChange> albumUpdateChanges = Cf.arrayList();
            ListF<AlbumParam> albumUpdateDB = Cf.arrayList();
            albumUpdateDB.add(AlbumParam.isPublic(newPublicity));
            albumUpdateChanges.addAll(AlbumUtils.generateSetPublicChangeFields(newPublicity));
            Instant now = Instant.now();
            albumUpdateChanges.addAll(AlbumUtils.generateSetDateModifyChangeFields(now));
            albumUpdateDB.add(AlbumParam.dateModified(now));
            albumDao.setParams(album, albumUpdateDB);
            AlbumSetAttrActivity albumSetAttrActivity = new AlbumSetAttrActivity(
                    principal, album, Option.empty(), Option.empty(), Option.empty(), Option.empty(), Option.of(newPublicity)
            );
            callbacks.callAfterAlbumSetAttrOutsideTransaction(albumSetAttrActivity);

            AlbumDelta updateAlbumDelta = AlbumUtils.createUpdateAlbumDelta(album, albumUpdateChanges, currentRevision);
            albumDeltaDao.insert(updateAlbumDelta);

            albumDao.updateAlbumRevision(album, currentRevision);
            albumDeltaDao.updateCurrentRevision(uid, currentRevision);
            return Tuple2.tuple(true, currentRevision);
        });

        if (statusWithRevision._1) {
            xivaPushGenerator.sendAlbumsDatabaseChangedPush(uid, statusWithRevision._2);
            xivaPushGenerator.sendAlbumPublicPush(uid, album);
        }
    }

    public void setAttrs(DjfsUid uid, Album album, Option<String> titleO, Option<String> rawCoverO,
                         Option<Double> coverOffsetYO, Option<String> rawLayoutO, Option<String> descriptionO,
                         Option<String> flagsO, Option<Long> fotkiAlbumIdO, AlbumSetAttrCallbacks callbacks) {
        DjfsPrincipal principal = DjfsPrincipal.cons(uid);
        Tuple2<Boolean, Long> statusWithRevision = transactionUtils.executeInNewOrCurrentTransaction(uid, () -> {
            long currentRevision = getAndLockCurrentRevision(uid);
            currentRevision += 1;

            Option<String> prevAlbumTitleO = titleO.isPresent() ? Option.of(album.getTitle()) : Option.empty();
            ListF<AlbumSetAttrUpdater> updaters = Cf.arrayList();
            updaters.addAll(setTitle(titleO));
            updaters.addAll(setLayout(rawLayoutO));
            updaters.addAll(setCover(uid, album, rawCoverO, coverOffsetYO));
            updaters.addAll(setCoverOffset(coverOffsetYO));
            updaters.addAll(setDescription(descriptionO));
            updaters.addAll(setFlags(getUniqFlagsList(flagsO)));
            updaters.addAll(setFotkiAlbumId(fotkiAlbumIdO));
            if (titleO.exists(title -> !title.equals(album.getTitle()))
                    || !rawCoverO.equals(album.getCoverId())
                    || !fotkiAlbumIdO.equals(album.getFotkiAlbumId())) {
                updaters.addAll(setDateModified());
            }
            saveDeltasAndUpdateAlbumDB(album, currentRevision, updaters);

            AlbumSetAttrActivity albumSetAttrActivity = new AlbumSetAttrActivity(
                    principal, album, rawCoverO, coverOffsetYO, prevAlbumTitleO, titleO, Option.empty()
            );
            callbacks.callAfterAlbumSetAttrOutsideTransaction(albumSetAttrActivity);

            albumDao.updateAlbumRevision(album, currentRevision);
            albumDeltaDao.updateCurrentRevision(uid, currentRevision);
            return Tuple2.tuple(true, currentRevision);
        });

        if (statusWithRevision._1) {
            Album updatedAlbum = albumDao.findAlbum(uid, album.getId()).get();
            xivaPushGenerator.sendAlbumsDatabaseChangedPush(uid, statusWithRevision._2);
            if (titleO.isPresent()) {
                xivaPushGenerator.sendAlbumTitleChangedPush(uid, updatedAlbum);
            } else if (rawCoverO.isPresent()) {
                xivaPushGenerator.sendAlbumCoverChangedPush(uid, updatedAlbum);
            }
        }
    }

    public void postProcessFiles(ListF<DjfsResourceId> resourceIds) {
        for (DjfsResourceId resourceId : resourceIds) {
            ListF<DjfsResource> resources = djfsResourceDao.find(resourceId);
            if (resources.isEmpty()) {
                continue;
            }

            DjfsResource resource = resources.first();
            if (!(resource instanceof FileDjfsResource)) {
                continue;
            }
            FileDjfsResource file = (FileDjfsResource) resource;

            DjfsResourceArea fileArea = file.getPath().getArea();
            if (!fileArea.equals(DjfsResourceArea.TRASH) && !fileArea.equals(DjfsResourceArea.HIDDEN)) {
                continue;
            }

            // WARNING: Here we have only files in trash, so we don't know about shared files and folders
            // and we clean album items for owner only
//            Option<ShareInfo> shareInfoO = shareInfoManager.getWithRoot(file);
//            if (shareInfoO.isPresent()) {
//                Assume.assumeTrue(shareInfoO.get().getOwnerUid().equals(file.getUid()));
//                // TODO: fetch all participants and clean resource from their albums
//            }

            DjfsUid uid = resourceId.getUid();
            String objId = resourceId.getFileId().getValue();

            ListF<AlbumItem> items = albumItemDao.find(uid, objId);
            if (items.isEmpty()) {
                continue;
            }

            SetF<ObjectId> albumIds = items.map(AlbumItem::getAlbumId).unique();
            MapF<ObjectId, Album> id2album = albumDao.findAlbums(uid, albumIds)
                    .filter(a -> cleanTypes.containsTs(a.getType())).toMap(a -> Tuple2.tuple(a.getId(), a));
            if (id2album.isEmpty()) {
                continue;
            }

            for (AlbumItem item : items) {
                Option<Album> albumO = id2album.getO(item.getAlbumId());
                if (!albumO.isPresent()) {
                    continue;
                }
                Album album = albumO.get();

                transactionUtils.executeInNewOrCurrentTransaction(uid, () -> {
                    albumItemDao.removeFromAlbum(uid, album.getId(), item.getObjectId());
                    albumDao.incrementAlbumRevision(album);
                });

                xivaPushGenerator.sendAlbumItemRemovedPush(uid, album, item, true);
            }
        }
    }

    @Override
    protected Option<Double> itemOrderIndex(FileDjfsResource file) {
        return Option.of(
                (double)
                        file    .getEtime()
                                .getOrElse(() -> {
                                    logger.info("No etime for file: " + file);
                                    return InstantUtils.toSecondsLong(file.getCreationTime());
                                }));
    }
}
