package ru.yandex.chemodan.app.djfs.core.album;

import java.util.Objects;

import org.bson.types.ObjectId;
import org.joda.time.Instant;
import org.postgresql.util.PSQLException;
import org.postgresql.util.ServerErrorMessage;
import org.springframework.dao.DataIntegrityViolationException;
import org.springframework.jdbc.core.RowMapper;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.CollectionF;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.chemodan.app.djfs.core.db.EntityAlreadyExistsException;
import ru.yandex.chemodan.app.djfs.core.db.pg.PgShardedDao;
import ru.yandex.chemodan.app.djfs.core.db.pg.PgShardedDaoContext;
import ru.yandex.chemodan.app.djfs.core.db.pg.ResultSetUtils;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.DjfsResourceId;
import ru.yandex.chemodan.app.djfs.core.user.DjfsUid;
import ru.yandex.misc.lang.Validate;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

/**
 * @author eoshch
 */
public class PgAlbumDao extends PgShardedDao implements AlbumDao {
    private static final Logger logger = LoggerFactory.getLogger(PgAlbumDao.class);

    private final static RowMapper<Album> M = (rs, rowNum) -> Album.builder()
            .id(new ObjectId(rs.getBytes("id")))
            .uid(DjfsUid.cons(rs.getLong("uid")))

            .title(rs.getString("title"))
            .coverId(Option.ofNullable(rs.getBytes("cover_id")).map(ObjectId::new))

            .coverOffsetY(ResultSetUtils.getDoubleO(rs, "cover_offset_y"))
            .description(ResultSetUtils.getStringO(rs, "description"))

            .publicKey(ResultSetUtils.getStringO(rs, "public_key"))
            .publicUrl(ResultSetUtils.getStringO(rs, "public_url"))
            .shortUrl(ResultSetUtils.getStringO(rs, "short_url"))

            .isPublic(rs.getBoolean("is_public"))
            .isBlocked(rs.getBoolean("is_blocked"))

            .blockReason(ResultSetUtils.getStringO(rs, "block_reason"))
            .flags(ResultSetUtils.getArrayO(rs, "flags", String.class).map(Cf::list))
            .layout(ResultSetUtils.getStringO(rs, "layout").map(AlbumLayoutType.R::fromValue))
            .dateCreated(ResultSetUtils.getInstantO(rs, "date_created"))
            .dateModified(ResultSetUtils.getInstantO(rs, "date_modified"))

            .socialCoverStid(ResultSetUtils.getStringO(rs, "social_cover_stid"))
            .fotkiAlbumId(ResultSetUtils.getLongO(rs, "fotki_album_id"))

            .revision(ResultSetUtils.getLongO(rs, "revision"))

            .hidden(ResultSetUtils.getBooleanO(rs, "hidden").getOrElse(false))
            .geoId(ResultSetUtils.getLongO(rs, "geo_id"))

            .type(AlbumType.R.fromValue(ResultSetUtils.getStringO(rs, "album_type").getOrElse("personal")))
            .isDescSorting(ResultSetUtils.getBooleanO(rs, "is_desc_sorting"))
            .albumItemsSorting(ResultSetUtils.getStringO(rs, "album_items_sorting"))
            .coverAuto(ResultSetUtils.getBooleanO(rs, "cover_auto"))
            .build();

    public final static RowMapper<ExtendedAlbum> EXT_M = (rs, rowNum) -> {
        Album album = M.mapRow(rs, rowNum);
        Option<DjfsResourceId> coverId =
                ResultSetUtils.getStringO(rs, "cover_obj_id").map(x -> DjfsResourceId.cons(album.getUid(), x));
        return new ExtendedAlbum(album, rs.getInt("items_count"), coverId);
    };

    public PgAlbumDao(PgShardedDaoContext context) {
        super(context);
    }

    @Override
    public void deleteAll(DjfsUid uid) {
        String sql = collectStats(uid) + " DELETE FROM disk.albums WHERE uid = :uid";
        jdbcTemplate(uid).update(sql, Cf.map("uid", uid));
    }

    @Override
    public void insert(Album album) {
        String sql = collectStats(album)
                + " INSERT INTO disk.albums (id, uid, title, cover_id, cover_offset_y, description, public_key, "
                + " public_url, short_url, is_public, is_blocked, block_reason, flags, layout, date_created, "
                + " date_modified, social_cover_stid, fotki_album_id, revision, album_type, hidden, geo_id, "
                + " cover_auto) "
                + " VALUES (:id, :uid, :title, :cover_id, :cover_offset_y, :description, :public_key, "
                + " :public_url, :short_url, :is_public, :is_blocked, :block_reason, :flags, :layout, :date_created, "
                + " :date_modified, :social_cover_stid, :fotki_album_id, :revision, :album_type::disk.album_type,"
                + " :hidden, :geo_id, :cover_auto)";

        MapF<String, Object> params = Cf.toMap(Tuple2List.fromPairs(
                "uid", album.getUid(),
                "id", album.getId().toByteArray(),
                "title", album.getTitle(),
                "cover_id", album.getCoverId().map(ObjectId::toByteArray).getOrNull(),
                "cover_offset_y", album.getCoverOffsetY().getOrNull(),
                "description", album.getDescription().getOrNull(),
                "public_key", album.getPublicKey().getOrNull(),
                "public_url", album.getPublicUrl().getOrNull(),
                "short_url", album.getShortUrl().getOrNull(),
                "is_public", album.isPublic(),
                "is_blocked", album.isBlocked(),
                "block_reason", album.getBlockReason().getOrNull(),
                "flags", album.getFlags().getOrNull(),
                "layout", album.getLayout().getOrNull(),
                "date_created", album.getDateCreated().getOrNull(),
                "date_modified", album.getDateModified().getOrNull(),
                "social_cover_stid", album.getSocialCoverStid().getOrNull(),
                "fotki_album_id", album.getFotkiAlbumId().getOrNull(),
                "revision", album.getRevision().getOrNull(),
                "hidden", album.isHidden(),
                "geo_id", album.getGeoId().getOrNull(),
                "album_type", album.getType().value(),
                "cover_auto", album.getCoverAuto()
        ));

        try {
            jdbcTemplate(album.getUid()).update(sql, params);
        } catch (DataIntegrityViolationException e) {
            Throwable cause = e.getCause();
            if (cause instanceof PSQLException) {
                ServerErrorMessage error = ((PSQLException) cause).getServerErrorMessage();
                if (error != null && (Objects.equals(error.getConstraint(), "pk_albums")
                        || Objects.equals(error.getConstraint(), "uk_albums_id")))
                {
                    logger.info("PgAlbumDao.insert(Album) handled exception for user "
                            + album.getUid() + " : ", e);
                    throw new EntityAlreadyExistsException(album.getId().toHexString(), e);
                }
            }
            throw e;
        }
    }

    @Override
    public ListF<Album> getAlbums(DjfsUid uid, AlbumType type) {
        String typeCondition = "album_type = :album_type::disk.album_type";
        if (type == AlbumType.PERSONAL) {
            typeCondition += " OR album_type IS NULL";
        }

        String sql = collectStats(uid)
                + " SELECT * FROM disk.albums AS a WHERE uid = :uid AND (" + typeCondition + ")";
        MapF<String, Object> params = Cf.map(
                "uid", uid,
                "album_type", type.value()
        );

        return jdbcTemplate(uid).query(sql, M, params);
    }

    @Override
    public ListF<ExtendedAlbum> getExtendedAlbums(DjfsUid uid, ListF<AlbumType> types) {
        String typeCondition;
        MapF<String, Object> params = Cf.map("uid", uid);
        if (types.size() == 1) {
            typeCondition = "album_type = :album_type::disk.album_type";
            params = params.plus1("album_type", types.single().value());
        } else {
            typeCondition = "album_type = ANY (ARRAY[ :album_types ]::disk.album_type[])";
            params = params.plus1("album_types", types.map(AlbumType::value));
        }

        if (types.containsTs(AlbumType.PERSONAL)) {
            typeCondition += " OR album_type IS NULL";
        }

        String sql = collectStats(uid)
                + " SELECT *,"
                + " (SELECT COUNT(*) FROM disk.album_items WHERE uid = uid AND album_id = a.id) AS items_count,"
                + " (SELECT obj_id FROM disk.album_items WHERE uid = uid AND id = a.cover_id) AS cover_obj_id"
                + " FROM disk.albums AS a WHERE uid = :uid AND (" + typeCondition + ")"
                + " ORDER BY a.date_modified DESC, a.id";

        return jdbcTemplate(uid).query(sql, EXT_M, params);
    }

    @Override
    public long getMaxAlbumVersion(DjfsUid uid) {
        String sql = collectStats(uid) + " SELECT MAX(revision) FROM disk.albums WHERE uid = ?";
        return jdbcTemplate(uid).queryForLong(sql, uid);
    }

    @Override
    public ListF<Album> findAlbums(DjfsUid uid, CollectionF<ObjectId> albumIds) {
        String sql = collectStats(uid)
            + " SELECT * FROM disk.albums WHERE uid = :uid AND id IN (:album_ids)";
        MapF<String, Object> params = Cf.map(
            "uid", uid,
            "album_ids", albumIds.map(ObjectId::toByteArray)
        );
        return jdbcTemplate(uid).query(sql, M, params);
    }

    @Override
    public Option<Album> findGeoAlbum(DjfsUid uid, long regionId) {
        String sql = collectStats(uid)
                + " SELECT * FROM disk.albums WHERE uid = :uid AND album_type = 'geo' AND geo_id = :geo_id ";
        MapF<String, Object> params = Cf.map(
                "uid", uid,
                "geo_id", regionId
        );
        return jdbcTemplate(uid).queryForOption(sql, M, params);
    }

    @Override
    public Option<ExtendedAlbum> findExtendedGeoAlbum(DjfsUid uid, long regionId) {
        String sql = collectStats(uid)
                + " SELECT *,"
                + " (SELECT COUNT(*) FROM disk.album_items WHERE uid = uid AND album_id = a.id) AS items_count,"
                + " (SELECT obj_id FROM disk.album_items WHERE uid = uid AND id = a.cover_id) AS cover_obj_id"
                + " FROM disk.albums AS a WHERE uid = :uid AND album_type = 'geo' AND geo_id = :geo_id ";
        MapF<String, Object> params = Cf.map(
                "uid", uid,
                "geo_id", regionId
        );
        return jdbcTemplate(uid).queryForOption(sql, EXT_M, params);
    }

    @Override
    public boolean updateAlbumRevision(Album album, long revision) {
        return updateAlbumRevision(album, revision, Option.empty());
    }

    @Override
    public boolean updateAlbumRevision(Album album, long revision, Option<Instant> dateModified) {
        String sql = collectStats(album.getUid())
                + " UPDATE disk.albums SET revision = :revision"
                + (dateModified.isPresent() ? ", date_modified = :date_modified " : "")
                + " WHERE uid = :uid AND id = :album_id AND (revision < :revision OR revision IS NULL)";
        // update album revision only if it's less than current

        MapF<String, Object> params = Cf.map(
                "uid", album.getUid(),
                "album_id", album.getId().toByteArray(),
                "revision", revision
        ).plus(dateModified.toMap(v -> "date_modified", v -> v));
        int updated = jdbcTemplate(album.getUid()).update(sql, params);
        return updated > 0;
    }

    @Override
    public void incrementAlbumRevision(Album album) {
        String sql = collectStats(album.getUid())
                + " UPDATE disk.albums SET revision = coalesce(revision + 1, 1) WHERE uid = :uid AND id = :album_id";
        // revision might be null here (we treat it as 0 value), if we increment null value, we should set it to 1
        // that's why we use coalesce here

        MapF<String, Object> params = Cf.map(
                "uid", album.getUid(),
                "album_id", album.getId().toByteArray()
        );
        jdbcTemplate(album.getUid()).update(sql, params);
    }

    @Override
    public boolean updateAlbumsRevision(ListF<Album> albums, DjfsUid uid, long revision) {
        return updateAlbumsRevision(albums, uid, revision, Option.empty());
    }

    @Override
    public boolean updateAlbumsRevision(ListF<Album> albums, DjfsUid uid, long revision, Option<Instant> dateModified) {
        if (albums.size() == 0) {
            return false;
        }
        if (albums.size() == 1) {
            return updateAlbumRevision(albums.single(), revision, dateModified);
        }

        String sql = collectStats(uid)
                + " UPDATE disk.albums SET revision = :revision"
                + (dateModified.isPresent() ? ", date_modified = :date_modified " : "")
                + " WHERE uid = :uid AND id IN (:album_ids) AND (revision < :revision OR revision IS NULL)";
        // update album revision only if it's less than current

        MapF<String, Object> params = Cf.map(
                "uid", uid,
                "album_ids", albums.map(Album::getId).map(ObjectId::toByteArray),
                "revision", revision
        ).plus(dateModified.toMap(v -> "date_modified", v -> v));
        int updated = jdbcTemplate(uid).update(sql, params);
        return updated > 0;
    }

    @Override
    public void makeVisible(Album album) {
        setHidden(album, false);
    }

    @Override
    public void setCover(Album album, ObjectId coverItemId) {
        String sql = collectStats(album.getUid())
                + " UPDATE disk.albums SET cover_id = :cover_id WHERE uid = :uid AND id = :album_id";

        MapF<String, Object> params = Cf.map(
                "uid", album.getUid(),
                "album_id", album.getId().toByteArray(),
                "cover_id", coverItemId.toByteArray()
        );
        jdbcTemplate(album.getUid()).update(sql, params);
    }

    @Override
    public void setCoverAuto(Album album, boolean coverAuto) {
        final String sql = collectStats(album.getUid())
                + " UPDATE disk.albums SET cover_auto = :cover_auto WHERE uid = :uid AND id = :album_id";

        final MapF<String, Object> params = Cf.map(
                "uid", album.getUid(),
                "album_id", album.getId().toByteArray(),
                "cover_auto", coverAuto
        );
        jdbcTemplate(album.getUid()).update(sql, params);
    }

    @Override
    public void setTitle(Album album, String title) {
        String sql = collectStats(album.getUid())
                + " UPDATE disk.albums SET title = :title WHERE uid = :uid AND id = :album_id";

        MapF<String, Object> params = Cf.map(
                "uid", album.getUid(),
                "album_id", album.getId().toByteArray(),
                "title", title
        );
        jdbcTemplate(album.getUid()).update(sql, params);
    }

    @Override
    public void setParams(Album album, ListF<AlbumParam> albumParams) {
        if (albumParams.isEmpty()) {
            return;
        }

        // Проверяем, что никто случайно дважды не записал один и тот же name
        Validate.unique(albumParams.map(AlbumParam::getName));

        String paramsQueryPattern = albumParams.map(AlbumParam::getName)
                .map(name -> name + " = :" + name).mkString(", ");

        String sql = collectStats(album.getUid())
                + " UPDATE disk.albums SET " + paramsQueryPattern + " WHERE uid = :uid AND id = :album_id";

        MapF<String, Object> params = Cf.map(
                "uid", album.getUid(),
                "album_id", album.getId().toByteArray()
        ).plus(albumParams.toTuple2List(AlbumParam::getName, AlbumParam::getValue).toMap());
        jdbcTemplate(album.getUid()).update(sql, params);
    }

    @Override
    public void updateDateModified(Album album, Instant dateModified) {
        String sql = collectStats(album.getUid())
                + " UPDATE disk.albums SET date_modified = :date_modified WHERE uid = :uid AND id = :album_id";

        MapF<String, Object> params = Cf.map(
                "uid", album.getUid(),
                "album_id", album.getId().toByteArray(),
                "date_modified", dateModified
        );
        jdbcTemplate(album.getUid()).update(sql, params);
    }

    @Override
    public boolean delete(Album album) {
        String sql = collectStats(album.getUid()) + " DELETE FROM disk.albums WHERE uid = :uid AND id = :album_id";
        return jdbcTemplate(album.getUid()).update(sql, Cf.map("uid", album.getUid(), "album_id", album.getId().toByteArray())) > 0;
    }

    @Override
    public void makeHidden(Album album) {
        setHidden(album, true);
    }

    @Override
    public Option<Album> findAlbum(DjfsUid uid, ObjectId albumId) {
        String sql = collectStats(uid)
                + " SELECT * FROM disk.albums WHERE uid = :uid AND id = :album_id ";
        MapF<String, Object> params = Cf.map(
                "uid", uid,
                "album_id", albumId.toByteArray()
        );
        return jdbcTemplate(uid).queryForOption(sql, M, params);
    }

    @Override
    public Option<ExtendedAlbum> findExtendedAlbum(DjfsUid uid, ObjectId albumId) {
        String sql = collectStats(uid)
                + " SELECT *,"
                + " (SELECT COUNT(*) FROM disk.album_items WHERE uid = uid AND album_id = a.id) AS items_count,"
                + " (SELECT obj_id FROM disk.album_items WHERE uid = uid AND id = a.cover_id) AS cover_obj_id"
                + " FROM disk.albums AS a WHERE uid = :uid AND album_id = :album_id ";
        MapF<String, Object> params = Cf.map(
                "uid", uid,
                "album_id", albumId.toByteArray()
        );
        return jdbcTemplate(uid).queryForOption(sql, EXT_M, params);
    }

    private void setHidden(Album album, boolean value) {
        String sql = collectStats(album.getUid())
                + " UPDATE disk.albums SET hidden = :hidden WHERE uid = :uid AND id = :album_id";

        MapF<String, Object> params = Cf.map(
                "uid", album.getUid(),
                "album_id", album.getId().toByteArray(),
                "hidden", value
        );
        jdbcTemplate(album.getUid()).update(sql, params);
    }
}
