package ru.yandex.chemodan.app.djfs.core.album;

import java.util.Objects;

import org.postgresql.util.PSQLException;
import org.postgresql.util.ServerErrorMessage;
import org.springframework.dao.DataIntegrityViolationException;
import org.springframework.jdbc.core.RowMapper;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.chemodan.app.djfs.core.db.EntityAlreadyExistsException;
import ru.yandex.chemodan.app.djfs.core.db.pg.PgShardedDao;
import ru.yandex.chemodan.app.djfs.core.db.pg.PgShardedDaoContext;
import ru.yandex.chemodan.app.djfs.core.db.pg.ResultSetUtils;
import ru.yandex.chemodan.app.djfs.core.user.DjfsUid;
import ru.yandex.chemodan.app.djfs.core.util.UuidUtils;
import ru.yandex.commune.dynproperties.DynamicProperty;
import ru.yandex.misc.lang.CharsetUtils;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

public class PgAlbumDeltaDao extends PgShardedDao implements AlbumDeltaDao {
    private static final Logger logger = LoggerFactory.getLogger(PgAlbumDeltaDao.class);

    private final DynamicProperty<Boolean> disableLinearization = new DynamicProperty<>(
            "disk-djfs-albums-disable-linearization", true);

    private final static RowMapper<AlbumDeltaRaw> M = (rs, rowNum) -> AlbumDeltaRaw.builder()
            .uid(DjfsUid.cons(rs.getLong("uid")))
            .id(UuidUtils.toHexString(ResultSetUtils.getUuid(rs, "id")))
            .revision(rs.getLong("revision"))
            .changes(rs.getString("changes"))
            .dateCreated(ResultSetUtils.getInstantO(rs, "date_created"))
            .albumType(Option.of(AlbumType.R.fromValue(Option.ofNullable(rs.getString("album_type")).getOrElse("geo"))))
            .build();

    public PgAlbumDeltaDao(PgShardedDaoContext context) {
        super(context);
    }

    @Override
    public void insert(AlbumDelta delta) {
        String sql = collectStats(delta)
                + " INSERT INTO disk.album_deltas (uid, id, revision, changes, date_created, album_type) "
                + " VALUES (:uid, :id, :revision, :changes::jsonb, :date_created, :album_type)";

        MapF<String, Object> params = Cf.toMap(Tuple2List.fromPairs(
                "uid", delta.getUid(),
                "id", UuidUtils.fromHex(delta.getId()),
                "revision", delta.getRevision(),
                "changes", CharsetUtils.decodeUtf8(AlbumDelta.B.getSerializer().serializeListJson(delta.getChanges())),
                "date_created", delta.getDateCreated(),
                "album_type", delta.getAlbumType()
        ));

        try {
            jdbcTemplate(delta.getUid()).update(sql, params);
        } catch (DataIntegrityViolationException e) {
            Throwable cause = e.getCause();
            if (cause instanceof PSQLException) {
                ServerErrorMessage error = ((PSQLException) cause).getServerErrorMessage();
                if (error != null && (Objects.equals(error.getConstraint(), "pk_album_deltas"))) {
                    logger.info("PgAlbumDeltaDao.insert(AlbumDelta) handled exception for user "
                            + delta.getUid() + " : ", e);
                    throw new EntityAlreadyExistsException(delta.getId(), e);
                }
            }
            throw e;
        }
    }

    @Override
    public ListF<AlbumDeltaRaw> findRaw(DjfsUid uid) {
        String sql = collectStats(uid)
                + " SELECT * FROM disk.album_deltas WHERE uid = :uid ORDER BY revision";

        MapF<String, Object> params = Cf.toMap(Tuple2List.fromPairs(
                "uid", uid
        ));

        return jdbcTemplate(uid).query(sql, M, params);
    }

    @Override
    public void deleteAll(DjfsUid uid) {
        String sql = collectStats(uid)
                + " DELETE FROM disk.album_deltas WHERE uid = :uid";

        MapF<String, Object> params = Cf.toMap(Tuple2List.fromPairs(
                "uid", uid
        ));

        jdbcTemplate(uid).update(sql, params);
    }

    @Override
    public Option<AlbumDeltaRaw> findRaw(DjfsUid uid, long revision) {
        String sql = collectStats(uid)
                + " SELECT * FROM disk.album_deltas WHERE uid = :uid AND revision = :revision";

        MapF<String, Object> params = Cf.toMap(Tuple2List.fromPairs(
                "uid", uid,
                "revision", revision
        ));

        return jdbcTemplate(uid).queryForOption(sql, M, params);
    }

    @Override
    public ListF<AlbumDeltaRaw> findRaw(DjfsUid uid, long minRevision, long maxRevision, int limit) {
        String sql = collectStats(uid)
                + " SELECT * FROM disk.album_deltas"
                + " WHERE uid = :uid AND revision > :min_rev AND revision <= :max_rev ORDER BY revision LIMIT :limit";

        MapF<String, Object> params = Cf.toMap(Tuple2List.fromPairs(
                "uid", uid,
                "min_rev", minRevision,
                "max_rev", maxRevision,
                "limit", limit
        ));

        return jdbcTemplate(uid).query(sql, M, params);
    }

    @Override
    public int count(DjfsUid uid, long minRevision, long maxRevision) {
        String sql = collectStats(uid)
                + " SELECT COUNT(*) FROM disk.album_deltas WHERE uid = ? AND revision > ? AND revision <= ?";
        return jdbcTemplate(uid).queryForInt(sql, uid, minRevision, maxRevision);
    }

    @Override
    public Option<Long> getCurrentRevisionWithLock(DjfsUid uid) {
        return getCurrentRevision(uid, true);
    }

    @Override
    public Option<Long> getCurrentRevisionWithoutLock(DjfsUid uid) {
        return getCurrentRevision(uid, false);
    }

    @Override
    public void updateCurrentRevision(DjfsUid uid, long value) {
        String sql = collectStats(uid) + " UPDATE disk.albums_info SET revision = :revision WHERE uid = :uid";
        MapF<String, Object> params = Cf.map(
                "uid", uid,
                "revision", value
        );
        jdbcTemplate(uid).update(sql, params);
    }

    @Override
    public boolean tryInitializeCurrentRevision(DjfsUid uid) {
        String sql = collectStats(uid)
                + " INSERT INTO disk.albums_info (uid, revision) VALUES (?, 0)"
                + " ON CONFLICT ON CONSTRAINT pk_albums_info DO NOTHING";
        return jdbcTemplate(uid).update(sql, uid) > 0;
    }

    @Override
    public void removeCurrentRevision(DjfsUid uid) {
        String sql = collectStats(uid)
                + " DELETE FROM disk.albums_info WHERE uid = :uid";
        jdbcTemplate(uid).update(sql, Cf.map("uid", uid));
    }

    private Option<Long> getCurrentRevision(DjfsUid uid, boolean lockForUpdate) {
        String sql = collectStats(uid) + " SELECT revision FROM disk.albums_info WHERE uid = ?";
        if (lockForUpdate) {
            if (SynchronizedHandlerHolder.get()) {
                sql += " FOR UPDATE";
            } else {
                sql += " FOR UPDATE NOWAIT";
            }
        }
        return jdbcTemplate(uid).queryForOption(sql, Long.class, uid);
    }
}
