package ru.yandex.chemodan.app.djfs.core.album.worker;

import org.joda.time.Duration;
import org.joda.time.Instant;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.djfs.core.album.AlbumBaseProcessingTask;
import ru.yandex.chemodan.app.djfs.core.album.FileOperationPostProcessingTask;
import ru.yandex.chemodan.app.djfs.core.album.GeoAlbumCoverUpdateTask;
import ru.yandex.chemodan.app.djfs.core.album.GeoAlbumResourceProcessingTask;
import ru.yandex.chemodan.app.djfs.core.index.BazingaManagerProperties;
import ru.yandex.chemodan.app.djfs.core.tasks.DjfsAlbumsTaskQueueName;
import ru.yandex.chemodan.app.djfs.core.user.DjfsUid;
import ru.yandex.chemodan.bazinga.YcridOnetimeTaskSupport;
import ru.yandex.chemodan.bazinga.YcridTaskParameters;
import ru.yandex.chemodan.util.retry.RetryManager;
import ru.yandex.commune.bazinga.BazingaTaskManager;
import ru.yandex.commune.bazinga.impl.TaskOverridesManager;
import ru.yandex.commune.bazinga.impl.worker.WorkerTaskRegistry;
import ru.yandex.commune.bazinga.scheduler.ActiveUidBehavior;
import ru.yandex.commune.bazinga.scheduler.ActiveUidDropType;
import ru.yandex.commune.bazinga.scheduler.ActiveUidDuplicateBehavior;
import ru.yandex.commune.bazinga.scheduler.ExecutionContext;
import ru.yandex.commune.bazinga.scheduler.TaskQueueName;
import ru.yandex.commune.dynproperties.DynamicProperty;
import ru.yandex.misc.bender.annotation.BenderBindAllFields;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

/**
 * @author yak-dmitriy
 */
public class DjfsAlbumsQueueTask extends YcridOnetimeTaskSupport<DjfsAlbumsQueueTask.Parameters> {

    private static final Logger logger = LoggerFactory.getLogger(DjfsAlbumsQueueTask.class);

    private final DynamicProperty<Integer> subTaskBatchSize = new DynamicProperty<>(
            "disk-djfs-albums-subtask-batch-size", 1000);
    private final DynamicProperty<Integer> subTaskWatchdogTime = new DynamicProperty<>(
            "disk-djfs-albums-subtask-watchdog-time-sec", 2);

    private final DjfsAlbumsTaskDao djfsAlbumsTaskDao;

    private final GeoAlbumCoverUpdateTask geoAlbumCoverUpdateTask;
    private final GeoAlbumResourceProcessingTask geoAlbumResourceProcessingTask;
    private final FileOperationPostProcessingTask fileOperationPostProcessingTask;

    private final BazingaTaskManager bazingaTaskManager;
    private final BazingaManagerProperties bazingaTaskManagerProperties;
    private final TaskOverridesManager taskOverridesManager;
    private final WorkerTaskRegistry workerTaskRegistry;

    public DjfsAlbumsQueueTask(DjfsAlbumsTaskDao djfsAlbumsTaskDao, GeoAlbumCoverUpdateTask geoAlbumCoverUpdateTask,
            GeoAlbumResourceProcessingTask geoAlbumResourceProcessingTask, FileOperationPostProcessingTask fileOperationPostProcessingTask,
            BazingaTaskManager bazingaTaskManager, BazingaManagerProperties bazingaTaskManagerProperties,
            TaskOverridesManager taskOverridesManager, WorkerTaskRegistry workerTaskRegistry) {
        super(Parameters.class);
        this.djfsAlbumsTaskDao = djfsAlbumsTaskDao;
        this.geoAlbumCoverUpdateTask = geoAlbumCoverUpdateTask;
        this.geoAlbumResourceProcessingTask = geoAlbumResourceProcessingTask;
        this.fileOperationPostProcessingTask = fileOperationPostProcessingTask;
        this.bazingaTaskManager = bazingaTaskManager;
        this.bazingaTaskManagerProperties = bazingaTaskManagerProperties;
        this.taskOverridesManager = taskOverridesManager;
        this.workerTaskRegistry = workerTaskRegistry;
    }

    public DjfsAlbumsQueueTask(String uid) {
        super(new Parameters(uid));
        this.djfsAlbumsTaskDao = null;
        this.geoAlbumCoverUpdateTask = null;
        this.geoAlbumResourceProcessingTask = null;
        this.fileOperationPostProcessingTask = null;
        this.bazingaTaskManager = null;
        this.bazingaTaskManagerProperties = null;
        this.taskOverridesManager = null;
        this.workerTaskRegistry = null;
    }

    @Override
    protected void doExecute(Parameters parameters, ExecutionContext context) {

        ListF<DjfsAlbumsSubtask> subTasks = djfsAlbumsTaskDao.getSubtasks(
                DjfsUid.cons(parameters.uid), subTaskBatchSize.get());

        if (subTasks.length() == 0) {
            scheduleQueueWatchdogTask(parameters.uid);
            return;
        }

        MapF<String, AlbumBaseProcessingTask> mapTasks = Cf.list(
                geoAlbumCoverUpdateTask,
                geoAlbumResourceProcessingTask,
                fileOperationPostProcessingTask).toMap(task -> task.id().getId(), task -> task);

        for (DjfsAlbumsSubtask subTask : subTasks) {

            Option<AlbumBaseProcessingTask> task0 = mapTasks.getO(subTask.getTaskId());

            if (task0.isPresent()) {

                try {
                    task0.get().doExecute(subTask.getParameters(), context);
                } catch (Exception e) {
                    handleFailedSubTask(subTask, task0.get(), e);
                    continue;
                }

                logger.info("Executed " + subTask.getTaskId() + " for user: " + subTask.getParameters().getUid() +
                        ", resource_id: " + subTask.getParameters().getResourceId());

            } else {

                logger.error("No linearized task " + subTask.getTaskId() + " for user: " + subTask.getParameters().getUid() +
                        ", resource_id: " + subTask.getParameters().getResourceId());

            }
            djfsAlbumsTaskDao.deleteSubtask(subTask);
        }

        scheduleQueueWatchdogTask(parameters.uid);
    }

    private void scheduleQueueWatchdogTask(String uid) {
        new RetryManager().withRetryPolicy(
                bazingaTaskManagerProperties.getMaxRetries(),
                bazingaTaskManagerProperties.getRetryDelayMillis()
        ).run(() ->
                bazingaTaskManager.schedule(
                        new DjfsAlbumsQueueWatchdogTask(uid),
                        Instant.now().plus(Duration.standardSeconds(subTaskWatchdogTime.get()))
                )
        );
    }

    private void handleFailedSubTask(DjfsAlbumsSubtask subTask, AlbumBaseProcessingTask task, Exception e) {
        Option<Instant> nextTime = taskOverridesManager.getWithOverrides(workerTaskRegistry.getOnetimeTask(task.id()))
                .reschedulePolicy().rescheduleAt(new Instant(), subTask.getRetryCount() + 1);
        if (nextTime.isPresent()) {
            djfsAlbumsTaskDao.setNextRetry(subTask, subTask.getRetryCount() + 1, nextTime.get());
            logger.warn("Retry " + subTask.getTaskId() + " for user: " + subTask.getParameters().getUid() +
                    ", resource_id: " + subTask.getParameters().getResourceId() + ", error: " + e);
        } else {
            djfsAlbumsTaskDao.deleteSubtask(subTask);
            logger.error("Deleted task " + subTask.getTaskId() + " on retry policy for user: " + subTask.getParameters().getUid() +
                    ", resource_id: " + subTask.getParameters().getResourceId());
        }
    }

    @Override
    public TaskQueueName queueName() {
        return DjfsAlbumsTaskQueueName.ALBUMS_QUEUE_TASKS;
    }

    @Override
    public int priority() {
        return 0;
    }

    @Override
    public Duration timeout() {
        return Duration.standardSeconds(100);
    }

    @BenderBindAllFields
    static class Parameters extends YcridTaskParameters {
        private final String uid;

        Parameters(String uid) {
            this.uid = uid;
        }
    }

    @Override
    public ActiveUidBehavior activeUidBehavior() {
        return new ActiveUidBehavior(ActiveUidDropType.WHEN_FINISHED, ActiveUidDuplicateBehavior.DO_NOTHING);
    }
}
