package ru.yandex.chemodan.app.djfs.core.album.worker;

import java.util.UUID;

import org.joda.time.Instant;
import org.postgresql.util.PSQLException;
import org.springframework.dao.DataIntegrityViolationException;
import org.springframework.jdbc.core.RowMapper;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.chemodan.app.djfs.core.album.AlbumBaseProcessingTask;
import ru.yandex.chemodan.app.djfs.core.db.pg.PgShardedDao;
import ru.yandex.chemodan.app.djfs.core.db.pg.PgShardedDaoContext;
import ru.yandex.chemodan.app.djfs.core.user.DjfsUid;
import ru.yandex.commune.bazinga.BazingaBender;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

/**
 * @author yak-dmitriy
 */
public class DjfsAlbumsTaskDao extends PgShardedDao {
    private static final Logger logger = LoggerFactory.getLogger(DjfsAlbumsTaskDao.class);

    private final static RowMapper<DjfsAlbumsSubtask> M = (rs, rowNum) -> {
        String jsonData = rs.getString("task_data");
        AlbumBaseProcessingTask.Parameters parameters = null;
        if (jsonData != null && jsonData.length() > 0) {
            parameters = BazingaBender.mapper.parseJson(AlbumBaseProcessingTask.Parameters.class, jsonData);
        }
        DjfsAlbumsSubtask djfsAlbumsSubtask = new DjfsAlbumsSubtask(
                UUID.fromString(rs.getString("id")),
                DjfsUid.cons(rs.getLong("uid")),
                rs.getString("task_name"),
                parameters,
                rs.getInt("retry_count"),
                new Instant(rs.getTimestamp("next_schedule"))
        );
        return djfsAlbumsSubtask;
    };

    public DjfsAlbumsTaskDao(PgShardedDaoContext context) {
        super(context);
    }

    public void addSubtask(DjfsAlbumsSubtask task) {
        String sql = collectStats(task.getUid())
                + " INSERT INTO disk.task_queue (uid, task_name, task_data)"
                + " VALUES (:uid, :task_name, :task_data::jsonb);";
        MapF<String, Object> params = Cf.map(
                "uid", task.getUid(),
                "task_name", task.getTaskId(),
                "task_data", new String(BazingaBender.mapper.serializeJson(task.getParameters()))
        );

        try {
            jdbcTemplate(task.getUid()).update(sql, params);
        } catch (DataIntegrityViolationException e) {
            Throwable cause = e.getCause();
            if (cause instanceof PSQLException) {
                logger.info("DjfsAlbumsTaskDao.addSubtask(task) handled exception for user "
                        + task.getUid() + " : ", e);
            }
            throw e;
        }
    }

    public ListF<DjfsAlbumsSubtask> getSubtasks(DjfsUid uid, int limit) {
        String sql = collectStats(uid)
                + " SELECT id, uid, task_name, task_data, retry_count, next_schedule"
                + " FROM disk.task_queue"
                + " WHERE uid = :uid AND next_schedule <= now()"
                + " ORDER BY retry_count, next_schedule"
                + (limit > 0 ? " LIMIT :limit;" : ";");
        MapF<String, Object> params;
        if (limit > 0) {
            params = Cf.map("uid", uid,"limit", limit);
        } else {
            params = Cf.map("uid", uid);
        }

        return jdbcTemplate(uid).query(sql, M, params);
    }

    public void setNextRetry(DjfsAlbumsSubtask task, Integer retryCount, Instant nextSchedule) {
        String sql = collectStats(task.getUid())
                + " UPDATE disk.task_queue SET retry_count = :retry_count, next_schedule = :next_schedule WHERE id = :id;";
        MapF<String, Object> params = Cf.map(
                "id", task.getId(),
                "retry_count", retryCount,
                "next_schedule", nextSchedule.toDateTime()
        );

        try {
            jdbcTemplate(task.getUid()).update(sql, params);
        } catch (DataIntegrityViolationException e) {
            Throwable cause = e.getCause();
            if (cause instanceof PSQLException) {
                logger.info("DjfsAlbumsTaskDao.incRetryCount(task) handled exception for user "
                        + task.getUid() + " : ", e);
            }
            throw e;
        }
    }

    public void deleteSubtask(DjfsAlbumsSubtask task) {
        String sql = collectStats(task.getUid())
                + " DELETE FROM disk.task_queue WHERE id = :id;";
        MapF<String, Object> params = Cf.map(
                "id", task.getId()
        );

        try {
            jdbcTemplate(task.getUid()).update(sql, params);
        } catch (DataIntegrityViolationException e) {
            Throwable cause = e.getCause();
            if (cause instanceof PSQLException) {
                logger.info("DjfsAlbumsTaskDao.deleteSubtask(task) handled exception for user "
                        + task.getUid() + " : ", e);
            }
            throw e;
        }
    }
}
