package ru.yandex.chemodan.app.djfs.core.changelog;

import lombok.RequiredArgsConstructor;
import org.bson.types.ObjectId;
import org.joda.time.Instant;

import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.djfs.core.EventManager;
import ru.yandex.chemodan.app.djfs.core.filesystem.event.FolderCreatedEvent;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.DjfsResourceType;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.FolderDjfsResource;
import ru.yandex.chemodan.app.djfs.core.share.event.UserKickedFromGroupEvent;
import ru.yandex.chemodan.app.djfs.core.share.event.UserLeftGroupEvent;
import ru.yandex.chemodan.app.djfs.core.util.InstantUtils;

/**
 * @author eoshch
 */
@RequiredArgsConstructor
public class ChangelogEventHandler {
    private final ChangelogDao changelogDao;
    private final ChangelogManager changelogManager;

    @EventManager.EventHandler
    public void handle(FolderCreatedEvent event) {
        changelogManager.createChangelog(event.eventTime, Changelog.OperationType.NEW, event.folder,
                event.getShareInfoO());
    }

    private void onFolderRemoved(Instant timestamp, FolderDjfsResource folder) {
        Changelog changelog = Changelog.builder()
                .id(new ObjectId())
                .path(folder.getPath())
                .publik(folder.isPublic())
                .visible(folder.isVisible())
                .dtime(Option.of(timestamp))
                .version(InstantUtils.toVersion(timestamp))
                .fileId(folder.getFileId().get().getValue())
                .operationType(Changelog.OperationType.DELETED)
                .resourceType(DjfsResourceType.DIR)
                .build();
        changelogDao.insert(changelog);
    }

    @EventManager.EventHandler
    public void handle(UserKickedFromGroupEvent event) {
        if (!event.getUserShareRoot().isPresent()) {
            return;
        }

        FolderDjfsResource folder = event.getUserShareRoot().get();
        if (!folder.getPath().getArea().isChangelogWritten()) {
            return;
        }

        onFolderRemoved(event.getInstant(), folder);
    }

    @EventManager.EventHandler
    public void handle(UserLeftGroupEvent event) {
        if (!event.getUserShareRoot().isPresent()) {
            return;
        }

        FolderDjfsResource folder = event.getUserShareRoot().get();
        if (!folder.getPath().getArea().isChangelogWritten()) {
            return;
        }

        onFolderRemoved(event.getInstant(), folder);
    }
}
