package ru.yandex.chemodan.app.djfs.core.changelog;

import lombok.RequiredArgsConstructor;
import org.bson.types.ObjectId;
import org.joda.time.Instant;

import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.bolts.internal.NotImplementedException;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.AntiVirusScanStatus;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.DjfsResource;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.DjfsResourceType;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.FileDjfsResource;
import ru.yandex.chemodan.app.djfs.core.share.ShareInfo;
import ru.yandex.chemodan.app.djfs.core.user.DjfsUid;
import ru.yandex.chemodan.app.djfs.core.util.InstantUtils;

/**
 * @author eoshch
 */
@RequiredArgsConstructor
public class ChangelogManager {
    private final ChangelogDao changelogDao;

    public void createChangelog(Instant instant, Changelog.OperationType type, DjfsResource resource,
            Option<ShareInfo> shareInfo)
    {
        if (!resource.getPath().getArea().isChangelogWritten()) {
            return;
        }
        Changelog.ChangelogBuilder builder = makeChangelogBuilder(instant, type, resource, shareInfo);
        changelogDao.insert(builder.build());
    }

    public void createMoveChangelog(Instant instant, Changelog.OperationType type, DjfsResource sourceFolder,
            Option<ShareInfo> shareInfo, DjfsResource destinationFolder)
    {
        if (!sourceFolder.getPath().getArea().isChangelogWritten() &&
                !destinationFolder.getPath().getArea().isChangelogWritten()) {
            return;
        }
        Changelog.ChangelogBuilder builder = makeChangelogBuilder(instant, type, sourceFolder, shareInfo);
        builder.newPath(Option.of(destinationFolder.getPath()));
        changelogDao.insert(builder.build());
    }

    public Option<Changelog> makeChangelog(Instant instant, Changelog.OperationType type, DjfsResource resource,
            Option<ShareInfo> shareInfo)
    {
        if (!resource.getPath().getArea().isChangelogWritten()) {
            return Option.empty();
        }
        return Option.of(makeChangelogBuilder(instant, type, resource, shareInfo).build());
    }

    public void createChangelogs(ListF<Changelog> changelogs) {
        if (changelogs.isEmpty()) {
            return;
        }

        SetF<DjfsUid> uids = changelogs.map(Changelog::getUid).unique();
        if (uids.size() != 1) {
            throw new NotImplementedException();  // todo: split by uid here and call method using appropriate dao
        }
        changelogDao.insert(uids.toList().get(0), changelogs);
    }

    private Changelog.ChangelogBuilder makeChangelogBuilder(Instant instant, Changelog.OperationType type,
            DjfsResource resource, Option<ShareInfo> shareInfo)
    {
        Changelog.ChangelogBuilder builder = Changelog.builder()
                .id(new ObjectId())
                .operationType(type)
                .path(resource.getPath())
                .groupId(shareInfo.map(ShareInfo::getGroupId))
                .groupPath(shareInfo.map(x -> x.getGroupPath().getPath()))
                .publik(resource.isPublic())
                .visible(resource.isVisible())
                .fileId(resource.getFileId().get().getValue())

                .version(InstantUtils.toVersion(instant))
                .dtime(Option.of(instant));

        if (resource instanceof FileDjfsResource) {
            FileDjfsResource file = (FileDjfsResource) resource;

            builder.resourceType(DjfsResourceType.FILE)
                    .md5(Option.of(file.getMd5()))
                    .sha256(Option.of(file.getSha256()))
                    .size(Option.of(file.getSize()))
                    .modificationTime(Option.of(file.getModificationTime()))
                    .exifTime(file.getExifTime())
                    .mimetype(file.getMimetype())
                    .mediaType(file.getMediaType())
                    .antiVirusScanStatus(file.getAntiVirusScanStatus().orElse(Option.of(AntiVirusScanStatus.ERROR)))
                    .hasPreview(Option.of(file.getPreviewStid().isPresent()))
                    .hasExternalSetprop(Option.of(file.getExternalProperties().isNotEmpty()));
        } else {
            builder.resourceType(DjfsResourceType.DIR);
        }

        return builder;
    }
}
