package ru.yandex.chemodan.app.djfs.core.changelog;

import org.bson.types.ObjectId;
import org.joda.time.Duration;
import org.joda.time.Instant;

import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.djfs.core.db.mongo.DjfsBenderFactory;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.AntiVirusScanStatus;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.DjfsResourcePath;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.DjfsResourceType;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.MediaType;
import ru.yandex.chemodan.app.djfs.core.util.InstantUtils;
import ru.yandex.chemodan.app.djfs.core.util.ZipUtils;
import ru.yandex.commune.mongo3.MongoBenderParserSerializer;
import ru.yandex.commune.mongo3.bender.MongoId;
import ru.yandex.misc.bender.annotation.BenderBindAllFields;
import ru.yandex.misc.bender.annotation.BenderPart;
import ru.yandex.misc.lang.DefaultObject;

@BenderBindAllFields
public class MongoChangelog extends DefaultObject {
    public static final MongoBenderParserSerializer<ObjectId, MongoChangelog> B =
            DjfsBenderFactory.createForMongo(MongoChangelog.class);

    @MongoId
    public ObjectId id;
    public long version;
    public String uid;
    public Option<String> key;
    public Option<String> type;
    public Option<String> op;

    public Option<Integer> rights;
    public Option<String> shared;
    public Option<String> gid;
    @BenderPart(name = "group_path", strictName = true)
    public Option<String> group_path;

    public byte[] zdata;
    public Option<Instant> dtime;

    public MongoChangelog() {
        this.key = Option.empty();
        this.type = Option.empty();
        this.op = Option.empty();
        this.rights = Option.empty();
        this.shared = Option.empty();
        this.gid = Option.empty();
        this.group_path = Option.empty();
        this.dtime = Option.empty();
    }

    public Changelog to() {
        ChangelogZData zData = ChangelogZData.B.getParser().parseJson(ZipUtils.decompress(zdata));

        return Changelog.builder()
                .id(id)
                .version(version)
                .path(DjfsResourcePath.cons(uid, zData.key.orElse(key).get()))
                .resourceType(DjfsResourceType.R.valueOf(zData.type.orElse(type).get()))
                .operationType(Changelog.OperationType.R.valueOf(zData.op.orElse(op).get()))
                .fileId(zData.fid)
                .visible(zData.visible == 1)
                .publik(zData.publik == 1)
                .dtime(dtime.map(x -> x.plus(Duration.standardHours(3))))
                .groupId(gid)
                .groupPath(group_path)
                .sha256(zData.sha256)
                .md5(zData.md5)
                .size(zData.size)
                .mimetype(zData.mimetype)
                .mediaType(zData.mediaType.map(MediaType.R::valueOf))
                .antiVirusScanStatus(zData.drweb.map(AntiVirusScanStatus.R::fromValue))
                .hasPreview(zData.has_preview)
                .hasExternalSetprop(zData.external_setprop.map(x -> x == 1))
                .exifTime(zData.etime.map(InstantUtils::fromSeconds))
                .modificationTime(zData.mtime.map(InstantUtils::fromSeconds))
                .build();
    }

    public static MongoChangelog cons(Changelog changelog) {
        MongoChangelog result = new MongoChangelog();
        result.id = changelog.getId();
        result.uid = changelog.getPath().getUid().asString();
        result.version = changelog.getVersion();
        result.dtime = changelog.getDtime().map(x -> x.plus(Duration.standardHours(3)));

        ChangelogZData zData = new ChangelogZData();
        zData.key = Option.of(changelog.getPath().getPath());
        zData.fid = changelog.getFileId();
        zData.op = Option.of(changelog.getOperationType().name().toLowerCase());
        zData.type = Option.of(changelog.getResourceType().name().toLowerCase());
        zData.publik = changelog.isPublik() ? 1 : 0;
        zData.visible = changelog.isVisible() ? 1 : 0;

        zData.sha256 = changelog.getSha256();
        zData.md5 = changelog.getMd5();
        zData.size = changelog.getSize();

        zData.mimetype = changelog.getMimetype();
        zData.mediaType = changelog.getMediaType().map(MediaType::getStringRepresentation);
        zData.drweb = changelog.getAntiVirusScanStatus().map(AntiVirusScanStatus::getIntRepresentation);

        zData.has_preview = changelog.getHasPreview();
        zData.external_setprop = changelog.getHasExternalSetprop().getOrElse(false) ? Option.of(1) : Option.empty();

        zData.etime = changelog.getExifTime().map(InstantUtils::toSecondsLong);
        zData.mtime = changelog.getModificationTime().map(InstantUtils::toSecondsLong);

        result.zdata = ZipUtils.compress(ChangelogZData.B.getSerializer().serializeJson(zData), 1);

        result.gid = changelog.getGroupId();
        result.group_path = changelog.getGroupPath();
        return result;
    }
}
