package ru.yandex.chemodan.app.djfs.core.changelog;

import org.joda.time.Instant;
import org.springframework.jdbc.core.RowMapper;
import org.springframework.jdbc.core.namedparam.MapSqlParameterSource;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.djfs.core.db.pg.PgShardedDao;
import ru.yandex.chemodan.app.djfs.core.db.pg.PgShardedDaoContext;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.AntiVirusScanStatus;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.DjfsResourcePath;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.DjfsResourceType;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.MediaType;
import ru.yandex.chemodan.app.djfs.core.user.DjfsUid;
import ru.yandex.chemodan.app.djfs.core.util.InstantUtils;
import ru.yandex.chemodan.app.djfs.core.util.UuidUtils;
import ru.yandex.misc.lang.CharsetUtils;

/**
 * @author eoshch
 */
public class PgChangelogDao extends PgShardedDao implements ChangelogDao {
    private final static RowMapper<Changelog> M = (rs, rowNum) -> {
        ChangelogZData zData = ChangelogZData.B.getParser().parseJson(rs.getString("zdata"));
        long uid = rs.getLong("uid");

        return Changelog.builder()
                .path(DjfsResourcePath.cons(uid, rs.getString("path")))
                .resourceType(DjfsResourceType.R.valueOf(rs.getString("type")))
                .operationType(Changelog.OperationType.R.valueOf(rs.getString("op")))
                .version(rs.getLong("version"))
                .fileId(zData.fid)
                .visible(zData.visible == 1)
                .publik(zData.publik == 1)
                .dtime(Option.ofNullable(rs.getTimestamp("dtime")).map(Instant::new))
                .groupId(Option.ofNullable(rs.getString("gid")).map(x -> x.replace("-", "")))
                .groupPath(Option.ofNullable(rs.getString("group_path")))
                .sha256(zData.sha256)
                .md5(zData.md5)
                .size(zData.size)
                .mimetype(zData.mimetype)
                .mediaType(zData.mediaType.map(MediaType.R::valueOf))
                .antiVirusScanStatus(zData.drweb.map(AntiVirusScanStatus.R::fromValue))
                .hasPreview(zData.has_preview)
                .hasExternalSetprop(zData.external_setprop.map(x -> x == 1))
                .exifTime(zData.etime.map(InstantUtils::fromSeconds))
                .modificationTime(zData.mtime.map(InstantUtils::fromSeconds))
                .newPath(zData.newKey.map(x -> DjfsResourcePath.cons(uid, x)))
                .build();
    };

    public PgChangelogDao(PgShardedDaoContext context) {
        super(context);
    }

    @Override
    public void insert(Changelog changelog) {
        String sql = collectStats(changelog)
                + " INSERT INTO disk.changelog (id, uid, path, type, op, version, zdata, dtime, gid, group_path) "
                + "VALUES (?, ?, ?, ?::disk.resource_type, ?::disk.changelog_op_type, ?, ?::json, ?, ?, ?)";

        ChangelogZData zData = makeZdata(changelog);

        jdbcTemplate(changelog.getUid()).update(sql, changelog.getId().toByteArray(), changelog.getUid(),
                changelog.getPath().getPath(), changelog.getResourceType().name().toLowerCase(),
                changelog.getOperationType().name().toLowerCase(), changelog.getVersion(),
                new String(ChangelogZData.B.getSerializer().serializeJson(zData), CharsetUtils.UTF8_CHARSET),
                changelog.getDtime(), changelog.getGroupId().map(UuidUtils::fromHex), changelog.getGroupPath());
    }

    @Override
    public void insert(DjfsUid uid, ListF<Changelog> changelogs) {
        if (changelogs.isEmpty()) {
            return;
        }

        MapSqlParameterSource parameters = new MapSqlParameterSource();
        String template = "(:id_%1$s,:uid_%1$s,:path_%1$s,:type_%1$s::disk.resource_type,"
                + ":op_%1$s::disk.changelog_op_type,:version_%1$s,:zdata_%1$s::json,:dtime_%1$s,"
                + ":gid_%1$s,:group_path_%1$s)";

        StringBuilder values = new StringBuilder();
        for (int i = 0; i < changelogs.size(); i++) {
            if (i != 0) {
                values.append(",");
            }

            Changelog changelog = changelogs.get(i);
            ChangelogZData zData = makeZdata(changelog);
            String serializedZData = new String(ChangelogZData.B.getSerializer().serializeJson(zData),
                    CharsetUtils.UTF8_CHARSET);

            values.append(String.format(template, i + 1));
            parameters.addValue("id_" + (i + 1), changelog.getId().toByteArray());
            parameters.addValue("uid_" + (i + 1), changelog.getUid());
            parameters.addValue("path_" + (i + 1), changelog.getPath().getPath());
            parameters.addValue("type_" + (i + 1), changelog.getResourceType().name().toLowerCase());
            parameters.addValue("op_" + (i + 1), changelog.getOperationType().name().toLowerCase());
            parameters.addValue("version_" + (i + 1), changelog.getVersion());
            parameters.addValue("zdata_" + (i + 1), serializedZData);
            parameters.addValue("dtime_" + (i + 1), changelog.getDtime().getOrNull());
            parameters.addValue("gid_" + (i + 1), changelog.getGroupId().map(UuidUtils::fromHex).getOrNull());
            parameters.addValue("group_path_" + (i + 1), changelog.getGroupPath().getOrNull());
        }

        String sql = collectStats(uid)
                + " INSERT INTO disk.changelog (id, uid, path, type, op, version, zdata, dtime, gid, group_path) "
                + " VALUES " + values;

        jdbcTemplate(uid).update(sql, parameters);
    }

    @Override
    public ListF<Changelog> findAll(DjfsUid uid) {
        String sql = collectStats(uid) + " SELECT * FROM disk.changelog WHERE uid = :uid";
        return jdbcTemplate(uid).query(sql, M, Cf.map("uid", uid));
    }

    @Override
    public void deleteAll(DjfsUid uid) {
        String sql = collectStats(uid) + " DELETE FROM disk.changelog WHERE uid = :uid";
        jdbcTemplate(uid).update(sql, Cf.map("uid", uid));
    }

    private ChangelogZData makeZdata(Changelog changelog) {
        ChangelogZData zData = new ChangelogZData();

        zData.fid = changelog.getFileId();
        zData.publik = changelog.isPublik() ? 1 : 0;
        zData.visible = changelog.isVisible() ? 1 : 0;

        zData.sha256 = changelog.getSha256();
        zData.md5 = changelog.getMd5();
        zData.size = changelog.getSize();

        zData.mimetype = changelog.getMimetype();
        zData.mediaType = changelog.getMediaType().map(MediaType::getStringRepresentation);
        zData.drweb = changelog.getAntiVirusScanStatus().map(AntiVirusScanStatus::getIntRepresentation);

        zData.has_preview = changelog.getHasPreview();
        zData.external_setprop = changelog.getHasExternalSetprop().getOrElse(false) ? Option.of(1) : Option.empty();

        zData.etime = changelog.getExifTime().map(InstantUtils::toSecondsLong);
        zData.mtime = changelog.getModificationTime().map(InstantUtils::toSecondsLong);

        zData.newKey = changelog.getNewPath().map(DjfsResourcePath::getPath);

        return zData;
    }
}
