package ru.yandex.chemodan.app.djfs.core.client;

import java.io.IOException;
import java.net.URI;

import org.apache.http.HttpResponse;
import org.apache.http.client.HttpClient;
import org.apache.http.client.methods.HttpGet;
import org.joda.time.Instant;
import org.joda.time.format.ISODateTimeFormat;

import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.djfs.core.user.DjfsUid;
import ru.yandex.chemodan.util.bender.ISOInstantUnmarshaller;
import ru.yandex.chemodan.util.exception.NotFoundException;
import ru.yandex.misc.bender.Bender;
import ru.yandex.misc.bender.MembersToBind;
import ru.yandex.misc.bender.config.BenderConfiguration;
import ru.yandex.misc.bender.config.CustomMarshallerUnmarshallerFactoryBuilder;
import ru.yandex.misc.bender.custom.ReadableInstantConfigurableMarshaller;
import ru.yandex.misc.bender.parse.BenderJsonParser;
import ru.yandex.misc.io.InputStreamSourceUtils;
import ru.yandex.misc.io.http.UriBuilder;
import ru.yandex.misc.io.http.apache.v4.Abstract200ExtendedResponseHandler;
import ru.yandex.misc.io.http.apache.v4.ApacheHttpClientUtils;

public class DataApiHttpClient {
    private final String baseUrl;
    private final HttpClient httpClient;

    private static final BenderConfiguration benderConfigucation = BenderConfiguration.cons(MembersToBind.WITH_ANNOTATIONS, false,
            CustomMarshallerUnmarshallerFactoryBuilder.cons()
            .add(Instant.class, new ReadableInstantConfigurableMarshaller(ISODateTimeFormat.dateTime()))
            .add(Instant.class, new ISOInstantUnmarshaller())
            .build()
    );

    public static final BenderJsonParser<DataapiMordaBlockList> dataapiMordaBlockListParser =
            Bender.jsonParser(DataapiMordaBlockList.class, benderConfigucation);

    public static final BenderJsonParser<DataapiMordaBlock> dataapiMordaBlockParser =
            Bender.jsonParser(DataapiMordaBlock.class, benderConfigucation);

    public static final BenderJsonParser<ProfileAddress> dataapiProfileAddressParser =
            Bender.jsonParser(ProfileAddress.class, benderConfigucation);

    public DataApiHttpClient(String baseUrl, HttpClient httpClient) {
        this.baseUrl = baseUrl;
        this.httpClient = httpClient;
    }

    public DataapiMordaBlockList fetchNostalgyBlocks(DjfsUid uid, int maxCountToFetch) {
        URI uri = UriBuilder
                .cons(baseUrl)
                .appendPath("/v1/personality/profile/disk/cool_lenta/morda_blocks")
                .addParam("limit", maxCountToFetch)
                .addParam("query.order.asc", "false")
                .addParam("request_total_count", "false")
                .build();

        HttpGet httpGet = new HttpGet(uri);
        httpGet.addHeader("X-Uid", uid.asString());
        return ApacheHttpClientUtils.execute(httpGet, httpClient, new MordaBlockListResponseHandler());
    }

    public DataapiMordaBlockList fetchBlocksFromPool(DjfsUid uid, int offset, int limit) {
        URI uri = UriBuilder
                .cons(baseUrl)
                .appendPath("/v1/personality/profile/disk/cool_lenta/all_blocks")
                .addParam("offset", offset)
                .addParam("limit", limit)
                .addParam("request_total_count", "false")
                .build();

        HttpGet httpGet = new HttpGet(uri);
        httpGet.addHeader("X-Uid", uid.asString());
        return ApacheHttpClientUtils.execute(httpGet, httpClient, new MordaBlockListResponseHandler());
    }

    public DataapiMordaBlock fetchBlockById(DjfsUid uid, String blockId) {
        URI uri = UriBuilder
                .cons(baseUrl)
                .appendPath("/v1/personality/profile/disk/cool_lenta/morda_blocks/" + blockId)
                .build();

        HttpGet httpGet = new HttpGet(uri);
        httpGet.addHeader("X-Uid", uid.asString());

        return ApacheHttpClientUtils.execute(httpGet, httpClient, new MordaBlockResponseHandler());
    }

    public DataapiMordaBlock fetchBlockByIdFromPool(DjfsUid uid, String blockId) {
        URI uri = UriBuilder
                .cons(baseUrl)
                .appendPath("/v1/personality/profile/disk/cool_lenta/all_blocks/" + blockId)
                .build();

        HttpGet httpGet = new HttpGet(uri);
        httpGet.addHeader("X-Uid", uid.asString());

        return ApacheHttpClientUtils.execute(httpGet, httpClient, new MordaBlockResponseHandler());
    }

    public Option<ProfileAddress> fetchAddress(DjfsUid uid, ProfileAddressType addressType) {
        URI uri = UriBuilder
                .cons(baseUrl)
                .appendPath("/v2/personality/profile/addresses/" + addressType.value() + "/")
                .build();

        HttpGet httpGet = new HttpGet(uri);
        httpGet.addHeader("X-Uid", uid.asString());

        return ApacheHttpClientUtils.execute(httpGet, httpClient, new ProfileAddressResponseHandler());
    }

    private class MordaBlockListResponseHandler extends Abstract200ExtendedResponseHandler<DataapiMordaBlockList> {

        @Override
        protected DataapiMordaBlockList handle200Response(HttpResponse response) throws IOException {
            return dataapiMordaBlockListParser.parseJson(
                    InputStreamSourceUtils.wrap(response.getEntity().getContent())
            );
        }
    }

    private class MordaBlockResponseHandler extends Abstract200ExtendedResponseHandler<DataapiMordaBlock> {

        @Override
        public DataapiMordaBlock handleResponse(HttpResponse response) throws IOException {
            if (response.getStatusLine().getStatusCode() == 404) {
                throw new NotFoundException("Block not found");
            }
            return super.handleResponse(response);
        }

        @Override
        protected DataapiMordaBlock handle200Response(HttpResponse response) throws IOException {
            return dataapiMordaBlockParser.parseJson(
                    InputStreamSourceUtils.wrap(response.getEntity().getContent())
            );
        }
    }

    private class ProfileAddressResponseHandler extends Abstract200ExtendedResponseHandler<Option<ProfileAddress>> {

        @Override
        public Option<ProfileAddress> handleResponse(HttpResponse response) throws IOException {
            if (response.getStatusLine().getStatusCode() == 404) {
                return Option.empty();
            }
            return super.handleResponse(response);
        }

        @Override
        protected Option<ProfileAddress> handle200Response(HttpResponse response) throws IOException {
            return Option.of(dataapiProfileAddressParser.parseJson(
                    InputStreamSourceUtils.wrap(response.getEntity().getContent())
            ));
        }
    }
}
