package ru.yandex.chemodan.app.djfs.core.client;

import java.io.IOException;
import java.net.URI;

import org.apache.http.HttpResponse;
import org.apache.http.client.HttpClient;
import org.apache.http.client.ResponseHandler;
import org.apache.http.client.methods.HttpGet;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.bolts.collection.Unit;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.DjfsFileId;
import ru.yandex.chemodan.app.djfs.core.user.DjfsUid;
import ru.yandex.chemodan.util.exception.NotFoundException;
import ru.yandex.misc.enums.StringEnum;
import ru.yandex.misc.io.InputStreamSourceUtils;
import ru.yandex.misc.io.http.UriBuilder;
import ru.yandex.misc.io.http.apache.v4.Abstract200ExtendedResponseHandler;
import ru.yandex.misc.io.http.apache.v4.ApacheHttpClientUtils;
import ru.yandex.misc.lang.StringUtils;


public class DiskSearchHttpClient {
    private final String searchBaseUrl;
    private final String searchProducerBaseUrl;
    private final String searchProducerService;
    private final HttpClient httpClient;

    private static final ListF<FetchParameter> defaultExtractedParamsList = Cf.list(FetchParameter.ID,
            FetchParameter.AESTHETICS, FetchParameter.WIDTH, FetchParameter.HEIGHT, FetchParameter.ORIENTATION);

    public enum FetchParameter implements StringEnum {
        ID("id"),
        AESTHETICS("cost_disk_aethetic_0"),
        WIDTH("width"),
        HEIGHT("height"),
        ORIENTATION("orientation"),
        LATITUDE("latitude"),
        LONGITUDE("longitude");

        private final String value;

        FetchParameter(String value) {
            this.value = value;
        }

        @Override
        public String value() {
            return value;
        }
    }

    public DiskSearchHttpClient(
            HttpClient httpClient, String searchBaseUrl, String searchProducerBaseUrl, String searchProducerService
    ) {
        this.searchBaseUrl = searchBaseUrl;
        this.searchProducerBaseUrl = searchProducerBaseUrl;
        this.searchProducerService = searchProducerService;
        this.httpClient = httpClient;
    }

    public DiskSearchResponse getExtractedData(DjfsUid uid, ListF<DjfsFileId> fileIds, boolean isFastMovedUser) {
        return getExtractedData(uid, fileIds, defaultExtractedParamsList, isFastMovedUser);
    }

    public FaceClustersSnapshot getFaceClusters(DjfsUid uid) {
        return executeGet(
            "/api/faces/clusters", uid, Tuple2List.tuple2List(), faceClustersResponseHandler
        );
    }

    public void reindexFaces(DjfsUid uid) {
        final URI uri = UriBuilder.cons(searchProducerBaseUrl)
                .appendPath("/reindex")
                .addParam("prefix", uid.asString())
                .addParam("service", searchProducerService)
                .build();
        HttpGet httpGet = new HttpGet(uri);
        httpGet.addHeader("X-Uid", uid.asString());
        ApacheHttpClientUtils.execute(httpGet, httpClient, unitResponseHandler);
    }

    public FaceClusterPhotos getClusterPhotos(DjfsUid uid, String clusterId) {
        return executeGet(
            "/api/faces/cluster-photos",
            uid,
            Tuple2List.fromPairs(
                "cluster_id", clusterId,
                "resource_get", "width,height,cost_disk_aethetic_0,resource_id"
            ),
            faceClusterPhotosResponseHandler
        );
    }

    public FacesDeltas getFacesDeltas(DjfsUid uid, long knownVersion) {
        return executeGet(
                "/api/faces/deltas",
                uid,
                Tuple2List.fromPairs(
                        "version", "" + knownVersion,
                        "resource_get", "width,height,cost_disk_aethetic_0,resource_id"
                ),
                facesDeltasResponseHandler
        );
    }

    public DiskSearchResponse getExtractedData(DjfsUid uid,
                                               ListF<DjfsFileId> fileIds,
                                               ListF<FetchParameter> fields,
                                               boolean isFastMovedUser)
    {
        Tuple2List<String, String> params =
            Tuple2List.fromPairs("get", StringUtils.join(fields.map(FetchParameter::value), ","))
            .plus(Cf.list(Tuple2.tuple("fast-moved", "true")).filter(item -> isFastMovedUser))
            .plus(fileIds.map(fileId -> Tuple2.tuple("id", fileId.getValue())));
        return executeGet("get", uid, params, diskSearchResponseHandler);
    }

    private <T> T executeGet(String path,
                             DjfsUid uid,
                             Tuple2List<String, String> params,
                             ResponseHandler<T> responseHandler)
    {
        URI uri =
            UriBuilder.cons(searchBaseUrl)
            .appendPath(path)
            .addParam("uid", uid.asString())
            .addParams(params)
            .build();
        HttpGet httpGet = new HttpGet(uri);
        httpGet.addHeader("X-Uid", uid.asString());
        return ApacheHttpClientUtils.execute(httpGet, httpClient, responseHandler);
    }

    private static class DiskSearchResponseHandler extends Abstract200ExtendedResponseHandler<DiskSearchResponse> {

        @Override
        public DiskSearchResponse handleResponse(HttpResponse response) throws IOException {
            if (response.getStatusLine().getStatusCode() == 404) {
                throw new NotFoundException("Items not found");
            }
            return super.handleResponse(response);
        }

        @Override
        protected DiskSearchResponse handle200Response(HttpResponse response) throws IOException {
            return DiskSearchResponse.jsonParser.parseJson(
                InputStreamSourceUtils.wrap(response.getEntity().getContent())
            );
        }
    }

    private final static DiskSearchResponseHandler diskSearchResponseHandler = new DiskSearchResponseHandler();

    private static class FaceClustersResponseHandler extends Abstract200ExtendedResponseHandler<FaceClustersSnapshot> {
        @Override
        protected FaceClustersSnapshot handle200Response(HttpResponse response) throws IOException {
            return FaceClustersSnapshot.jsonParser.parseJson(
                InputStreamSourceUtils.wrap(response.getEntity().getContent())
            );
        }
    }

    private final static FaceClustersResponseHandler faceClustersResponseHandler = new FaceClustersResponseHandler();

    private static class FaceClusterPhotosResponseHandler
        extends Abstract200ExtendedResponseHandler<FaceClusterPhotos>
    {
        @Override
        protected FaceClusterPhotos handle200Response(HttpResponse response) throws IOException {
            return FaceClusterPhotos.jsonParser.parseJson(
                InputStreamSourceUtils.wrap(response.getEntity().getContent())
            );
        }
    }

    private final static FaceClusterPhotosResponseHandler faceClusterPhotosResponseHandler =
        new FaceClusterPhotosResponseHandler();

    private static class FacesDeltasResponseHandler
        extends Abstract200ExtendedResponseHandler<FacesDeltas>
    {
        @Override
        protected FacesDeltas handle200Response(HttpResponse response) throws IOException {
            return FacesDeltas.jsonParser.parseJson(
                InputStreamSourceUtils.wrap(response.getEntity().getContent())
            );
        }
    }

    private final static FacesDeltasResponseHandler facesDeltasResponseHandler = new FacesDeltasResponseHandler();

    private static class UnitResponseHandler extends Abstract200ExtendedResponseHandler<Unit>
    {
        @Override
        protected Unit handle200Response(HttpResponse response) {
            return Unit.U;
        }
    }

    private final static UnitResponseHandler unitResponseHandler = new UnitResponseHandler();
}
