package ru.yandex.chemodan.app.djfs.core.client;

import org.apache.http.HttpRequestInterceptor;
import org.apache.http.HttpResponseInterceptor;
import org.apache.http.client.HttpClient;
import org.apache.http.impl.client.CloseableHttpClient;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Import;
import org.springframework.context.annotation.Profile;

import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.blackbox.ProdBlackboxContextConfiguration;
import ru.yandex.chemodan.boot.value.OverridableValuePrefix;
import ru.yandex.chemodan.http.LoggingHttpInterceptor;
import ru.yandex.chemodan.mpfs.MpfsClient;
import ru.yandex.chemodan.mpfs.MpfsClientImpl;
import ru.yandex.chemodan.mpfs.MpfsUserInitParamsExtractor;
import ru.yandex.chemodan.util.http.HttpClientConfigurator;

/**
 * @author eoshch
 */
@Configuration
@Profile(ActivateRealClient.PROFILE)
@Import({
        ProdBlackboxContextConfiguration.class
})
public class DjfsRealClientContextConfiguration {
    @Bean
    @Qualifier("mpfs")
    @OverridableValuePrefix("mpfs")
    public HttpClientConfigurator mpfsHttpClientConfigurator() {
        return new HttpClientConfigurator();
    }

    @Bean
    @Qualifier("mpfs_long")
    @OverridableValuePrefix("mpfs_long")
    public HttpClientConfigurator mpfsLongHttpClientConfigurator() {
        return new HttpClientConfigurator();
    }

    @Bean
    public MpfsClient mpfsClient(@Value("${mpfs.host}") String host,
                                 @Value("${mpfs.ignore-user-not-initialized-exception}") boolean ignoreUserNotInitializedException)
    {
        HttpClient httpClient = mpfsHttpClientConfigurator().createBuilder()
                .multiThreaded()
                .withInterceptorLast((HttpRequestInterceptor) LoggingHttpInterceptor.MPFS_CLIENT_INTERCEPTOR)
                .withInterceptorFirst((HttpResponseInterceptor) LoggingHttpInterceptor.MPFS_CLIENT_INTERCEPTOR)
                .build();

        HttpClient longRequestsHttpClient = mpfsLongHttpClientConfigurator().createBuilder()
                .multiThreaded()
                .withInterceptorLast((HttpRequestInterceptor) LoggingHttpInterceptor.MPFS_CLIENT_INTERCEPTOR)
                .withInterceptorFirst((HttpResponseInterceptor) LoggingHttpInterceptor.MPFS_CLIENT_INTERCEPTOR)
                .build();

        return new MpfsClientImpl(httpClient, longRequestsHttpClient, host,
                MpfsUserInitParamsExtractor.empty(), false, Option.empty(), Option.empty(), ignoreUserNotInitializedException);
    }

    @Bean
    public DataApiHttpClient dataApiHttpClient(
            @Value("${dataapi.host}") String host,
            @Qualifier("dataapi") HttpClientConfigurator configurator)
    {
        return new DataApiHttpClient(host, configurator.configure());
    }

    @Bean
    public DiskSearchHttpClient diskSearchHttpClient(
            @Value("${disk_search.host}") String searchHost,
            @Value("${disk_search.producer.base-url}") String searchProducerBaseUrl,
            @Value("${disk_search.producer.service}") String searchProducerService,
            @Qualifier("disk_search") HttpClientConfigurator configurator)
    {
        CloseableHttpClient httpClient = configurator.createBuilder()
                .withInterceptorLast((HttpRequestInterceptor) LoggingHttpInterceptor.MPFS_CLIENT_INTERCEPTOR)
                .withInterceptorFirst((HttpResponseInterceptor) LoggingHttpInterceptor.MPFS_CLIENT_INTERCEPTOR)
                .build();
        return new DiskSearchHttpClient(httpClient, searchHost, searchProducerBaseUrl, searchProducerService);
    }

    @Bean
    public OperationCallbackHttpClient operationCallbackHttpClient(
            @Qualifier("operation_callback") HttpClientConfigurator configurator)
    {
        return new OperationCallbackHttpClient(configurator.configure());
    }

    @Bean
    @Qualifier("dataapi")
    @OverridableValuePrefix("dataapi")
    public HttpClientConfigurator dataApiHttpClientConfigurator() {
        return new HttpClientConfigurator();
    }

    @Bean
    @Qualifier("disk_search")
    @OverridableValuePrefix("disk_search")
    public HttpClientConfigurator diskSearchHttpClientConfigurator() {
        return new HttpClientConfigurator();
    }

    @Bean
    @Qualifier("operation_callback")
    @OverridableValuePrefix("operation_callback")
    public HttpClientConfigurator operationCallbackHttpClientConfigurator() {
        return new HttpClientConfigurator();
    }

    @Bean
    public LogReaderHttpClient logReaderHttpClient(
            @Value("${logreader.host}") String host,
            @Qualifier("logreader") HttpClientConfigurator configurator)
    {
        return new LogReaderHttpClient(host, configurator.configure());
    }

    @Bean
    @Qualifier("logreader")
    @OverridableValuePrefix("logreader")
    public HttpClientConfigurator logReaderHttpClientConfigurator() {
        return new HttpClientConfigurator();
    }
}
