package ru.yandex.chemodan.app.djfs.core.db.mongo;

import java.util.Arrays;
import java.util.List;
import java.util.stream.Collectors;

import com.mongodb.MongoTimeoutException;
import com.mongodb.ServerAddress;
import org.bson.BsonDocument;
import org.bson.BsonInt32;
import org.bson.BsonInt64;
import org.bson.BsonNumber;
import org.bson.json.JsonWriterSettings;
import org.joda.time.format.ISODateTimeFormat;

import ru.yandex.bolts.function.Function0;
import ru.yandex.bolts.function.Function0V;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

/**
 * @author eoshch
 */
public class MongoUtil {
    private static final Logger logger = LoggerFactory.getLogger(MongoUtil.class);

    private static final JsonWriterSettings settings = JsonWriterSettings.builder()
            .indent(false)
            .int64Converter((value, writer) -> writer.writeNumber(value.toString()))
            .dateTimeConverter(
                    (value, writer) -> writer.writeString(ISODateTimeFormat.dateTime().withZoneUTC().print(value)))
            .build();

    private static ServerAddress parseSingleHost(String host) {
        if (host.contains(":")) {
            String[] parts = host.split(":");
            return new ServerAddress(parts[0], Integer.parseInt(parts[1]));
        }
        return new ServerAddress(host);
    }

    public static List<ServerAddress> parseHost(String host) {
        return Arrays.stream(host.split(",")).map(MongoUtil::parseSingleHost).collect(Collectors.toList());
    }

    /**
     * Retries MongoTimeoutException
     * https://jira.mongodb.org/browse/JAVA-499
     */
    public static void retryTimeoutOnce(Function0V f) {
        try {
            f.apply();
        } catch (MongoTimeoutException e) {
            logger.info("Mongo request failed with timeout, will retry", e);
            f.apply();
        }
    }

    /**
     * Retries MongoTimeoutException
     * https://jira.mongodb.org/browse/JAVA-499
     */
    public static <T> T retryTimeoutOnce(Function0<T> f) {
        try {
            return f.apply();
        } catch (MongoTimeoutException e) {
            logger.info("Mongo request failed with timeout, will retry", e);
            return f.apply();
        }
    }

    public static String serializeToJson(BsonDocument document) {
        return document.toJson(settings);
    }

    public static BsonNumber toNumber(int value) {
        return new BsonInt32(value);
    }

    public static BsonNumber toNumber(long value) {
        if (Integer.MIN_VALUE < value && value < Integer.MAX_VALUE) {
            return new BsonInt32((int) value);
        }
        return new BsonInt64(value);
    }
}
