package ru.yandex.chemodan.app.djfs.core.db.mongo.logging;

import java.text.DecimalFormat;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.TimeUnit;

import com.mongodb.event.CommandEvent;
import com.mongodb.event.CommandFailedEvent;
import com.mongodb.event.CommandListener;
import com.mongodb.event.CommandStartedEvent;
import com.mongodb.event.CommandSucceededEvent;
import org.bson.BsonDocument;
import org.bson.BsonValue;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.chemodan.log.CloseableMdc;
import ru.yandex.misc.log.mlf.Level;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

/**
 * @author eoshch
 */
public class MongoCommandListener implements CommandListener {
    private static final Logger requestsLogger = LoggerFactory.getLogger("requests");
    private static final Logger logger = LoggerFactory.getLogger(MongoCommandListener.class);
    private static final DecimalFormat msFormat = new DecimalFormat("#0.000");

    private static final MapF<String, String> replicaSetNameMap = Cf.map(
            "disk-mpfs-sys","system",
            "disk-misc-01", "common"
    );

    private final ConcurrentHashMap<Integer, Tuple2<String, BsonDocument>> commands = new ConcurrentHashMap<>();
    private final ReplicaSetCache replicaSetCache;

    public MongoCommandListener(ReplicaSetCache replicaSetCache) {
        this.replicaSetCache = replicaSetCache;
    }

    @Override
    public void commandStarted(CommandStartedEvent event) {
        commands.put(event.getRequestId(), Tuple2.tuple(event.getDatabaseName(), event.getCommand().clone()));
    }

    @Override
    public void commandSucceeded(CommandSucceededEvent event) {
        Tuple2<String, BsonDocument> tuple = commands.remove(event.getRequestId());
        long elapsedTimeMs = event.getElapsedTime(TimeUnit.MILLISECONDS);
        try (final CloseableMdc.Instance ignored = CloseableMdc.put("module", "logging")) {
            try (final CloseableMdc.Instance ignored2 = CloseableMdc.put("name", "mpfs.requests")) {
                if (requestsLogger.isEnabledFor(Level.INFO)) {
                    requestsLogger.info(format(true, tuple._1, tuple._2, event, elapsedTimeMs));
                }
            }
        }
    }

    @Override
    public void commandFailed(CommandFailedEvent event) {
        Tuple2<String, BsonDocument> tuple = commands.remove(event.getRequestId());
        long elapsedTimeMs = event.getElapsedTime(TimeUnit.MILLISECONDS);
        try (final CloseableMdc.Instance ignored = CloseableMdc.put("module", "logging")) {
            try (final CloseableMdc.Instance ignored2 = CloseableMdc.put("name", "mpfs.requests")) {
                if (requestsLogger.isEnabledFor(Level.INFO)) {
                    requestsLogger.info(format(false, tuple._1, tuple._2, event, elapsedTimeMs));
                }
                if (logger.isEnabledFor(Level.ERROR)) {
                    logger.error(event.getThrowable());
                }
            }
        }
    }

    private String format(boolean completed, String databaseName, BsonDocument request, CommandEvent event,
            long elapsedTimeMs)
    {
        StringBuilder sb = new StringBuilder();

        String replicaSetName = replicaSetCache.resolve(event.getConnectionDescription().getServerAddress())
                .getOrElse("UNKNOWN");

        if (replicaSetNameMap.containsKeyTs(replicaSetName)) {
            replicaSetName = replicaSetNameMap.getTs(replicaSetName);
        }

        String commandName = event.getCommandName();
        String collectionName = Option.ofNullable(request.get("collection"))
                .orElse(Option.ofNullable(request.get(commandName)))
                .map(x -> x.isString() ? x.asString().getValue() : x.toString()).getOrElse("UNKNOWN");

        sb.append(completed ? "completed " : "failed ");
        sb.append(replicaSetName);
        sb.append('.');
        sb.append(databaseName);
        sb.append('.');
        sb.append(collectionName);
        sb.append('.');
        sb.append(commandName);
        sb.append('(');
        sb.append(request.toJson());

        sb.append(", host=");
        sb.append(event.getConnectionDescription().getServerAddress().toString());

        if (event instanceof CommandSucceededEvent) {
            Option<Long> cursorId = getCursorId(((CommandSucceededEvent) event).getResponse());
            if (cursorId.isPresent()) {
                sb.append(", cursor=");
                sb.append(cursorId.get());
            }
        }

        // mode has to be last because of mpfs-stat-mongodb-requests.py
        sb.append(", mode=");
        sb.append(event.getConnectionDescription().getServerType().toString());
        sb.append(").0 ");

        if (commandName.equals("find") || commandName.equals("getMore")) {
            if (event instanceof CommandSucceededEvent) {
                sb.append(getReturnedDocumentCount(((CommandSucceededEvent) event).getResponse()));

                // packet byte size not supported for now
                sb.append(" 0 ");
            }
        }

        sb.append(msFormat.format(elapsedTimeMs / 1000d));
        return sb.toString();
    }

    private Option<Long> getCursorId(BsonDocument response) {
        BsonValue cursor = response.get("cursor");
        if (cursor == null) {
            return Option.empty();
        }
        if (!cursor.isDocument()) {
            return Option.empty();
        }
        BsonValue rawId = cursor.asDocument().get("id");
        if (!rawId.isInt64()) {
            return Option.empty();
        }
        long id = rawId.asInt64().getValue();
        if (id == 0) {
            return Option.empty();
        }
        return Option.of(id);
    }

    private int getReturnedDocumentCount(BsonDocument response) {
        BsonValue cursor = response.get("cursor");
        if (cursor == null) {
            return 0;
        }
        if (!cursor.isDocument()) {
            return 0;
        }
        BsonValue firstBatch = cursor.asDocument().get("firstBatch");
        if (firstBatch != null && firstBatch.isArray()) {
            return firstBatch.asArray().size();
        }
        BsonValue nextBatch = cursor.asDocument().get("nextBatch");
        if (nextBatch != null && nextBatch.isArray()) {
            return nextBatch.asArray().size();
        }
        return 0;
    }
}
