package ru.yandex.chemodan.app.djfs.core.db.pg;

import java.text.DecimalFormat;

import ru.yandex.chemodan.log.CloseableMdc;
import ru.yandex.chemodan.util.jdbc.logging.LastAccessedDsAwareQueryInterceptor;
import ru.yandex.chemodan.util.jdbc.logging.LastAccessedDsHolder;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.log.mlf.Level;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.spring.jdbc.intercept.QueryInfo;
import ru.yandex.misc.spring.jdbc.intercept.QueryResultInfo;

/**
 * @author eoshch
 */
public class DjfsLoggingQueryInterceptor implements LastAccessedDsAwareQueryInterceptor {
    private static final Logger requestsLogger = LoggerFactory.getLogger("requests");
    private static final DecimalFormat msFormat = new DecimalFormat("#0.000000");

    private final LastAccessedDsHolder lastDs;

    public DjfsLoggingQueryInterceptor() {
        lastDs = new LastAccessedDsHolder();
    }

    @Override
    public Object queryStarted(QueryInfo q) {
        return System.currentTimeMillis();
    }

    @Override
    public void queryCompleted(Object tag, QueryResultInfo result) {
        long elapsedTimeMs = System.currentTimeMillis() - (Long) tag;

        if (requestsLogger.isEnabledFor(Level.INFO)) {
            StringBuilder sb = new StringBuilder();
            sb.append(result.isSuccessfully() ? "completed" : "failed");
            sb.append(" dbname=");
            sb.append(lastDs.get().filterMap(x -> x.url.getDbName()).getOrElse("unknown"));
            sb.append(" user=xxxxxxxxxxxxxxxxxxxx password=xxxxxxxxxxxxxxxxxxxx host=");
            sb.append(lastDs.get().filterMap(x -> x.url.getHost()).getOrElse("unknown"));
            sb.append(" port=");
            sb.append(lastDs.get().filterMap(x -> x.url.getPort()).map(Object::toString).getOrElse("unknown"));
            sb.append(" is_master=");
            sb.append(lastDs.get().map(x -> x.isMaster ? "true" : "false").getOrElse("unknown"));

            String params = lastDs.get().map(x -> x.url.getParameters().mkString(" ", "=")).getOrNull();
            if (!StringUtils.isBlank(params)) {
                sb.append(" ");
                sb.append(params);
            }

            sb.append(" \"");
            sb.append(result.getQueryInfo().getQFormatted());
            sb.append("\" ");
            sb.append(result.getRowCount().getOrElse(0));
            sb.append(" ");
            sb.append(msFormat.format(elapsedTimeMs / 1000d));

            try (final CloseableMdc.Instance ignored = CloseableMdc.put("module", "logging")) {
                try (final CloseableMdc.Instance ignored2 = CloseableMdc.put("name", "mpfs.requests.postgres")) {
                    requestsLogger.info(sb.toString());
                }
            }
        }
    }

    @Override
    public LastAccessedDsHolder getLastAccessedDsHolder() {
        return lastDs;
    }
}
