package ru.yandex.chemodan.app.djfs.core.db.pg;

import org.joda.time.Duration;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Import;

import ru.yandex.chemodan.util.jdbc.DataSourceProperties;
import ru.yandex.chemodan.util.jdbc.JdbcDatabaseConfigurator;
import ru.yandex.chemodan.util.jdbc.JdbcDatabaseConfiguratorContextConfiguration;
import ru.yandex.chemodan.util.jdbc.JdbcTemplateParams;
import ru.yandex.chemodan.util.retry.RetryConf;
import ru.yandex.chemodan.util.sharpei.SharpeiCachingManager;
import ru.yandex.chemodan.util.sharpei.SharpeiClient;
import ru.yandex.chemodan.util.sharpei.SharpeiClientWithRetries;
import ru.yandex.chemodan.util.sharpei.SharpeiManager;
import ru.yandex.commune.db.shard2.ShardManager2;

/**
 * @author eoshch
 */
@Import({
        JdbcDatabaseConfiguratorContextConfiguration.class,
        DjfsRealPgContextConfiguration.class,
})
public class DjfsPgContextConfiguration {
    @Bean
    public SharpeiShardResolver sharpeiShardResolver(SharpeiManager cachingManager)
    {
        return new SharpeiShardResolver(cachingManager);
    }

    @Bean
    public SharpeiManager sharpeiManager(
            @Value("${sharpei.client-cache-ttl-minutes}") Integer ttl,
            @Value("${sharpei.disable-cache:-false}") boolean disableSharpeiCache,
            SharpeiClient sharpeiClient
    )
    {
        SharpeiClientWithRetries retryingClient = new SharpeiClientWithRetries(sharpeiClient,
                    new RetryConf(9, Duration.millis(100)), new RetryConf(2, Duration.ZERO));
        if (disableSharpeiCache) {
            return new SharpeiManager(retryingClient);
        } else {
            return new SharpeiCachingManager(() -> ttl, retryingClient);
        }
    }

    @Bean
    public ShardManager2 shardManager2(SharpeiShardResolver sharpeiShardResolver,
            DataSourceProperties dataSourceProperties,
            SharpeiClient sharpeiClient, JdbcDatabaseConfiguratorContextConfiguration jdbcDatabaseConfiguratorFactory)
    {
        JdbcDatabaseConfigurator jdbcDatabaseConfigurator = jdbcDatabaseConfiguratorFactory
                .consSharpeiJdbcConfigurator(dataSourceProperties, sharpeiClient);

        JdbcTemplateParams jdbcTemplateParams = jdbcDatabaseConfigurator.consTemplateParams();
        jdbcTemplateParams = jdbcTemplateParams.withPreparedStatementSetterF(DjfsArgPreparedStatementSetter::new);
        return jdbcDatabaseConfigurator.configureSharded(sharpeiShardResolver, jdbcTemplateParams,
                DjfsLoggingQueryInterceptor::new);
    }

    @Bean
    public PgShardResolver postgresqlShardResolver(ShardManager2 shardManager2,
            SharpeiShardResolver sharpeiShardResolver)
    {
        return new PgShardResolver(shardManager2, sharpeiShardResolver);
    }

    @Bean
    public PgShardedDaoContext pgShardedDaoContext(PgShardResolver pgShardResolver) {
        return new PgShardedDaoContext(pgShardResolver);
    }

    @Bean
    public TransactionUtils transactionUtils(PgShardResolver pgShardResolver) {
        return new TransactionUtils(pgShardResolver);
    }
}
