package ru.yandex.chemodan.app.djfs.core.db.pg;

import java.sql.Array;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Types;
import java.util.UUID;

import org.joda.time.Instant;
import org.postgresql.geometric.PGpoint;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.djfs.core.util.UuidUtils;
import ru.yandex.misc.bender.BenderMapper;
import ru.yandex.misc.geo.Coordinates;

/**
 * @author eoshch
 */
public class ResultSetUtils {
    public static Option<Integer> getIntO(ResultSet rs, String column) throws SQLException {
        int result = rs.getInt(column);
        return rs.wasNull() ? Option.empty() : Option.of(result);
    }

    public static Option<Long> getLongO(ResultSet rs, String column) throws SQLException {
        long result = rs.getLong(column);
        return rs.wasNull() ? Option.empty() : Option.of(result);
    }

    public static Option<Coordinates> getCoordinatesO(ResultSet rs, String column) throws SQLException {
        PGpoint result = (PGpoint) rs.getObject(column);
        if (rs.wasNull()) {
            return Option.empty();
        }
        return Option.of(new Coordinates(result.x, result.y));
    }

    public static Option<String> getStringO(ResultSet rs, String column) throws SQLException {
        return Option.ofNullable(rs.getString(column));
    }

    public static UUID getUuid(ResultSet rs, String column) throws SQLException {
        return UuidUtils.from(rs.getString(column));
    }

    public static Option<UUID> getUuidO(ResultSet rs, String column) throws SQLException {
        return Option.ofNullable(rs.getString(column)).map(UuidUtils::from);
    }

    public static Instant getInstant(ResultSet rs, String column) throws SQLException {
        return new Instant(rs.getTimestamp(column));
    }

    public static Option<Instant> getInstantO(ResultSet rs, String column) throws SQLException {
        return Option.ofNullable(rs.getTimestamp(column)).map(Instant::new);
    }

    public static Option<Double> getDoubleO(ResultSet rs, String column) throws SQLException {
        double result = rs.getDouble(column);
        return rs.wasNull() ? Option.empty() : Option.of(result);
    }

    public static Option<Boolean> getBooleanO(ResultSet rs, String column) throws SQLException {
        boolean result = rs.getBoolean(column);
        return rs.wasNull() ? Option.empty() : Option.of(result);
    }

    @SuppressWarnings("unchecked")
    public static <T> Option<T[]> getArrayO(ResultSet rs, String column, Class<T> aClass) throws SQLException {
        Array result = rs.getArray(column);
        return result == null ? Option.empty() : Option.of((T[]) result.getArray());
    }

    public static Option<ListF<String>> getStringArrayO(ResultSet rs, String column) throws SQLException {
        Array resultArray = rs.getArray(column);
        if (resultArray == null) {
            return Option.empty();
        }
        if (resultArray.getBaseType() != Types.VARCHAR) {
            throw new IllegalArgumentException("Wrong array type: " + resultArray.getBaseType());
        }
        ListF<String> result = Cf.list(((String[]) resultArray.getArray()));
        return rs.wasNull() ? Option.empty() : Option.of(result);
    }

    public static <T> Option<T> getObjectO(ResultSet rs, String column, Class<T> aClass) throws SQLException {
        BenderMapper mapper = new BenderMapper();
        byte[] jsonData = rs.getBytes(column);
        return Option.when(jsonData != null && jsonData.length > 0, () -> mapper.parseJson(aClass, jsonData));
    }
}
