package ru.yandex.chemodan.app.djfs.core.diskinfo;

import lombok.RequiredArgsConstructor;
import org.joda.time.Instant;

import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.djfs.core.db.EntityAlreadyExistsException;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.DjfsResourcePath;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.DjfsResourceType;
import ru.yandex.chemodan.app.djfs.core.user.DjfsUid;
import ru.yandex.chemodan.app.djfs.core.util.InstantUtils;

/**
 * @author eoshch
 */
@RequiredArgsConstructor
public class DiskInfoManager {

    public static final String GEO_INDEXER_SYNC_TIMESTAMP_PATH = "/geo_indexer_sync_finished";

    private static final String GEO_ALBUMS_GENERATING = "/geo_albums_generation";

    private final DiskInfoDao diskInfoDao;

    public void ensureRootExists(DjfsUid uid) {
        DiskInfo root = DiskInfo.builder()
                .id(DjfsResourcePath.getPgId(uid, "/"))
                .uid(uid)
                .path("/")
                .type(DjfsResourceType.DIR)
                .build();

        try {
            diskInfoDao.insert(root);
        } catch (EntityAlreadyExistsException e) {
            // we want to be sure root exists
        }
    }

    public void ensureLimitExists(DjfsUid uid) {
        DiskInfo limit = DiskInfo.builder()
                .id(DjfsResourcePath.getPgId(uid, "/limit"))
                .parentId(Option.of(DjfsResourcePath.getPgId(uid, "/")))
                .uid(uid)
                .path("/limit")
                .type(DjfsResourceType.FILE)
                .data(Option.of(DiskInfoData.longData(1L)))
                .version(Option.of(InstantUtils.toVersion(Instant.now())))
                .build();

        try {
            diskInfoDao.insert(limit);
        } catch (EntityAlreadyExistsException e) {
            // we want to be sure limit exists
        }
    }

    public void setIndexerSyncGeoTimestamp(DjfsUid uid, Instant timestamp) {
        DiskInfo limit = DiskInfo.builder()
                .id(DjfsResourcePath.getPgId(uid, GEO_INDEXER_SYNC_TIMESTAMP_PATH))
                .parentId(Option.of(DjfsResourcePath.getPgId(uid, "/")))
                .uid(uid)
                .path(GEO_INDEXER_SYNC_TIMESTAMP_PATH)
                .type(DjfsResourceType.FILE)
                .data(Option.of(DiskInfoData.longData(InstantUtils.toSecondsLong(timestamp))))
                .version(Option.of(InstantUtils.toVersion(Instant.now())))
                .build();

        try {
            diskInfoDao.insert(limit);
        } catch (EntityAlreadyExistsException ignore) {
        }
    }

    public Option<Instant> getIndexerSyncGeoTimestamp(DjfsUid uid) {
        return diskInfoDao.find(uid, GEO_INDEXER_SYNC_TIMESTAMP_PATH)
            .flatMapO(DiskInfo::getData).filter(DiskInfoData::isLongData)
            .map(DiskInfoData::getLongValue).map(InstantUtils::fromSeconds);
    }

    public void enableGeoAlbumsGenerationInProgress(DjfsUid uid) {
        setGeoAlbumGenerationInProgressValue(uid, true);
    }

    public void disableGeoAlbumsGenerationInProgress(DjfsUid uid) {
        setGeoAlbumGenerationInProgressValue(uid, false);
    }

    public boolean isGeoAlbumsGeneratingInProgress(DjfsUid uid) {
        return diskInfoDao.find(uid, GEO_ALBUMS_GENERATING).flatMapO(DiskInfo::getData).filter(DiskInfoData::isJsonData)
                .map(DiskInfoData::getJsonData).flatMapO(DiskInfoData.JsonData::getGeoAlbumsGenerationInProgress)
                .getOrElse(Boolean.FALSE);
    }

    private void setGeoAlbumGenerationInProgressValue(DjfsUid uid, boolean value) {
        DiskInfo diskInfo = DiskInfo.builder()
                .id(DjfsResourcePath.getPgId(uid, GEO_ALBUMS_GENERATING))
                .parentId(Option.of(DjfsResourcePath.getPgId(uid, "/")))
                .uid(uid)
                .path(GEO_ALBUMS_GENERATING)
                .type(DjfsResourceType.FILE)
                .data(Option.of(DiskInfoData.jsonData(new DiskInfoData.JsonData(Option.of(value)))))
                .version(Option.of(InstantUtils.toVersion(Instant.now())))
                .build();

        diskInfoDao.insertOrUpdateDiskInfoValue(diskInfo);
    }
}
