package ru.yandex.chemodan.app.djfs.core.diskinfo;

import java.util.List;

import lombok.AllArgsConstructor;
import lombok.Builder;
import lombok.EqualsAndHashCode;
import lombok.ToString;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.Either;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.djfs.core.db.mongo.DjfsBenderFactory;
import ru.yandex.chemodan.app.djfs.core.db.mongo.UnsupportedMongoDataException;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.DjfsResourceType;
import ru.yandex.chemodan.app.djfs.core.user.DjfsUid;
import ru.yandex.chemodan.app.djfs.core.util.UuidUtils;
import ru.yandex.commune.json.JsonArray;
import ru.yandex.commune.json.JsonNumber;
import ru.yandex.commune.json.JsonObject;
import ru.yandex.commune.json.JsonString;
import ru.yandex.commune.json.JsonValue;
import ru.yandex.commune.mongo3.MongoBenderParserSerializer;
import ru.yandex.commune.mongo3.bender.MongoId;
import ru.yandex.misc.bender.annotation.BenderBindAllFields;
import ru.yandex.misc.lang.Assume;

/**
 * @author eoshch
 */
@BenderBindAllFields
@Builder
@AllArgsConstructor
@ToString
@EqualsAndHashCode
public class MongoDiskInfo {
    public static final MongoBenderParserSerializer<String, MongoDiskInfo> B =
            DjfsBenderFactory.createForMongo(MongoDiskInfo.class);

    private static final Assume A = new Assume() {
        @Override
        public void fail(String message) {
            throw new UnsupportedMongoDataException(message);
        }
    };

    @MongoId
    private final String id;
    private final String uid;
    private final Option<String> parent;
    private final Option<Long> data;
    private final Option<Long> version;
    private final String key;
    private final String type;

    public static Either<Long, MapF<String, Either<String, Integer>>> toData(JsonValue jsonValue) {
        if (jsonValue instanceof JsonNumber) {
            return Either.left(((JsonNumber) jsonValue).longValue());
        }

        if (jsonValue instanceof JsonObject) {
            A.isTrue(((JsonObject) jsonValue).isEmpty());
            return Either.right(Cf.hashMap());
        }

        if (jsonValue instanceof JsonArray) {
            MapF<String, Either<String, Integer>> map = Cf.hashMap();
            for (JsonValue entry : ((JsonArray) jsonValue).getArray()) {
                A.isTrue(entry instanceof JsonArray);
                @SuppressWarnings("ConstantConditions")
                List<JsonValue> pair = ((JsonArray) entry).getArray();
                A.sizeIs(2, pair);
                A.isTrue(pair.get(0) instanceof JsonString);

                JsonString key = (JsonString) pair.get(0);
                JsonValue value = pair.get(1);

                if (value instanceof JsonNumber) {
                    map.put(key.getString(), Either.right(((JsonNumber) value).intValue()));
                } else if (pair.get(1) instanceof JsonString) {
                    map.put(key.getString(), Either.left(((JsonString) value).getString()));
                } else {
                    A.fail();
                }
            }
            return Either.right(map);
        }
        throw new UnsupportedMongoDataException();
    }

    public DiskInfo to() {
        return DiskInfo.builder()
                .id(UuidUtils.fromHex(id))
                .uid(DjfsUid.cons(uid))
                .parentId(parent.map(UuidUtils::fromHex))
                .data(data.map(DiskInfoData::longData))
                .version(version)
                .path(key)
                .type(DjfsResourceType.R.valueOf(type))
                .build();
    }

    public static MongoDiskInfo cons(DiskInfo diskInfo) {
        return MongoDiskInfo.builder()
                .id(UuidUtils.toHexString(diskInfo.getId()))
                .uid(diskInfo.getUid().asString())
                .parent(diskInfo.getParentId().map(UuidUtils::toHexString))
                .key(diskInfo.getPath())
                .type(diskInfo.getType().name().toLowerCase())
                .data(diskInfo.getData().filter(DiskInfoData::isLongData).map(DiskInfoData::getLongValue))
                .version(diskInfo.getVersion())
                .build();
    }
}
