package ru.yandex.chemodan.app.djfs.core.diskinfo;

import com.mongodb.ErrorCategory;
import com.mongodb.MongoWriteException;
import com.mongodb.client.MongoCollection;
import com.mongodb.client.model.Filters;
import com.mongodb.client.model.Projections;
import com.mongodb.client.model.UpdateOptions;
import com.mongodb.client.model.Updates;
import org.bson.BsonDocument;
import org.joda.time.Instant;

import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.internal.NotImplementedException;
import ru.yandex.chemodan.app.djfs.core.db.EntityAlreadyExistsException;
import ru.yandex.chemodan.app.djfs.core.db.mongo.MongoShardedDao;
import ru.yandex.chemodan.app.djfs.core.db.mongo.MongoShardedDaoContext;
import ru.yandex.chemodan.app.djfs.core.db.mongo.MongoUtil;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.DjfsResourcePath;
import ru.yandex.chemodan.app.djfs.core.user.DjfsUid;
import ru.yandex.chemodan.app.djfs.core.util.InstantUtils;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

/**
 * @author eoshch
 */
public class MongoDiskInfoDao extends MongoShardedDao<String, MongoDiskInfo> implements DiskInfoDao {
    private static final Logger logger = LoggerFactory.getLogger(MongoDiskInfoDao.class);

    public MongoDiskInfoDao(MongoShardedDaoContext dependencies) {
        super(dependencies, "disk_info", "disk_info", MongoDiskInfo.B);
    }

    @Override
    public Option<DiskInfo> find(DjfsUid uid, String path) {
        return collectionX(uid).findOne(Filters.eq("_id", DjfsResourcePath.getMongoId(uid, path)))
                .map(MongoDiskInfo::to);
    }

    public ListF<DiskInfo> find(DjfsUid uid) {
        return collectionX(uid).find(Filters.eq("uid", uid.asString())).map(MongoDiskInfo::to);
    }

    @Override
    public void insert(DiskInfo diskInfo) {
        try {
            collectionX(diskInfo).insertOne(MongoDiskInfo.cons(diskInfo));
        } catch (MongoWriteException e) {
            if (ErrorCategory.fromErrorCode((e.getError().getCode())) == ErrorCategory.DUPLICATE_KEY) {
                logger.warn("MongoDiskInfoDao.insert(DiskInfo) handled exception for uid "
                        + diskInfo.getUid().asString() + " id " + diskInfo.getId() + " : ", e);
                throw new EntityAlreadyExistsException("id=" + diskInfo.getId() + " uid="
                        + diskInfo.getUid().asString(), e);
            }
            throw e;
        }
    }

    @Override
    public Option<Long> findLimit(DjfsUid uid) {
        MongoCollection<BsonDocument> collection = collection(uid);
        BsonDocument r = MongoUtil.retryTimeoutOnce(
                () -> collection.find(
                        Filters.and(
                                Filters.eq("_id", DjfsResourcePath.getMongoId(uid, "/limit")),
                                Filters.eq("uid", uid.asString()),
                                Filters.eq("key", "/limit")
                        )).projection(
                        Projections.fields(
                                Projections.include("data"),
                                Projections.excludeId()
                        )).first());
        return Option.ofNullable(r).map(x -> x.asDocument().getNumber("data").longValue());
    }

    @Override
    public Option<Long> findTotalUsed(DjfsUid uid) {
        MongoCollection<BsonDocument> collection = collection(uid);
        BsonDocument r = MongoUtil.retryTimeoutOnce(
                () -> collection.find(
                        Filters.and(
                                Filters.eq("_id", DjfsResourcePath.getMongoId(uid, "/total_size")),
                                Filters.eq("uid", uid.asString()),
                                Filters.eq("key", "/total_size")
                        )).projection(
                        Projections.fields(
                                Projections.include("data"),
                                Projections.excludeId()
                        )).first());
        return Option.ofNullable(r).map(x -> x.asDocument().getNumber("data").longValue());
    }

    @Override
    public Option<Long> findTrashUsed(DjfsUid uid) {
        MongoCollection<BsonDocument> collection = collection(uid);
        BsonDocument r = MongoUtil.retryTimeoutOnce(
                () -> collection.find(
                        Filters.and(
                                Filters.eq("_id", DjfsResourcePath.getMongoId(uid, "/trash_size")),
                                Filters.eq("uid", uid.asString()),
                                Filters.eq("key", "/trash_size")
                        )).projection(
                        Projections.fields(
                                Projections.include("data"),
                                Projections.excludeId()
                        )).first());
        return Option.ofNullable(r).map(x -> x.asDocument().getNumber("data").longValue());
    }

    @Override
    public Option<Long> findFileCount(DjfsUid uid) {
        MongoCollection<BsonDocument> collection = collection(uid);
        BsonDocument r = MongoUtil.retryTimeoutOnce(
                () -> collection.find(
                        Filters.and(
                                Filters.eq("_id", DjfsResourcePath.getMongoId(uid, "/files_count")),
                                Filters.eq("uid", uid.asString()),
                                Filters.eq("key", "/files_count")
                        )).projection(
                        Projections.fields(
                                Projections.include("data"),
                                Projections.excludeId()
                        )).first());
        return Option.ofNullable(r).map(x -> x.asDocument().getNumber("data").longValue());
    }

    @Override
    public Option<DownloadTraffic> findDownloadTraffic(DjfsUid uid) {
        throw new NotImplementedException();
    }

    @Override
    public void incrementTotalUsed(DjfsUid uid, long delta) {
        MongoCollection<BsonDocument> collection = collection(uid);
        MongoUtil.retryTimeoutOnce(
                () -> collection.updateOne(
                        Filters.and(
                                Filters.eq("_id", DjfsResourcePath.getMongoId(uid, "/total_size")),
                                Filters.eq("uid", uid.asString()),
                                Filters.eq("key", "/total_size"),
                                Filters.eq("parent", DjfsResourcePath.getMongoId(uid, "/"))
                        ),
                        Updates.combine(
                                Updates.inc("data", delta),
                                Updates.set("version", InstantUtils.toVersion(Instant.now())),
                                Updates.setOnInsert("type", "file")),
                        new UpdateOptions().upsert(true)));
    }

    @Override
    public void incrementTrashUsed(DjfsUid uid, long delta) {
        MongoCollection<BsonDocument> collection = collection(uid);
        MongoUtil.retryTimeoutOnce(
                () -> collection.updateOne(
                        Filters.and(
                                Filters.eq("_id", DjfsResourcePath.getMongoId(uid, "/trash_size")),
                                Filters.eq("uid", uid.asString()),
                                Filters.eq("key", "/trash_size"),
                                Filters.eq("parent", DjfsResourcePath.getMongoId(uid, "/"))
                        ),
                        Updates.combine(
                                Updates.inc("data", delta),
                                Updates.set("version", InstantUtils.toVersion(Instant.now())),
                                Updates.setOnInsert("type", "file")),
                        new UpdateOptions().upsert(true)));
    }

    @Override
    public void setLimit(DjfsUid uid, long limit) {
        MongoCollection<BsonDocument> collection = collection(uid);
        MongoUtil.retryTimeoutOnce(
                () -> collection.updateOne(
                        Filters.and(
                                Filters.eq("_id", DjfsResourcePath.getMongoId(uid, "/limit")),
                                Filters.eq("uid", uid.asString()),
                                Filters.eq("key", "/limit"),
                                Filters.eq("parent", DjfsResourcePath.getMongoId(uid, "/"))
                        ),
                        Updates.combine(
                                Updates.set("data", limit),
                                Updates.set("version", InstantUtils.toVersion(Instant.now())),
                                Updates.setOnInsert("type", "file")),
                        new UpdateOptions().upsert(true)));
    }

    @Override
    public void setTotalUsed(DjfsUid uid, long used) {
        MongoCollection<BsonDocument> collection = collection(uid);
        MongoUtil.retryTimeoutOnce(
                () -> collection.updateOne(
                        Filters.and(
                                Filters.eq("_id", DjfsResourcePath.getMongoId(uid, "/total_size")),
                                Filters.eq("uid", uid.asString()),
                                Filters.eq("key", "/total_size"),
                                Filters.eq("parent", DjfsResourcePath.getMongoId(uid, "/"))
                        ),
                        Updates.combine(
                                Updates.set("data", used),
                                Updates.set("version", InstantUtils.toVersion(Instant.now())),
                                Updates.setOnInsert("type", "file")),
                        new UpdateOptions().upsert(true)));
    }

    @Override
    public void setTrashUsed(DjfsUid uid, long used) {
        MongoCollection<BsonDocument> collection = collection(uid);
        MongoUtil.retryTimeoutOnce(
                () -> collection.updateOne(
                        Filters.and(
                                Filters.eq("_id", DjfsResourcePath.getMongoId(uid, "/trash_size")),
                                Filters.eq("uid", uid.asString()),
                                Filters.eq("key", "/trash_size"),
                                Filters.eq("parent", DjfsResourcePath.getMongoId(uid, "/"))
                        ),
                        Updates.combine(
                                Updates.set("data", used),
                                Updates.set("version", InstantUtils.toVersion(Instant.now())),
                                Updates.setOnInsert("type", "file")),
                        new UpdateOptions().upsert(true)));
    }

    @Override
    public void insertOrUpdateDiskInfoValue(DiskInfo diskInfo) {
        throw new NotImplementedException();
    }

    @Override
    public void incrementResetOverdraftCounter(DjfsUid uid) {
        throw new NotImplementedException();
    }

    @Override
    public Option<Integer> getOverdraftCounter(DjfsUid uid) {
        throw new NotImplementedException();
    }
}
