package ru.yandex.chemodan.app.djfs.core.diskinfo;

import java.util.Map;
import java.util.Objects;

import org.joda.time.Instant;
import org.postgresql.util.PSQLException;
import org.postgresql.util.ServerErrorMessage;
import org.springframework.dao.DataIntegrityViolationException;
import org.springframework.jdbc.core.RowMapper;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.chemodan.app.djfs.core.db.EntityAlreadyExistsException;
import ru.yandex.chemodan.app.djfs.core.db.pg.PgShardedDao;
import ru.yandex.chemodan.app.djfs.core.db.pg.PgShardedDaoContext;
import ru.yandex.chemodan.app.djfs.core.db.pg.ResultSetUtils;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.DjfsResourcePath;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.DjfsResourceType;
import ru.yandex.chemodan.app.djfs.core.user.DjfsUid;
import ru.yandex.chemodan.app.djfs.core.util.InstantUtils;
import ru.yandex.misc.bender.Bender;
import ru.yandex.misc.bender.parse.BenderJsonParser;
import ru.yandex.misc.bender.serialize.BenderJsonSerializer;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.regex.Pattern2;

import static ru.yandex.chemodan.app.djfs.core.diskinfo.DiskInfo.OVERDRAFT_RESET_COUNT_FIELD;

/**
 * @author eoshch
 */
public class PgDiskInfoDao extends PgShardedDao implements DiskInfoDao {
    private static final Logger logger = LoggerFactory.getLogger(MongoDiskInfoDao.class);

    private static final BenderJsonSerializer<DiskInfoData.JsonData> jsonDiskInfoDataSerializer = Bender.jsonSerializer(DiskInfoData.JsonData.class);

    private static final BenderJsonParser<DiskInfoData.JsonData> jsonDiskInfoDataParser = Bender.jsonParser(DiskInfoData.JsonData.class);

    private static final Pattern2 LONG_PATTERN = Pattern2.compile("\\d+");

    private static final RowMapper<DiskInfo> M = (rs, rowNum) -> {
        // byte[] dataJson = rs.getBytes("data");
        String dataJsonString = rs.getString("data");
        // JsonValue parsedData = dataBender.getParser().parseJson(dataJson);

        long version = rs.getLong("version");
        Option<Long> versionO = rs.wasNull() ? Option.empty() : Option.of(version);

        return DiskInfo.builder()
                .id(ResultSetUtils.getUuid(rs, ("id")))
                .uid(DjfsUid.cons(rs.getLong("uid")))
                .parentId(ResultSetUtils.getUuidO(rs, "parent"))
                .data(Option.ofNullable(dataJsonString).map(PgDiskInfoDao::getDiskInfoDataFromDbString))
                .version(versionO)
                .path(rs.getString("path"))
                .type(DjfsResourceType.R.valueOf(rs.getString("type")))
                .build();
    };

    public PgDiskInfoDao(PgShardedDaoContext dependencies) {
        super(dependencies);
    }

    @Override
    public Option<DiskInfo> find(DjfsUid uid, String path) {
        String sql = collectStats(uid) + " SELECT * FROM disk.disk_info WHERE uid = :uid AND path = :path";
        Map<String, Object> parameters = Cf.map("uid", uid, "path", path);
        return jdbcTemplate(uid).queryForOption(sql, M, parameters);
    }

    @Override
    public ListF<DiskInfo> find(DjfsUid uid) {
        String sql = collectStats(uid) + " SELECT * FROM disk.disk_info WHERE uid = :uid";
        Map<String, Object> parameters = Cf.map("uid", uid);
        return jdbcTemplate(uid).query(sql, M, parameters);
    }

    @Override
    public void insert(DiskInfo diskInfo) {
        String sql = collectStats(diskInfo)
                + " INSERT INTO disk.disk_info (id, uid, parent, version, path, type, data) "
                + " VALUES (:id, :uid, :parent, :version, :path, :type::disk.resource_type, :data::json)";

        Map<String, Object> parameters = Cf.toMap(Tuple2List.fromPairs(
                "id", diskInfo.getId(),
                "uid", diskInfo.getUid(),
                "parent", diskInfo.getParentId().getOrNull(),
                "version", diskInfo.getVersion().getOrNull(),
                "path", diskInfo.getPath(),
                "type", diskInfo.getType(),
                "data", diskInfo.getData().map(PgDiskInfoDao::serializeDiskInfoDataToDbString).getOrNull()
        ));

        try {
            jdbcTemplate(diskInfo).update(sql, parameters);
        } catch (DataIntegrityViolationException e) {
            Throwable cause = e.getCause();
            if (cause instanceof PSQLException) {
                ServerErrorMessage error = ((PSQLException) cause).getServerErrorMessage();
                if (Objects.equals(error.getConstraint(), "pk_disk_info")) {
                    logger.warn("PgDiskInfoDao.insert(DiskInfo) handled exception for uid "
                            + diskInfo.getUid().asString() + " id " + diskInfo.getId() + " : ", e);
                    throw new EntityAlreadyExistsException("id=" + diskInfo.getId() + " uid="
                            + diskInfo.getUid().asString(), e);
                }
            }
            throw e;
        }
    }

    @Override
    public Option<Long> findLimit(DjfsUid uid) {
        String sql = collectStats(uid)
                + " SELECT data FROM disk.disk_info WHERE id = :id AND uid = :uid AND path = :path";
        Map<String, Object> parameters = Cf.map(
                "id", DjfsResourcePath.getPgId(uid, "/limit"),
                "uid", uid,
                "path", "/limit");
        return jdbcTemplate(uid).queryForOption(sql, (x, y) -> x.getLong("data"), parameters);
    }

    @Override
    public Option<Long> findTotalUsed(DjfsUid uid) {
        String sql = collectStats(uid)
                + " SELECT data FROM disk.disk_info WHERE id = :id AND uid = :uid AND path = :path";
        Map<String, Object> parameters = Cf.map(
                "id", DjfsResourcePath.getPgId(uid, "/total_size"),
                "uid", uid,
                "path", "/total_size");
        return jdbcTemplate(uid).queryForOption(sql, (x, y) -> x.getLong("data"), parameters);
    }

    @Override
    public Option<Long> findTrashUsed(DjfsUid uid) {
        String sql = collectStats(uid)
                + " SELECT data FROM disk.disk_info WHERE id = :id AND uid = :uid AND path = :path";
        Map<String, Object> parameters = Cf.map(
                "id", DjfsResourcePath.getPgId(uid, "/trash_size"),
                "uid", uid,
                "path", "/trash_size");
        return jdbcTemplate(uid).queryForOption(sql, (x, y) -> x.getLong("data"), parameters);
    }

    @Override
    public Option<Long> findFileCount(DjfsUid uid) {
        String sql = collectStats(uid)
                + " SELECT data FROM disk.disk_info WHERE id = :id AND uid = :uid AND path = :path";
        Map<String, Object> parameters = Cf.map(
                "id", DjfsResourcePath.getPgId(uid, "/files_count"),
                "uid", uid,
                "path", "/files_count");
        return jdbcTemplate(uid).queryForOption(sql, (x, y) -> x.getLong("data"), parameters);
    }

    @Override
    public Option<DownloadTraffic> findDownloadTraffic(DjfsUid uid) {
        String sql = collectStats(uid)
                + " SELECT data FROM disk.disk_info WHERE id = :id AND uid = :uid AND path = :path";
        Map<String, Object> parameters = Cf.map(
                "id", DjfsResourcePath.getPgId(uid, "/download_traffic"),
                "uid", uid,
                "path", "/download_traffic");
        return jdbcTemplate(uid).queryForOption(sql, (rs, rowNum) -> Bender.jsonParser(DownloadTraffic.class).parseJson((rs.getString("data"))), parameters);
    }

    @Override
    public void incrementTotalUsed(DjfsUid uid, long delta) {
        String sql = collectStats(uid) + " INSERT INTO disk.disk_info (id, uid, parent, version, path, type, data) "
                + " VALUES (:id, :uid, :parent, :version, :path, :type::disk.resource_type, :delta::text::jsonb) "
                + " ON CONFLICT ON CONSTRAINT pk_disk_info DO UPDATE"
                + " SET data = to_jsonb(disk.disk_info.data::text::bigint + :delta), version = :version";
        Map<String, Object> parameters = Cf.toMap(Tuple2List.fromPairs(
                "id", DjfsResourcePath.getPgId(uid, "/total_size"),
                "uid", uid,
                "parent", DjfsResourcePath.getPgId(uid, "/"),
                "path", "/total_size",
                "version", InstantUtils.toVersion(Instant.now()),
                "type", "file",
                "delta", delta));
        jdbcTemplate(uid).update(sql, parameters);
    }

    @Override
    public void incrementTrashUsed(DjfsUid uid, long delta) {
        String sql = collectStats(uid) + " INSERT INTO disk.disk_info (id, uid, parent, version, path, type, data) "
                + " VALUES (:id, :uid, :parent, :version, :path, :type::disk.resource_type, :delta::text::jsonb) "
                + " ON CONFLICT ON CONSTRAINT pk_disk_info DO UPDATE"
                + " SET data = to_jsonb(disk.disk_info.data::text::bigint + :delta), version = :version";
        Map<String, Object> parameters = Cf.toMap(Tuple2List.fromPairs(
                "id", DjfsResourcePath.getPgId(uid, "/trash_size"),
                "uid", uid,
                "parent", DjfsResourcePath.getPgId(uid, "/"),
                "path", "/trash_size",
                "version", InstantUtils.toVersion(Instant.now()),
                "type", "file",
                "delta", delta));
        jdbcTemplate(uid).update(sql, parameters);
    }

    @Override
    public void setLimit(DjfsUid uid, long limit) {
        String sql = collectStats(uid) + " INSERT INTO disk.disk_info (id, uid, parent, version, path, type, data) "
                + " VALUES (:id, :uid, :parent, :version, :path, :type::disk.resource_type, :data::json) "
                + " ON CONFLICT ON CONSTRAINT pk_disk_info DO UPDATE"
                + " SET data = :data::json, version = :version";
        Map<String, Object> parameters = Cf.toMap(Tuple2List.fromPairs(
                "id", DjfsResourcePath.getPgId(uid, "/limit"),
                "uid", uid,
                "parent", DjfsResourcePath.getPgId(uid, "/"),
                "path", "/limit",
                "version", InstantUtils.toVersion(Instant.now()),
                "type", "file",
                "data", Long.toString(limit)));
        jdbcTemplate(uid).update(sql, parameters);
    }

    @Override
    public void setTotalUsed(DjfsUid uid, long used) {
        String sql = collectStats(uid) + " INSERT INTO disk.disk_info (id, uid, parent, version, path, type, data) "
                + " VALUES (:id, :uid, :parent, :version, :path, :type::disk.resource_type, :data::json) "
                + " ON CONFLICT ON CONSTRAINT pk_disk_info DO UPDATE"
                + " SET data = :data::json, version = :version";
        Map<String, Object> parameters = Cf.toMap(Tuple2List.fromPairs(
                "id", DjfsResourcePath.getPgId(uid, "/total_size"),
                "uid", uid,
                "parent", DjfsResourcePath.getPgId(uid, "/"),
                "path", "/total_size",
                "version", InstantUtils.toVersion(Instant.now()),
                "type", "file",
                "data", Long.toString(used)));
        jdbcTemplate(uid).update(sql, parameters);
    }

    @Override
    public void setTrashUsed(DjfsUid uid, long used) {
        String sql = collectStats(uid) + " INSERT INTO disk.disk_info (id, uid, parent, version, path, type, data) "
                + " VALUES (:id, :uid, :parent, :version, :path, :type::disk.resource_type, :data::json) "
                + " ON CONFLICT ON CONSTRAINT pk_disk_info DO UPDATE"
                + " SET data = :data::json, version = :version";
        Map<String, Object> parameters = Cf.toMap(Tuple2List.fromPairs(
                "id", DjfsResourcePath.getPgId(uid, "/trash_size"),
                "uid", uid,
                "parent", DjfsResourcePath.getPgId(uid, "/"),
                "path", "/trash_size",
                "version", InstantUtils.toVersion(Instant.now()),
                "type", "file",
                "data", Long.toString(used)));
        jdbcTemplate(uid).update(sql, parameters);
    }

    @Override
    public void insertOrUpdateDiskInfoValue(DiskInfo diskInfo) {
        String sql = collectStats(diskInfo)
                + " INSERT INTO disk.disk_info AS disk_info (id, uid, parent, version, path, type, data) "
                + " VALUES (:id, :uid, :parent, :version, :path, :type::disk.resource_type, :data::json)"
                + " ON CONFLICT (id) DO"
                + " UPDATE SET data = :data::json, version = :version"
                + " WHERE disk_info.version < :version";

        Map<String, Object> parameters = Cf.toMap(Tuple2List.fromPairs(
                "id", diskInfo.getId(),
                "uid", diskInfo.getUid(),
                "parent", diskInfo.getParentId().getOrNull(),
                "version", diskInfo.getVersion().getOrNull(),
                "path", diskInfo.getPath(),
                "type", diskInfo.getType(),
                "data", diskInfo.getData().map(PgDiskInfoDao::serializeDiskInfoDataToDbString).getOrNull()
        ));
        jdbcTemplate(diskInfo).update(sql, parameters);
    }

    @Override
    public void incrementResetOverdraftCounter(DjfsUid uid) {
        String sql = collectStats(uid) +
                " UPDATE disk.disk_info as disk_info" +
                " SET data = jsonb_set(disk_info.data, '{" + OVERDRAFT_RESET_COUNT_FIELD + "}'," +
                " to_jsonb((disk_info.data ->> '" + OVERDRAFT_RESET_COUNT_FIELD + "')::int + 1))" +
                " WHERE uid = :uid" +
                " AND path = :path" +
                " AND jsonb_typeof(disk_info.data -> '" + OVERDRAFT_RESET_COUNT_FIELD + "') is not null";

        jdbcTemplate(uid).update(sql, Cf.map("uid", uid, "path", "/overdraft"));
    }

    @Override
    public Option<Integer> getOverdraftCounter(DjfsUid uid) {
        String sql = collectStats(uid)
                + " SELECT disk_info.data ->> '" + OVERDRAFT_RESET_COUNT_FIELD + "'" +
                " FROM disk.disk_info as disk_info" +
                " WHERE uid = :uid" +
                " AND path = :path";

        return jdbcTemplate(uid).queryForOption(sql, (x, y) -> x.getInt(1),
                Cf.map("uid", uid, "path", "/overdraft"));
    }

    public static String serializeDiskInfoDataToDbString(DiskInfoData diskInfoData) {
        if (diskInfoData.isLongData()) {
            return String.valueOf(diskInfoData.getLongValue());
        }
        if (diskInfoData.isRawData()) {
            return diskInfoData.getRawData();
        }
        return new String(jsonDiskInfoDataSerializer.serializeJson(diskInfoData.getJsonData()));
    }

    public static DiskInfoData getDiskInfoDataFromDbString(String dbString) {
        if (LONG_PATTERN.matches(dbString)) {
            return DiskInfoData.longData(Cf.Long.parse(dbString));
        }
        try {
            return DiskInfoData.jsonData(jsonDiskInfoDataParser.parseJson(dbString));
        } catch (Exception e) {
            return DiskInfoData.rawData(dbString);
        }
    }
}
