package ru.yandex.chemodan.app.djfs.core.filesystem;

import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.djfs.core.user.DjfsUid;
import ru.yandex.chemodan.app.djfs.core.user.UserData;

/**
 * A principal in computer security is an entity that can be authenticated by a computer system.
 * Contains information about an entity that invoked the request.
 * An entity can be: user, service, etc.
 *
 * Primarily used in access and permission checks.
 *
 * @author eoshch
 * @see <a href=https://en.wikipedia.org/wiki/Principal_(computer_security)>https://en.wikipedia.org/wiki/Principal_(computer_security)</a>
 */
public class DjfsPrincipal {
    public static DjfsPrincipal SYSTEM = new DjfsPrincipal(Type.SYSTEM);
    public static DjfsPrincipal SUPPORT = new DjfsPrincipal(Type.SUPPORT);
    public static DjfsPrincipal ANONYMOUS = new DjfsPrincipal(Type.ANONYMOUS);
    public static DjfsPrincipal INTERNAL_SERVICE = new DjfsPrincipal(Type.INTERNAL_SERVICE);

    private final Option<DjfsUid> uidO;
    private final Option<UserData> userO;
    private final Type type;

    private DjfsPrincipal(DjfsUid uid) {
        this.uidO = Option.of(uid);
        type = Type.UID;
        userO = Option.empty();
    }

    private DjfsPrincipal(Type type) {
        this.type = type;
        this.uidO = Option.empty();
        this.userO = Option.empty();
    }

    private DjfsPrincipal(UserData user) {
        userO = Option.of(user);
        uidO = Option.of(user.getUid());
        type = Type.UID;
    }

    public boolean isUid() {
        return type == Type.UID;
    }

    public boolean isSupport() {
        return type == Type.SUPPORT;
    }

    public boolean isSystem() {
        return type == Type.SYSTEM;
    }

    public boolean isAnonymous() {
        return type == Type.ANONYMOUS;
    }

    public boolean isInternalService() { return type == Type.INTERNAL_SERVICE; }

    public Type getType() {
        return type;
    }

    public DjfsUid getUid() {
        return uidO.get();
    }

    public Option<DjfsUid> getUidO() {
        return uidO;
    }

    public Option<UserData> getUserO() { return userO; }

    @Override
    public String toString() {
        return uidO.map(DjfsUid::toString).getOrElse(type.toString());
    }

    public static DjfsPrincipal cons(DjfsUid uid) {
        return new DjfsPrincipal(uid);
    }

    public static DjfsPrincipal cons(UserData user) {
        return new DjfsPrincipal(user);
    }

    private enum Type {
        ANONYMOUS,
        UID,
        SUPPORT,
        SYSTEM,
        INTERNAL_SERVICE,
    }
}
