package ru.yandex.chemodan.app.djfs.core.filesystem;

import java.util.UUID;

import org.joda.time.Instant;

import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.bolts.internal.NotImplementedException;
import ru.yandex.chemodan.app.djfs.core.FilenamePreviewStidMimetypeVersionFileId;
import ru.yandex.chemodan.app.djfs.core.db.EntityAlreadyExistsException;
import ru.yandex.chemodan.app.djfs.core.db.Sharded;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.DjfsFileId;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.DjfsResource;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.DjfsResourceArea;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.DjfsResourceId;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.DjfsResourcePath;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.FileDjfsResource;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.FolderDjfsResource;
import ru.yandex.chemodan.app.djfs.core.user.DjfsUid;
import ru.yandex.misc.geo.Coordinates;
import ru.yandex.misc.image.Dimension;

/**
 * @author eoshch
 */
public interface DjfsResourceDao {
    @Sharded
    Option<DjfsResource> find(DjfsUid uid, DjfsResourceArea area, UUID id);

    @Sharded
    Option<DjfsResource> find(DjfsUid uid, DjfsResourceArea area, UUID id, boolean forUpdate);

    @Sharded
    Option<DjfsResource> find2(DjfsUid uid, DjfsResourceArea area, UUID id);

    @Sharded
    Option<DjfsResource> find2(DjfsUid uid, DjfsResourceArea area, UUID id, boolean forUpdate);

    @Sharded
    Option<FileDjfsResource> find2File(DjfsUid uid, DjfsResourceArea area, UUID id);

    @Sharded
    Option<FolderDjfsResource> find2Folder(DjfsResourcePath path);

    @Sharded
    Option<DjfsResource> find(DjfsResourcePath path);

    @Sharded
    Option<DjfsResource> find2(DjfsResourcePath path);

    /**
     * ResourceId is not unique yet. Moving creates a destination copy first then deletes source entry.
     */
    @Sharded
    ListF<DjfsResource> find(DjfsResourceId resourceId);

    /**
     * ResourceId is not unique yet. Moving creates a destination copy first then deletes source entry.
     */
    @Sharded
    ListF<DjfsResource> find(DjfsResourceId resourceId, DjfsResourceArea area);

    @Sharded
    ListF<Tuple2<DjfsFileId, Option<Double>>> getAesthetics(DjfsUid uid, ListF<DjfsFileId> fileIds);

    @Sharded
    ListF<DjfsFileId> getFileIds(DjfsUid uid, int limit, Option<DjfsFileId> startingFileId);

    @Sharded
    @Deprecated  // temporary method to save aesthetics for file with coordinates and without aesthetics
    ListF<DjfsFileId> getFileIdForFilesWithCoordinatesWithoutAesthetics(DjfsUid uid, int limit);

    @Sharded
    Option<Tuple2<DjfsResource, ListF<UUID>>> findWithParentIds(DjfsResourcePath path);

    @Sharded
    ListF<Tuple2<DjfsResource, ListF<UUID>>> findWithParentIds(
            DjfsUid uid, ListF<String> fileIds, ListF<DjfsResourceArea> areasToBeSearched);

    @Sharded
    ListF<DjfsResource> findByPaths(DjfsUid uid, ListF<DjfsResourcePath> paths);

    /**
     * Immediate children - top level children, direct children. Returns folders sorted by fileId.
     *
     * @param startingFileId not inclusive
     * @param startingId     not inclusive
     */
    @Sharded
    default ListF<FolderDjfsResource> find2ImmediateChildFoldersOrderByFileIdThenId(FolderDjfsResource parent,
            Option<String> startingFileId, Option<UUID> startingId, int limit, boolean forUpdate)
    {
        return find2ImmediateChildFoldersOrderByFileIdThenId(parent.getUid(), parent.getPath().getArea(),
                parent.getId(), startingFileId, startingId, limit, forUpdate);
    }

    /**
     * Immediate children - top level children, direct children. Returns folders sorted by fileId.
     *
     * @param startingFileId not inclusive
     * @param startingId     not inclusive
     */
    @Sharded
    ListF<FolderDjfsResource> find2ImmediateChildFoldersOrderByFileIdThenId(DjfsUid uid, DjfsResourceArea area,
            UUID parentId, Option<String> startingFileId, Option<UUID> startingId, int limit, boolean forUpdate);

    /**
     * Immediate children - top level children, direct children.
     */
    @Sharded
    ListF<FolderDjfsResource> find2ImmediateChildFoldersWithoutFileId(DjfsUid uid, DjfsResourceArea area,
            UUID parentId);

    /**
     * Immediate children - top level children, direct children. Returns folders sorted by fileId.
     *
     * @param startingFileId not inclusive
     * @param startingId     not inclusive
     */
    @Sharded
    default ListF<FileDjfsResource> find2ImmediateChildFilesOrderByFileIdThenId(FolderDjfsResource parent,
            Option<String> startingFileId, Option<UUID> startingId, int limit, boolean forUpdate)
    {
        return find2ImmediateChildFilesOrderByFileIdThenId(parent.getUid(), parent.getPath().getArea(), parent.getId(),
                startingFileId, startingId, limit, forUpdate);
    }

    /**
     * Immediate children - top level children, direct children. Returns folders sorted by fileId.
     *
     * @param startingFileId not inclusive
     * @param startingId     not inclusive
     */
    @Sharded
    ListF<FileDjfsResource> find2ImmediateChildFilesOrderByFileIdThenId(DjfsUid uid, DjfsResourceArea area,
            UUID parentId, Option<String> startingFileId, Option<UUID> startingId, int limit, boolean forUpdate);

    @Sharded
    Option<FileDjfsResource> find2AdditionalFileFor(FileDjfsResource file);

    @Sharded
    ListF<FilenamePreviewStidMimetypeVersionFileId> find2FilenamePreviewStidMimetypeVersionFid(
            DjfsUid uid, ListF<DjfsFileId> fileIds);

    @Sharded
    default Option<FolderDjfsResource> find2Parent(DjfsResource resource) {
        return find2Folder(resource.getPath().getParent());
    }

    @Sharded
    ListF<DjfsResource> find2ImmediateChildren(DjfsResourcePath path, int limit);

    @Sharded
    long countImmediateChildren(DjfsResourcePath path);

    @Sharded
    ListF<DjfsResource> find2ImmediateChildren(DjfsResourcePath path, int offset, int limit, boolean ordered);

    @Sharded
    ListF<DjfsResource> find2ImmediateChildren(DjfsResource resource, int offset, int limit, boolean ordered);

    /**
     * Use in tests only!
     * Not ready for production!
     *
     * @deprecated remove me
     */
    @Sharded
    ListF<DjfsResource> findAll(DjfsUid uid, DjfsResourceArea partition);

    @Sharded
    void insert(FolderDjfsResource folder);

    @Sharded
    void insert2(FolderDjfsResource folder);

    @Sharded
    default void insert(DjfsUid uid, FolderDjfsResource... folders) {
        // todo: check that all folders belong to the same uid
        for (FolderDjfsResource folder : folders) {
            insert(folder);
        }
    }

    @Sharded
    void insert(FileDjfsResource file);

    @Sharded
    void insert2(FileDjfsResource file);

    @Sharded
    void insert2AndLinkTo(FileDjfsResource file, FileDjfsResource link);

    @Sharded
    default void insert(DjfsUid uid, FileDjfsResource... files) {
        // todo: check that all files belong to the same uid
        for (FileDjfsResource file : files) {
            insert(file);
        }
    }

    @Sharded
    default void insert(DjfsUid uid, ListF<DjfsResource> resources) {
        // todo: check that all resources belong to the same uid
        for (DjfsResource resource : resources) {
            insert(resource);
        }
    }

    @Sharded
    default void insert(DjfsResource resource) {
        if (resource instanceof FileDjfsResource) {
            insert((FileDjfsResource) resource);
        } else if (resource instanceof FolderDjfsResource) {
            insert((FolderDjfsResource) resource);
        } else {
            throw new NotImplementedException();
        }
    }

    @Sharded
    default void insert2(DjfsResource resource) {
        if (resource instanceof FileDjfsResource) {
            insert2((FileDjfsResource) resource);
        } else if (resource instanceof FolderDjfsResource) {
            insert2((FolderDjfsResource) resource);
        } else {
            throw new NotImplementedException();
        }
    }

    @Sharded
    void changeParent(DjfsUid uid, UUID id, UUID newParentId);

    @Sharded
    void changeParentAndName(DjfsUid uid, UUID id, UUID newParentId, String name);

    @Sharded
    void setVersion(DjfsUid uid, UUID id, long version);

    @Sharded
    void setModificationTime(DjfsUid uid, UUID id, Instant time);

    @Sharded
    void setYarovayaMark(DjfsUid uid, UUID id);

    @Sharded
    void setTrashAppendTimeAndOriginalPath(DjfsUid uid, UUID id, String path, Instant trashAppendTime);

    @Sharded
    void setTrashAppendTime(DjfsUid uid, ListF<UUID> ids, Instant trashAppendTime);

    @Sharded
    void setArea(DjfsUid uid, ListF<UUID> ids, DjfsResourceArea area);

    @Sharded
    void setHiddenAppendTime(DjfsUid uid, UUID id, Instant hiddenAppendTime);

    @Sharded
    void setPublishedAndRemovePublic(DjfsUid uid, UUID id);

    @Sharded
    void addFileId(DjfsResourcePath path, DjfsFileId fileId);

    @Sharded
    void removeFileId(DjfsResourcePath path);

    @Sharded
    void setAesthetics(DjfsUid uid, UUID id, double aesthetics);

    @Sharded
    void setCoordinates(DjfsUid uid, UUID id, Coordinates coordinates);

    @Sharded
    void setCoordinates(DjfsUid uid, Tuple2List<DjfsFileId, Coordinates> fileIdWithCoordinates, boolean skipExisting);

    @Sharded
    void setAesthetics(DjfsUid uid, Tuple2List<DjfsFileId, Double> fileIdWithAesthetics, boolean skipExisting);

    @Sharded
    void setDimensionsWithAngle(DjfsUid uid, UUID id, Dimension dimension, int angle);

    @Sharded
    boolean hasAnyParentWithYarovayaMark(DjfsResourcePath path);

    /**
     * Use in tests only!
     * Not ready for production!
     */
    @Sharded
    void delete(DjfsResourcePath path);

    @Sharded
    void checkAlreadyExistsException(Throwable e, DjfsResource resource) throws EntityAlreadyExistsException;

    @Sharded
    ListF<FileDjfsResource> getUserPhotosWithGeoCoordinates(DjfsUid uid);

    @Sharded
    long countAllFiles(DjfsUid uid);
}
