package ru.yandex.chemodan.app.djfs.core.filesystem;

import java.util.ArrayDeque;
import java.util.Comparator;
import java.util.List;
import java.util.UUID;

import com.mongodb.ReadPreference;
import lombok.RequiredArgsConstructor;
import org.joda.time.Instant;
import org.springframework.transaction.TransactionSystemException;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.bolts.function.Function;
import ru.yandex.bolts.function.Function1V;
import ru.yandex.chemodan.app.djfs.core.EventManager;
import ru.yandex.chemodan.app.djfs.core.changelog.Changelog;
import ru.yandex.chemodan.app.djfs.core.changelog.ChangelogManager;
import ru.yandex.chemodan.app.djfs.core.db.EntityAlreadyExistsException;
import ru.yandex.chemodan.app.djfs.core.db.pg.TransactionUtils;
import ru.yandex.chemodan.app.djfs.core.filesystem.event.FolderCreatedEvent;
import ru.yandex.chemodan.app.djfs.core.filesystem.event.PrivateFolderCreatedEvent;
import ru.yandex.chemodan.app.djfs.core.filesystem.event.SharedFolderCreatedEvent;
import ru.yandex.chemodan.app.djfs.core.filesystem.exception.DjfsNotImplementedException;
import ru.yandex.chemodan.app.djfs.core.filesystem.exception.FilesystemException;
import ru.yandex.chemodan.app.djfs.core.filesystem.exception.FolderTooDeepException;
import ru.yandex.chemodan.app.djfs.core.filesystem.exception.ForbiddenResourcePathException;
import ru.yandex.chemodan.app.djfs.core.filesystem.exception.NoFreeSpaceException;
import ru.yandex.chemodan.app.djfs.core.filesystem.exception.NoParentFolderException;
import ru.yandex.chemodan.app.djfs.core.filesystem.exception.OperationInAreaNotPermittedException;
import ru.yandex.chemodan.app.djfs.core.filesystem.exception.ParentIsFileException;
import ru.yandex.chemodan.app.djfs.core.filesystem.exception.ResourceExistsException;
import ru.yandex.chemodan.app.djfs.core.filesystem.exception.ResourceNotFoundException;
import ru.yandex.chemodan.app.djfs.core.filesystem.exception.SameSourceAndDestinationException;
import ru.yandex.chemodan.app.djfs.core.filesystem.exception.SourceIsParentForDestinationException;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.AntiVirusScanStatus;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.DjfsFileId;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.DjfsResource;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.DjfsResourceArea;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.DjfsResourceId;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.DjfsResourcePath;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.FileDjfsResource;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.FilesystemOperation;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.FolderDjfsResource;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.MediaType;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.exception.InvalidDjfsResourcePathException;
import ru.yandex.chemodan.app.djfs.core.filesystem.operation.move.MoveCallbacks;
import ru.yandex.chemodan.app.djfs.core.filesystem.operation.postprocess.QuickMovePostProcessTaskSubmitter;
import ru.yandex.chemodan.app.djfs.core.index.SearchPushGenerator;
import ru.yandex.chemodan.app.djfs.core.lastfiles.LastFilesCacheUpdater;
import ru.yandex.chemodan.app.djfs.core.lock.LockManager;
import ru.yandex.chemodan.app.djfs.core.lock.ResourceLockedException;
import ru.yandex.chemodan.app.djfs.core.notification.XivaPushGenerator;
import ru.yandex.chemodan.app.djfs.core.publication.LinkDataDao;
import ru.yandex.chemodan.app.djfs.core.share.Group;
import ru.yandex.chemodan.app.djfs.core.share.GroupDao;
import ru.yandex.chemodan.app.djfs.core.share.GroupLink;
import ru.yandex.chemodan.app.djfs.core.share.GroupLinkDao;
import ru.yandex.chemodan.app.djfs.core.share.ShareInfo;
import ru.yandex.chemodan.app.djfs.core.share.ShareInfoManager;
import ru.yandex.chemodan.app.djfs.core.user.DjfsUid;
import ru.yandex.chemodan.app.djfs.core.user.UserDao;
import ru.yandex.chemodan.app.djfs.core.user.UserData;
import ru.yandex.chemodan.app.djfs.core.user.UserNotInitializedException;
import ru.yandex.chemodan.app.djfs.core.util.InstantUtils;
import ru.yandex.chemodan.app.djfs.core.util.UuidUtils;
import ru.yandex.chemodan.app.djfs.core.util.YcridUtils;
import ru.yandex.chemodan.util.TimeUtils;
import ru.yandex.commune.dynproperties.DynamicProperty;
import ru.yandex.misc.geo.Coordinates;
import ru.yandex.misc.image.Dimension;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

/**
 * @author eoshch
 */
@RequiredArgsConstructor
public class Filesystem {
    private static final Logger logger = LoggerFactory.getLogger(Filesystem.class);
    private static final int AUTOSUFFIX_TRY_LIMIT = 100;
    private static final int FOLDER_MOVE_RESOURCES_THRESHOLD = 600;

    private final DynamicProperty<Integer> folderDepthMax = new DynamicProperty<>(
        "disk-djfs-folder-depth-max", 256
    );

    public Integer getFolderDepthMax() {
        return folderDepthMax.get();
    }

    private final DjfsResourceDao djfsResourceDao;
    private final UserDao userDao;
    private final LockManager lockManager;
    private final GroupDao groupDao;
    private final GroupLinkDao groupLinkDao;
    private final ShareInfoManager shareInfoManager;
    private final FilesystemAccessController filesystemPermissionChecker;
    private final EventManager eventManager;
    private final QuotaManager quotaManager;
    private final ChangelogManager changelogManager;
    private final SearchPushGenerator searchPushGenerator;
    private final XivaPushGenerator xivaPushGenerator;
    private final LastFilesCacheUpdater lastFilesCacheUpdater;
    private final TransactionUtils transactionUtils;
    private final LinkDataDao linkDataDao;
    private final SupportBlockedHidsDao blockedHidsDao;
    private final PgTrashCleanQueueDao pgTrashCleanQueueDao;
    private final QuickMovePostProcessTaskSubmitter quickMovePostProcessTaskSubmitter;

    public FolderDjfsResource createFolder(DjfsPrincipal principal, DjfsResourcePath path) {
        return createFolder(principal, path, x -> x);
    }

    public FolderDjfsResource createFolder(DjfsPrincipal principal, DjfsResourcePath path,
            Function<FolderDjfsResource.Builder, FolderDjfsResource.Builder> initialize)
    {
        path.checkNewPath(InvalidNewFolderNameException::new);
        path.checkDepth(getFolderDepthMax(), FolderTooDeepException::new);

        // todo: optimization: use PG constraints instead of additional queries
        Option<ShareInfo> shareInfo = shareInfoManager.get(path);

        // check permissions before everything else lest we leak information
        filesystemPermissionChecker.checkCreateFolder(principal, path, shareInfo);

        lockManager.checkUserAndPath(path);
        // todo: падать если participantPathToOwnerPath вернул empty?
        DjfsResourcePath newPath = shareInfo.filterMap(x -> x.participantPathToOwnerPath(path)).getOrElse(path);
        if (!newPath.getArea().isOperationPermitted(principal, newPath, FilesystemOperation.CREATE_FOLDER)) {
            throw new OperationInAreaNotPermittedException(principal, FilesystemOperation.CREATE_FOLDER, newPath);
        }

        if (!newPath.equals(path)) {
            lockManager.checkUserAndPath(newPath);
        }

        DjfsResourcePath parentPath = newPath.getParent();
        Option<DjfsResource> parentResourceO = djfsResourceDao.find(parentPath);
        DjfsResource parentResource;
        if (!parentResourceO.isPresent() && parentPath.isAreaRoot()) {
            parentResource = initializeArea(parentPath.getUid(), parentPath.getArea()).get2();
        } else if (!parentResourceO.isPresent()) {
            throw new NoParentFolderException(path);
        } else if (!(parentResourceO.get() instanceof FolderDjfsResource)) {
            throw new ParentIsFileException(path);
        } else {
            parentResource = parentResourceO.get();
        }

        Instant now = Instant.now();
        FolderDjfsResource folder = FolderDjfsResource.cons(newPath, parentResource.getId(), now, x -> {
            x.id(UUID.randomUUID());  // TODO: remove random uuid for id here (generate it in FileDjfsResource.cons and FolderDjfsResource.cons)
            initialize.apply(x.modifyUid(Option.of(path.getUid())));
        });

        try {
            djfsResourceDao.insert(folder);
        } catch (EntityAlreadyExistsException e) {
            throw new ResourceExistsException(folder.getPath(), e);
        }

        FolderCreatedEvent event;
        // todo: make events immutable?
        if (shareInfo.isPresent()) {
            SharedFolderCreatedEvent sharedEvent = new SharedFolderCreatedEvent();
            sharedEvent.shareInfo = shareInfo.get();
            sharedEvent.actor = path.getUid();
            sharedEvent.eventTime = now;
            sharedEvent.folder = folder;
            for (DjfsUid uid : shareInfo.get().allUids()) {
                sharedEvent.oldVersions.put(uid, userDao.find(uid).get().getVersion());
            }
            eventManager.send(sharedEvent);
            for (DjfsUid uid : shareInfo.get().allUids()) {
                // todo: make increment return old version
//                sharedEvent.oldVersions.put(uid, userDao.find(uid).get().version);
                if (newPath.getArea().isUserIndexVersionUpdated()) {
                    userDao.incrementVersionTo(uid, folder.getVersion().get());
                }
            }
//            event = sharedEvent;
        } else {
            PrivateFolderCreatedEvent privateEvent = new PrivateFolderCreatedEvent();
            // todo: make increment return old version
            privateEvent.oldVersion = userDao.find(path.getUid()).get().getVersion();
            privateEvent.actor = path.getUid();
            privateEvent.eventTime = now;
            privateEvent.folder = folder;
            eventManager.send(privateEvent);
            if (newPath.getArea().isUserIndexVersionUpdated()) {
                userDao.incrementVersionTo(folder.getUid(), folder.getVersion().get());
            }
//            event = privateEvent;
        }

//        event.actor = path.getUid();
//        event.eventTime = now;
//        event.folder = folder;

//        eventManager.send(event);
        return folder;
    }

    public Option<DjfsResourcePath> getAvailableAutosuffixedPath(DjfsResourcePath path) {
        for (int i = 1; i < AUTOSUFFIX_TRY_LIMIT; i++) {
            DjfsResourcePath nextPath = path.suffix(" (" + i + ")");
            Option<DjfsResource> resource = djfsResourceDao.find(nextPath);
            if (!resource.isPresent()) {
                return Option.of(nextPath);
            }
        }
        return Option.empty();
    }

    /**
     * Use in tests only!
     * Not ready for production!
     * <p>
     * TODO: move to test project
     */
    public FileDjfsResource createFile(DjfsPrincipal principal, DjfsResourcePath path) {
        return createFile(principal, path, x -> x);
    }

    /**
     * Use in tests only!
     * Not ready for production!
     * <p>
     * TODO: move to test project
     */
    public FileDjfsResource createFile(DjfsPrincipal principal, DjfsResourcePath path,
            Function<FileDjfsResource.Builder, FileDjfsResource.Builder> initialize)
    {
        Option<ShareInfo> shareInfoO = shareInfoManager.get(path);

        // check permissions before everything else lest we leak information
        filesystemPermissionChecker.checkCreateFile(principal, path, shareInfoO);

        lockManager.checkUserAndPath(path);
        // todo: падать если participantPathToOwnerPath вернул empty?
        DjfsResourcePath newPath = shareInfoO.filterMap(x -> x.participantPathToOwnerPath(path)).getOrElse(path);
        if (!newPath.equals(path)) {
            lockManager.checkUserAndPath(newPath);
        }

        DjfsResourcePath parentPath = newPath.getParent();
        Option<DjfsResource> parentResourceO = djfsResourceDao.find(parentPath);
        DjfsResource parentResource;
        if (!parentResourceO.isPresent() && parentPath.isAreaRoot()) {
            parentResource = initializeArea(parentPath.getUid(), parentPath.getArea()).get2();
        } else if (!parentResourceO.isPresent()) {
            throw new FilesystemException("no parent folder for " + path);
        } else if (!(parentResourceO.get() instanceof FolderDjfsResource)) {
            throw new FilesystemException("parent is not folder for " + path);
        } else {
            parentResource = parentResourceO.get();
        }

        Instant now = Instant.now();
        // todo: use FileDjfsResource.cons()
        FileDjfsResource file = initialize.apply(FileDjfsResource.builder()
                .id(newPath.getPgId())
                .parentId(Option.of(parentResource.getId()))
                .path(newPath)
                .isVisible(true)
                .uploadTime(Option.of(now))
                .creationTime(now)
                .modificationTime(now)
                .version(Option.of(now.getMillis() * 1000))
                .fileId(DjfsFileId.random())

                // todo: replace random values
                .hid(UuidUtils.randomToHexString())
                .md5(UuidUtils.randomToHexString())
                .sha256(UuidUtils.randomToHexString() + UuidUtils.randomToHexString())
                .antiVirusScanStatus(Option.of(AntiVirusScanStatus.CLEAN))
                .fileStid(UuidUtils.randomToHexString())
                .digestStid(UuidUtils.randomToHexString())
                .mimetype(Option.of("image/jpeg"))
                .mediaType(Option.of(MediaType.IMAGE))).build();

        try {
            djfsResourceDao.insert2(file);
        } catch (EntityAlreadyExistsException e) {
            throw new ResourceExistsException(file.getPath(), e);
        }

        return file;
    }

    public void ensureAllParentFoldersExist(DjfsPrincipal principal, DjfsResourcePath path) {
        for (DjfsResourcePath parent : path.getAllParents()) {
            if (parent.isRoot() || parent.isAreaRoot()) {
                continue;
            }
            try {
                createFolder(principal, parent);
            } catch (ResourceExistsException e) {
                // we want to be sure folder exists
            }
        }
    }

    /**
     * FileId can only be absent in MongoDB. There is a not null constraint in PostgreSQL.
     */
    public void ensureFileIdIsSet(DjfsPrincipal principal, DjfsResourcePath path) {
        filesystemPermissionChecker.checkWriteResource(principal, path);
        Option<ShareInfo> shareInfoO = shareInfoManager.get(path);
        DjfsResourcePath correctPath = shareInfoO.filterMap(x -> x.participantPathToOwnerPath(path)).getOrElse(path);
        djfsResourceDao.addFileId(correctPath, DjfsFileId.random());
    }

    public boolean isMoveImplemented(DjfsResourcePath sourcePath, DjfsResourcePath destinationPath, boolean force) {
        if (!sourcePath.getUid().equals(destinationPath.getUid())) {
            return false;
        }

        UserData user = userDao.findExistingAndNotBlocked(sourcePath.getUid());
        if (!user.isPg() || !user.isQuickMoveUser()) {
            return false;
        }

        Option<DjfsResource> sourceResourceO = djfsResourceDao.find(sourcePath);
        if (sourceResourceO.isPresent() && sourceResourceO.get() instanceof FileDjfsResource) {
            return false;  // TODO: add support for single not shared file
        }

        Option<DjfsResource> destinationResourceO = djfsResourceDao.find(destinationPath);
        if (destinationResourceO.isPresent() && force) {
            return false;
        }

        Option<ShareInfo> sourceShareInfoO = shareInfoManager.get(sourcePath);
        Option<ShareInfo> destinationShareInfoO = shareInfoManager.get(destinationPath);
        if (sourceShareInfoO.isPresent() || destinationShareInfoO.isPresent()) {
            return false;
        }

        ListF<Group> groups = groupDao.findAll(user.getUid());
        for (Group group : groups) {
            DjfsResourcePath groupPath = group.getPath();
            if (groupPath.equals(sourcePath) || groupPath.equals(destinationPath) ||
                    sourcePath.isParentFor(groupPath) || destinationPath.isParentFor(groupPath))
            {
                return false;
            }
        }

        ListF<GroupLink> groupLinks = groupLinkDao.findAll(user.getUid());
        for (GroupLink link : groupLinks) {
            DjfsResourcePath linkPath = link.getPath();
            if (linkPath.equals(sourcePath) || linkPath.equals(destinationPath) ||
                    sourcePath.isParentFor(linkPath) || destinationPath.isParentFor(linkPath))
            {
                return false;
            }
        }

        return true;
    }

    private void checkValidDestinationPath(DjfsResourcePath sourcePath, DjfsResourcePath destinationPath) {
        if (sourcePath.equals(destinationPath) || sourcePath.equals(destinationPath.getParent())) {
            throw new SameSourceAndDestinationException(sourcePath);
        }
        if (sourcePath.isParentFor(destinationPath)) {
            throw new SourceIsParentForDestinationException(sourcePath, destinationPath);
        }
    }

    private boolean isYarovayaMarkNeeded(DjfsResource resource) {
        DjfsUid uid = resource.getUid();
        DjfsResourcePath path = resource.getPath();

        // TODO: add check for shared folders here

        boolean hasPublicParents = !linkDataDao.find(uid, path.getAllParents()).isEmpty();
        return resource.hasAnyPublicField() || hasPublicParents || djfsResourceDao.hasAnyParentWithYarovayaMark(path);
    }

    private ListF<DjfsResource> getSubtree(DjfsResourcePath path, int limit) {
        ListF<DjfsResource> resources = Cf.arrayList();
        ArrayDeque<DjfsResourcePath> unprocessedFolderPaths = new ArrayDeque<>();
        unprocessedFolderPaths.add(path);

        while (!unprocessedFolderPaths.isEmpty() && limit > 0) {
            DjfsResourcePath folderPath = unprocessedFolderPaths.pop();

            ListF<DjfsResource> children = djfsResourceDao.find2ImmediateChildren(folderPath, limit);
            resources.addAll(children);

            unprocessedFolderPaths.addAll(children.filterByType(FolderDjfsResource.class).map(DjfsResource::getPath));

            limit -= children.length();
        }

        return resources;
    }

    private ListF<Changelog> prepareChangelogsForMoveSmallFolder(DjfsResource source,
            ListF<DjfsResource> destinationChildren, DjfsResource destination, Instant version)
    {
        ListF<Option<Changelog>> result = Cf.arrayList();

        result.add(changelogManager.makeChangelog(version, Changelog.OperationType.NEW, destination, Option.empty()));
        for (DjfsResource child : destinationChildren) {
            result.add(changelogManager.makeChangelog(version, Changelog.OperationType.NEW, child, Option.empty()));
        }
        result.add(changelogManager.makeChangelog(version, Changelog.OperationType.DELETED, source, Option.empty()));

        return result.filter(Option::isPresent).map(Option::get);
    }

    private ListF<DjfsResource> changeParentPathForChildren(DjfsResource source, ListF<DjfsResource> sourceChildren,
            DjfsResource destination)
    {
        return sourceChildren.map(child -> {
            DjfsResourcePath newChildPath = child.getPath().changeParent(source.getPath(), destination.getPath());
            return child.toBuilder().path(newChildPath).build();
        });
    }

    public DjfsResource moveSingleResource(DjfsPrincipal principal, DjfsResourcePath sourcePath,
            DjfsResourcePath destinationPath, boolean force, MoveCallbacks callbacks)
    {
        if (!isMoveImplemented(sourcePath, destinationPath, force)) {
            throw new DjfsNotImplementedException();
        }

        // WARNING!!! starting from here code relies on the statement that source and destination are not shared
        // resources. If you are trying to add support for shared resource here, rewrite the code and checks.

        // TODO: add space check when implement shared folders
        // TODO: when you start to implement move between area, add some check like that:
        //
        // if (!principal.isSystem() && !sourcePath.getArea().equals(destinationPath.getArea()) &&
        //      destinationPath.getArea().equals(DjfsResourceArea.PHOTOUNLIM))
        // {
        //    throw new SomeException();
        // }

        MoveIntent intent = new MoveIntent(principal, sourcePath, destinationPath, force);
        if (!sourcePath.getArea().isActivityPermitted(intent)) {
            throw new DjfsNotImplementedException();
        }
        if (!destinationPath.getArea().isActivityPermitted(intent)) {
            throw new DjfsNotImplementedException();
        }

        destinationPath.checkNewPath();
        if (destinationPath.isAreaRoot()) {
            throw new ForbiddenResourcePathException("area root is forbidden destination", destinationPath);
        }
        if (destinationPath.getArea() == DjfsResourceArea.TRASH && destinationPath.getDepth() != 2) {
            throw new ForbiddenResourcePathException("only move to trash root is allowed", destinationPath);
        }
        checkValidDestinationPath(sourcePath, destinationPath);

        filesystemPermissionChecker.checkReadResource(principal, sourcePath, Option.empty());
        filesystemPermissionChecker.checkWriteResource(principal, destinationPath, Option.empty());

        if (lockManager.isLocked(sourcePath)) {
            throw new ResourceLockedException(sourcePath);
        }
        if (lockManager.isLocked(destinationPath)) {
            throw new ResourceLockedException(destinationPath);
        }

        Option<DjfsResource> sourceO = djfsResourceDao.find2(sourcePath);
        if (!sourceO.isPresent()) {
            throw new ResourceNotFoundException(sourcePath);
        }
        DjfsResource source = sourceO.get();

        Option<DjfsResource> destinationParentO = djfsResourceDao.find2(destinationPath.getParent());
        if (!destinationParentO.isPresent()) {
            throw new NoParentFolderException(destinationPath);
        }
        DjfsResource destinationParent = destinationParentO.get();

        Option<DjfsResource> destinationO = djfsResourceDao.find2(destinationPath);
        if (destinationO.isPresent() && !force) {
            throw new ResourceExistsException(destinationPath);
        }

        Instant now = Instant.now();
        long version = InstantUtils.toVersion(now);

        DjfsResource destination = source.toBuilder()
                .path(destinationPath)
                .version(version)
                .modificationTime(now)
                .hasYarovayaMark(isYarovayaMarkNeeded(source))
                .build();

        ListF<DjfsResource> sourceChildren = getSubtree(source.getPath(), FOLDER_MOVE_RESOURCES_THRESHOLD);
        boolean isBigFolder = sourceChildren.length() >= FOLDER_MOVE_RESOURCES_THRESHOLD;
        ListF<DjfsResource> destinationChildren = changeParentPathForChildren(source, sourceChildren, destination);

        Option<Long> oldVersion = transactionUtils.executeInNewOrCurrentTransaction(sourcePath.getUid(), () -> {
            djfsResourceDao.changeParentAndName(
                    source.getUid(), source.getId(), destinationParent.getId(), destinationPath.getName());
            djfsResourceDao.setVersion(destination.getUid(), destination.getId(), version);
            djfsResourceDao.setModificationTime(destination.getUid(), destination.getId(),
                    destination.getModificationTimeO().get());

            if (source.getPath().getArea() != DjfsResourceArea.TRASH &&
                    destination.getPath().getArea() == DjfsResourceArea.TRASH)
            {
                djfsResourceDao.setTrashAppendTimeAndOriginalPath(destination.getUid(), destination.getId(),
                        source.getPath().getPath(), now);
            }

            if (destination.hasYarovayaMark()) {
                djfsResourceDao.setYarovayaMark(destination.getUid(), destination.getId());
            }

            if (isBigFolder && destinationPath.getArea() != DjfsResourceArea.TRASH) {
                changelogManager.createMoveChangelog(now, Changelog.OperationType.MOVED, source, Option.empty(),
                        destination);
                userDao.setMinimumDeltaVersion(source.getUid(), version);
            } else if (destinationPath.getArea() == DjfsResourceArea.TRASH) {
                // desktop client doesn't work with MOVED deltas if path starts with trash
                changelogManager.createChangelog(now, Changelog.OperationType.DELETED, source, Option.empty());
            } else {
                ListF<Changelog> changelogs = prepareChangelogsForMoveSmallFolder(
                        source, destinationChildren, destination, now);
                changelogManager.createChangelogs(changelogs);
            }

            Option<Long> versionBeforeIncrement = userDao.incrementVersionTo_ReturnOld(source.getUid(), version);

            callbacks.callAfterMoveInsideTransaction(destination);

            return versionBeforeIncrement;
        });

        MoveActivity moveActivity = new MoveActivity(principal, source, destination, force);
        searchPushGenerator.sendPush(moveActivity);

        ListF<DjfsResource> xivaChildren = isBigFolder ? Cf.list() : destinationChildren;
        if (oldVersion.isPresent()) {
            xivaPushGenerator.sendPush(moveActivity, Cf.map(source.getUid(), oldVersion.get()), xivaChildren);
        } else {
            xivaPushGenerator.sendPush(moveActivity, Cf.map(), xivaChildren);
        }

        callbacks.callAfterMoveOutsideTransaction(moveActivity);

        return destination;
    }

    public boolean isTrashAppendImplemented(DjfsResourcePath path) {
        DjfsResourcePath trashPath = path.changeParent(
                path.getParent(), DjfsResourcePath.areaRoot(path.getUid(), DjfsResourceArea.TRASH));
        return isMoveImplemented(path, trashPath, false);
    }

    public DjfsResource trashAppendResource(DjfsPrincipal principal, DjfsResourcePath path, MoveCallbacks callbacks) {
        DjfsResourcePath trashPath = path.changeParent(
                path.getParent(), DjfsResourcePath.areaRoot(path.getUid(), DjfsResourceArea.TRASH));

        if (find(principal, trashPath, Option.of(ReadPreference.primary())).isPresent()) {
            trashPath = trashPath.suffix("_" + TimeUtils.unixMicrosecods());
        }

        Function1V<DjfsResource> addTrashCleanQueueEntry = resource -> pgTrashCleanQueueDao.insert(
                new TrashCleanQueueTask(resource.getUid(), Instant.now().toDate()));

        Function1V<DjfsResource> unpublishResource = resource -> {
            if (resource.isPublic() || resource.getDownloadCounter().isPresent()) {
                djfsResourceDao.setPublishedAndRemovePublic(resource.getUid(), resource.getId());
            }
        };

        Function1V<DjfsResource> submitPostProcessOperation = resource -> {
            quickMovePostProcessTaskSubmitter.checkLimitAndSchedule(resource.getPath());
        };

        return moveSingleResource(principal, path, trashPath, false, callbacks
                .appendAfterMoveInsideTransactionCallback(addTrashCleanQueueEntry)
                .appendAfterMoveInsideTransactionCallback(unpublishResource)
                .appendAfterMoveInsideTransactionCallback(submitPostProcessOperation)
        );
    }

    public DjfsResource copySingleResource(DjfsPrincipal principal, DjfsResourcePath sourcePath,
            DjfsResourcePath destinationPath)
    {
        Option<ShareInfo> sourceShareInfo = shareInfoManager.get(sourcePath);
        Option<ShareInfo> destinationShareInfo = shareInfoManager.get(destinationPath);

        filesystemPermissionChecker.checkReadResource(principal, sourcePath, sourceShareInfo);
        filesystemPermissionChecker.checkWriteResource(principal, destinationPath, destinationShareInfo);

        DjfsResourcePath databaseSourcePath = sourceShareInfo.filterMap(x -> x.participantPathToOwnerPath(sourcePath))
                .getOrElse(sourcePath);
        DjfsResourcePath databaseDestinationPath =
                destinationShareInfo.filterMap(x -> x.participantPathToOwnerPath(destinationPath))
                        .getOrElse(destinationPath);

        Option<DjfsResource> sourceO = djfsResourceDao.find2(databaseSourcePath);
        if (!sourceO.isPresent()) {
            throw new FilesystemException("no source for copy");
        }

        Option<DjfsResource> destinationParentO = djfsResourceDao.find2(databaseDestinationPath.getParent());
        if (!destinationParentO.isPresent()) {
            throw new NoParentFolderException(destinationPath);
        }

        DjfsResource destinationParent = destinationParentO.get();
        if (!(destinationParent instanceof FolderDjfsResource)) {
            throw new ParentIsFileException(destinationPath);
        }

        Instant now = Instant.now();
        long version = InstantUtils.toVersion(now);

        DjfsResource source = sourceO.get();

        if (source instanceof FileDjfsResource) {
            long free = quotaManager.getFree(destinationParent, Option.empty());
            long size = ((FileDjfsResource) source).getSize();
            if (free < size) {
                throw new NoFreeSpaceException(destinationParent.getUid(), free, size);
            }
        }

        UUID parentId = destinationParent.getId();
        DjfsResource.Builder copyBuilder = makeCopy(principal, source, databaseDestinationPath, parentId, now);

        // make AdditionalFile copy if there is one
        Option<FileDjfsResource> additionalFileCopy = Option.empty();
        Option<FolderDjfsResource> newAdditionalFilesRootFolder = Option.empty();

        if (source instanceof FileDjfsResource && ((FileDjfsResource) source).isLivePhoto()) {
            Option<FileDjfsResource> additionalFileO =
                    djfsResourceDao.find2AdditionalFileFor((FileDjfsResource) source);
            if (additionalFileO.isPresent()) {
                FileDjfsResource additionalFile = additionalFileO.get();
                DjfsResourcePath additionalFilesRootFolderPath =
                        DjfsResourcePath.areaRoot(databaseDestinationPath.getUid(),
                                DjfsResourceArea.ADDITIONAL_DATA);

                DjfsResourcePath additionalFileDestinationPath =
                        additionalFilesRootFolderPath.getChildPath(UuidUtils.randomToHexString());
                UUID additionalFileParentId = additionalFile.getParentId().get();

                if (!databaseSourcePath.getUid().equals(databaseDestinationPath.getUid())) {
                    Option<FolderDjfsResource> additionalFilesRootFolder =
                            djfsResourceDao.find2(additionalFilesRootFolderPath).cast();
                    if (additionalFilesRootFolder.isPresent()) {
                        additionalFileParentId = additionalFilesRootFolder.get().getId();
                    } else {
                        FolderDjfsResource newFolder = makeAdditionalFilesRootFolder(databaseDestinationPath.getUid());
                        newAdditionalFilesRootFolder = Option.of(newFolder);
                        additionalFileParentId = newFolder.getId();
                    }
                }

                additionalFileCopy = Option.of((FileDjfsResource) makeCopy(principal, additionalFile,
                        additionalFileDestinationPath, additionalFileParentId, now).build());
            } else {
                ((FileDjfsResource.Builder) copyBuilder).isLivePhoto(false);
            }
        }

        // TODO: remove random uuid for id here (generate it in FileDjfsResource.cons and FolderDjfsResource.cons)
        copyBuilder.id(UUID.randomUUID());

        DjfsResource copy = copyBuilder.build();

        Option<FileDjfsResource> capturedAdditionalFileCopy = additionalFileCopy;
        Option<FolderDjfsResource> capturedNewAdditionalFilesRootFolder = newAdditionalFilesRootFolder;
        try {
            transactionUtils.executeInNewOrCurrentTransaction(databaseDestinationPath.getUid(), () -> {
                if (capturedAdditionalFileCopy.isPresent()) {
                    FileDjfsResource fileCopy = (FileDjfsResource) copy;

                    if (capturedNewAdditionalFilesRootFolder.isPresent()) {
                        initializeAdditionalFilesRootFolder(capturedNewAdditionalFilesRootFolder.get());
                    }
                    djfsResourceDao.insert2(capturedAdditionalFileCopy.get());

                    try {
                        djfsResourceDao.insert2AndLinkTo(fileCopy, capturedAdditionalFileCopy.get());
                    } catch (EntityAlreadyExistsException e) {
                        throw new ResourceExistsException(copy.getPath(), e);
                    }
                } else {
                    try {
                        djfsResourceDao.insert2(copy);
                    } catch (EntityAlreadyExistsException e) {
                        throw new ResourceExistsException(copy.getPath(), e);
                    }
                }

                changelogManager.createChangelog(now, Changelog.OperationType.NEW, copy, destinationShareInfo);
            });
        } catch (TransactionSystemException e) {
            try {
                djfsResourceDao.checkAlreadyExistsException(e, copy);
            } catch (EntityAlreadyExistsException ex) {
                throw new ResourceExistsException(copy.getPath(), ex);
            }
            throw e;
        }

        if (copy instanceof FileDjfsResource) {
            long size = ((FileDjfsResource) copy).getSize();
            quotaManager.incrementUsed(databaseDestinationPath, destinationShareInfo, size);
        }

        CopyActivity copyActivity = new CopyActivity(principal, source, copy, sourceShareInfo, destinationShareInfo);

        MapF<DjfsUid, Long> oldVersions = Cf.hashMap();
        if (destinationShareInfo.isPresent()) {
            for (DjfsUid uid : destinationShareInfo.get().allUids()) {
                Option<Long> oldVersion = userDao.incrementVersionTo_ReturnOld(uid, version);
                if (oldVersion.isPresent()) {
                    oldVersions.put(uid, oldVersion.get());
                }
            }
        } else {
            Option<Long> oldVersion = userDao.incrementVersionTo_ReturnOld(copy.getUid(), version);
            if (oldVersion.isPresent()) {
                oldVersions.put(copy.getUid(), oldVersion.get());
            }
        }

        xivaPushGenerator.sendPush(copyActivity, oldVersions);
        searchPushGenerator.sendPush(copyActivity);
        if (destinationShareInfo.isPresent()) {
            lastFilesCacheUpdater.sendUpdateTask(destinationShareInfo.get().getGroupId());
        }

        return copy;
    }

    private void initializeAdditionalFilesRootFolder(FolderDjfsResource additionalFilesRootFolder) {
        if (!additionalFilesRootFolder.getPath().isAreaRoot() ||
                !additionalFilesRootFolder.getPath().getArea().equals(DjfsResourceArea.ADDITIONAL_DATA))
        {
            throw new InvalidDjfsResourcePathException(additionalFilesRootFolder.getPath().getFullPath());
        }

        try {
            djfsResourceDao.insert2(additionalFilesRootFolder);
            userDao.addCollection(additionalFilesRootFolder.getUid(),
                    DjfsResourceArea.ADDITIONAL_DATA.mongoCollectionName);
        } catch (EntityAlreadyExistsException ignore) {
        }
    }

    private FolderDjfsResource makeAdditionalFilesRootFolder(DjfsUid uid) {
        long version = InstantUtils.toVersion(Instant.now());
        DjfsResourcePath rootPath = DjfsResourcePath.root(uid, DjfsResourceArea.ADDITIONAL_DATA);
        DjfsResource rootFolder = djfsResourceDao.find(rootPath).get();

        DjfsResourcePath areaRootPath = DjfsResourcePath.areaRoot(uid, DjfsResourceArea.ADDITIONAL_DATA);
        return FolderDjfsResource.builder()
                .id(areaRootPath.getPgId())
                .path(areaRootPath)
                .parentId(Option.of(rootFolder.getId()))
                .fileId(DjfsFileId.random())
                .version(Option.of(version))
                .build();
    }

    private DjfsResource.Builder makeCopy(DjfsPrincipal principal, DjfsResource resource, DjfsResourcePath newPath,
            UUID parentId, Instant instant)
    {
        DjfsResource.Builder result = resource.toBuilder()
                .id(newPath.getPgId())
                .parentId(parentId)
                .path(newPath)
                .fileId(DjfsFileId.random())
                .creationTime(instant)
                .modificationTime(instant)
                .version(InstantUtils.toVersion(instant))
                .externalProperty("source_platform", YcridUtils.getPlatform())

                .isPublic(false)
                .publicHash(Option.empty())
                .symlink(Option.empty())
                .downloadCounter(Option.empty())
                .shortUrl(Option.empty())

                .area(newPath.getArea())

                .hasYarovayaMark(false);

        if (principal.isUid()) {
            result.modifyUid(principal.getUid());
        }

        return result;
    }

    /**
     * WARNING
     * Only moves file record in filesystem. Does not update anything else yet.
     * Works only for PostgreSQL.
     */
    public void moveToHidden(DjfsPrincipal principal, FileDjfsResource file, DjfsResourcePath destination) {
        if (!principal.isSystem()) {
            throw new ru.yandex.bolts.internal.NotImplementedException();
        }

        Option<DjfsResource> parent = find(principal, destination.getParent(), Option.of(ReadPreference.primary()));
        if (!parent.isPresent()) {
            throw new ru.yandex.bolts.internal.NotImplementedException();
        }

        if (!(parent.get() instanceof FolderDjfsResource)) {
            throw new ru.yandex.bolts.internal.NotImplementedException();
        }

        if (destination.getArea() != DjfsResourceArea.HIDDEN) {
            throw new ru.yandex.bolts.internal.NotImplementedException();
        }

        djfsResourceDao.changeParent(file.getUid(), file.getId(), parent.get().getId());
        djfsResourceDao.setHiddenAppendTime(file.getUid(), file.getId(), Instant.now());
    }

    /**
     * Use in tests only!
     * Not ready for production!
     */
    public void initialize(DjfsUid uid) {
        initializeArea(uid, DjfsResourceArea.DISK);
        initializeArea(uid, DjfsResourceArea.TRASH);
        initializeArea(uid, DjfsResourceArea.PHOTOUNLIM);
    }

    /**
     * Use in tests only!
     * Not ready for production!
     *
     * @return Tuple of: [Root, AreaRoot]
     */
    public Tuple2<FolderDjfsResource, FolderDjfsResource> initializeArea(DjfsUid uid, DjfsResourceArea area) {
        // todo: update user index version?
        long version = InstantUtils.toVersion(Instant.now());
        DjfsResourcePath rootPath = DjfsResourcePath.root(uid, area);
        FolderDjfsResource rootFolder = FolderDjfsResource.builder()
                .id(rootPath.getPgId())
                .path(rootPath)
                .version(Option.of(version))
                .fileId(DjfsFileId.random())
                .build();

        DjfsResourcePath areaRootPath = DjfsResourcePath.areaRoot(uid, area);
        FolderDjfsResource areaRootFolder = FolderDjfsResource.builder()
                .id(areaRootPath.getPgId())
                .path(areaRootPath)
                .parentId(Option.of(rootFolder.getId()))
                .fileId(DjfsFileId.random())
                .version(Option.of(version))
                .build();

        Option<DjfsResource> rootFolderO = djfsResourceDao.find(rootPath);
        // todo: cases are equivalent, refactor

        if (rootFolderO.isPresent()) {
            rootFolder = (FolderDjfsResource) rootFolderO.get();
        } else {
            djfsResourceDao.insert(rootFolder);
        }

        areaRootFolder = FolderDjfsResource.builder()
            .id(areaRootPath.getPgId())
            .path(areaRootPath)
            .parentId(Option.of(rootFolder.getId()))
            .fileId(DjfsFileId.random())
            .version(Option.of(version))
            .build();
        djfsResourceDao.insert(areaRootFolder);

        userDao.addCollection(uid, area.mongoCollectionName);
        return Tuple2.tuple(rootFolder, areaRootFolder);
    }

    public Option<DjfsResource> find(DjfsPrincipal principal, DjfsResourcePath path, Option<ReadPreference> readPreference) {
        return path.getArea().find(principal, path, shareInfoManager, filesystemPermissionChecker, djfsResourceDao, readPreference);
    }

    public Option<DjfsResource> findWithoutCheckingPermission(DjfsResourcePath path) {
        Option<ShareInfo> shareInfoO = shareInfoManager.getWithRoot(path, Option.empty());
        DjfsResourcePath correctPath = shareInfoO.filterMap(x -> x.participantPathToOwnerPath(path)).getOrElse(path);
        return djfsResourceDao.find(correctPath);
    }


    public ListF<DjfsResource> find(DjfsPrincipal principal, DjfsResourceId resourceId, Option<ReadPreference> readPreference) {
        ListF<DjfsResource> resources = djfsResourceDao.find(resourceId);
        return resources.filter(x -> filesystemPermissionChecker.canReadResourceWithReadPreference(principal, x.getPath(), readPreference));
    }

    public ListF<DjfsResource> find(DjfsPrincipal principal, ListF<DjfsResourceId> resourceIds,
            ListF<DjfsResourceArea> areasToSearch, Option<ReadPreference> readPreference)
    {
        return findResourcesWithParentIds(principal, resourceIds, areasToSearch, readPreference).map(Tuple2::get1);
    }

    public ListF<DjfsResource> findByPaths(DjfsPrincipal principal, ListF<DjfsResourcePath> resourcePaths) {
        return findResourcesWithParentIdsByPaths(principal, resourcePaths, Option.of(ReadPreference.primary()));
    }

    public ListF<DjfsResource> findByPaths(DjfsPrincipal principal, ListF<DjfsResourcePath> resourcePaths,
            Option<ReadPreference> readPreference)
    {
        return findResourcesWithParentIdsByPaths(principal, resourcePaths, readPreference);
    }

    private ListF<DjfsResource> findResourcesWithParentIdsByPaths(
            DjfsPrincipal principal, ListF<DjfsResourcePath> resourcePaths, Option<ReadPreference> readPreference)
    {
        //TODO: Move separating requests by uid (or by shards) and handling errors to invoke and improve with concurrent execution
        MapF<DjfsUid, ListF<DjfsResourcePath>> uidToPathsMap = Cf.hashMap();
        for (DjfsResourcePath path : resourcePaths) {
            DjfsResourcePath correctPath = path.getArea().getCorrectPath(principal, path, filesystemPermissionChecker,
                    shareInfoManager, readPreference);

            if (uidToPathsMap.containsKeyTs(correctPath.getUid())) {
                uidToPathsMap.getTs(correctPath.getUid()).add(correctPath);
            } else {
                uidToPathsMap.put(correctPath.getUid(), Cf.arrayList(correctPath));
            }
        }

        ListF<DjfsResource> resources = Cf.arrayList();
        for (Tuple2<DjfsUid, ListF<DjfsResourcePath>> tuple : uidToPathsMap.entries()) {
            try {
                resources.addAll(djfsResourceDao.findByPaths(tuple.get1(), tuple.get2()));
            } catch (UserNotInitializedException e) {
                // continue
            }
        }
        return resources;
    }

    public Option<Tuple2<DjfsResource, ListF<UUID>>> findResourceWithParentIds(
            DjfsPrincipal principal, DjfsResourcePath path)
    {
        filesystemPermissionChecker.checkReadResource(principal, path);
        Option<ShareInfo> shareInfoO = shareInfoManager.get(path);
        DjfsResourcePath correctPath = shareInfoO.filterMap(x -> x.participantPathToOwnerPath(path)).getOrElse(path);

        Option<Tuple2<DjfsResource, ListF<UUID>>> resourceWithParentIds =
                djfsResourceDao.findWithParentIds(correctPath);

        if (!resourceWithParentIds.isPresent()) {
            return Option.empty();
        }
        return resourceWithParentIds;
    }

    public ListF<Tuple2<DjfsResource, ListF<UUID>>> findResourcesWithParentIds(
            DjfsPrincipal principal, ListF<DjfsResourceId> resourceIds, ListF<DjfsResourceArea> areasToSearch,
            Option<ReadPreference> readPreference)
    {
        //TODO: Move separating requests by uid (or by shards) and handling errors to invoke and improve with concurrent execution
        MapF<DjfsUid, ListF<String>> uidToFileIdsMap = Cf.hashMap();
        for (DjfsResourceId resourceId : resourceIds) {
            if (uidToFileIdsMap.containsKeyTs(resourceId.getUid())) {
                uidToFileIdsMap.getO(resourceId.getUid()).get().add(resourceId.getFileId().getValue());
            } else {
                uidToFileIdsMap.put(resourceId.getUid(), Cf.arrayList(resourceId.getFileId().getValue()));
            }
        }

        ListF<Tuple2<DjfsResource, ListF<UUID>>> resources = Cf.arrayList();
        for (Tuple2<DjfsUid, ListF<String>> tuple : uidToFileIdsMap.entries()) {
            ListF<Tuple2<DjfsResource, ListF<UUID>>> resourcesChunk;
            try {
                resourcesChunk = djfsResourceDao.findWithParentIds(tuple.get1(), tuple.get2(), areasToSearch);
            } catch (UserNotInitializedException e) {
                continue;
            }

            resources.addAll(resourcesChunk.filter(
                    x -> filesystemPermissionChecker.canReadResourceWithReadPreference(principal, x.get1().getPath(), readPreference)));
        }
        Comparator<Tuple2<DjfsResource, ListF<UUID>>> comparator = Comparator
                .<Tuple2<DjfsResource, ListF<UUID>>>comparingLong(x -> x.get1().getVersion().getOrElse(0L)).reversed()
                .thenComparing(x -> x.get1().getId());

        return resources.groupBy(x -> x.get1().getResourceId().get().toString()).mapValues(
                x -> x.sorted(comparator).first()).values().toList();
    }

    public void setAesthetics(DjfsPrincipal principal, DjfsResourceId resourceId, double aesthetics,
            boolean skipIfPresent)
    {
        ListF<DjfsResource> resources = find(principal, resourceId, Option.of(ReadPreference.primary()));
        for (DjfsResource resource : resources) {
            if (resource instanceof FolderDjfsResource) {
                continue;
            }
            if (skipIfPresent && ((FileDjfsResource) resource).getAesthetics().isPresent()) {
                continue;
            }

            djfsResourceDao.setAesthetics(resource.getUid(), resource.getId(), aesthetics);
        }
    }

    public void setDimensionsWithAngle(DjfsPrincipal principal, DjfsResourceId resourceId, Dimension dimensions,
            int angle, boolean skipIfPresent)
    {
        ListF<DjfsResource> resources = find(principal, resourceId, Option.of(ReadPreference.primary()));

        Option<FileDjfsResource> anyResourceO = resources.filterByType(FileDjfsResource.class).firstO();
        if (!anyResourceO.isPresent()) {
            return;
        }
        FileDjfsResource resource = anyResourceO.get();

        if (skipIfPresent && (resource.getWidth().isPresent() && resource.getHeight().isPresent() &&
                resource.getAngle().isPresent()))
        {
            return;
        }

        djfsResourceDao.setDimensionsWithAngle(resource.getUid(), resource.getId(), dimensions, angle);
    }

    public void setCoordinates(DjfsPrincipal principal, DjfsResourceId resourceId, Coordinates coordinates,
            boolean skipIfPresent)
    {
        ListF<DjfsResource> resources = find(principal, resourceId, Option.of(ReadPreference.primary()));
        for (DjfsResource resource : resources) {
            if (resource instanceof FolderDjfsResource) {
                continue;
            }
            if (skipIfPresent && ((FileDjfsResource) resource).getCoordinates().isPresent()) {
                continue;
            }

            djfsResourceDao.setCoordinates(resource.getUid(), resource.getId(), coordinates);
        }
    }

    public ListF<DjfsResource> filterBlocked(ListF<DjfsResource> resources) {
        SetF<String> hids = resources
                .filterByType(FileDjfsResource.class)
                .map(FileDjfsResource::getHid).unique();

        SetF<String> blockedHids = blockedHidsDao
                .find(hids.toList())
                .map(SupportBlockedHid::getResourceHid)
                .unique();

        return resources.filter(r -> !(r instanceof FileDjfsResource && blockedHids
                .containsTs(((FileDjfsResource) r).getHid())));
    }

    public List<DjfsResource> getChildren(DjfsPrincipal principal, DjfsResource resource, int offset, int amount,
                                          Option<ReadPreference> readPreference) {
        if (!(resource instanceof FolderDjfsResource)) {
            return Cf.list();
        }
        final ListF<DjfsResource> children = djfsResourceDao.find2ImmediateChildren(
                resource.getPath().getArea().getSharedResource(resource, shareInfoManager, filesystemPermissionChecker,
                        djfsResourceDao, readPreference),
                offset, amount, true
        );
        final UserData principalUser = principal.getUserO().orElse(() -> userDao.find(principal.getUid()))
                .getOrThrow(() -> new UserNotInitializedException(principal.getUid()));
        DjfsPrincipal userPrincipal = DjfsPrincipal.cons(principalUser);
        return children.filter(child -> filesystemPermissionChecker.canReadResourceWithReadPreference(userPrincipal,
                child.getPath(), readPreference));
    }
}
