package ru.yandex.chemodan.app.djfs.core.filesystem;

import java.util.Objects;
import java.util.function.Supplier;

import com.mongodb.ReadPreference;
import lombok.RequiredArgsConstructor;

import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.djfs.core.filesystem.exception.NoPermissionException;
import ru.yandex.chemodan.app.djfs.core.filesystem.exception.PrincipalBlockedException;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.DjfsResourcePath;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.FilesystemOperation;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.FilesystemPermission;
import ru.yandex.chemodan.app.djfs.core.share.ShareInfo;
import ru.yandex.chemodan.app.djfs.core.share.ShareInfoManager;
import ru.yandex.chemodan.app.djfs.core.user.DjfsUid;
import ru.yandex.chemodan.app.djfs.core.user.UserDao;
import ru.yandex.chemodan.app.djfs.core.user.UserData;
import ru.yandex.chemodan.app.djfs.core.user.UserNotInitializedException;

/**
 * Handles only permanent filesystem restrictions.
 * Supports shared folders, blocked users.
 * <p>
 * Temporary file system restrictions are handled by LockManager.
 *
 * @author eoshch
 * @see ru.yandex.chemodan.app.djfs.core.lock.LockManager
 */
@RequiredArgsConstructor
public class FilesystemAccessController {
    private final ShareInfoManager shareInfoManager;
    private final UserDao userDao;

    public final ShareAccessController share = new ShareAccessController();

    private boolean isPrincipalBlocked(DjfsPrincipal principal) {
        if (principal.isUid()) {
            final DjfsUid uid = principal.getUid();
            UserData user = principal.getUserO().getOrElse(() -> userDao.find(uid)
                    .getOrThrow(() -> new UserNotInitializedException(uid)));
            return user.isBlocked();
        }

        return false;
    }

    private boolean hasFilesystemPermission(DjfsPrincipal principal, DjfsResourcePath path, Option<ShareInfo> shareInfo,
            FilesystemPermission permission)
    {
        return hasFilesystemPermission(principal, path, () -> shareInfo, permission);
    }

    private boolean hasFilesystemPermission(DjfsPrincipal principal, DjfsResourcePath path, Supplier<Option<ShareInfo>> shareInfoProvider,
                                            FilesystemPermission permission)
    {
        if (principal.isSystem() || principal.isInternalService() || principal.isSupport()) {
            return true;
        }

        if (principal.isUid()) {
            final DjfsUid uid = principal.getUid();

            if (path.getArea().shouldCheckGroupPathPermissions()) {
                Option<ShareInfo> shareInfo = shareInfoProvider.get();
                if (shareInfo.isPresent()) {
                    return shareInfo.filterMap(x -> x.getPermissions(uid))
                            .map(x -> x.hasPermission(permission)).getOrElse(false);
                }
            }
            return Objects.equals(uid, path.getUid());
        }
        return false;
    }

    public boolean canReadResourceWithReadPreference(DjfsPrincipal principal, DjfsResourcePath path, Option<ReadPreference> readPreference) {
        return canReadResource(principal, path, () -> shareInfoManager.get(path, readPreference));
    }

    public boolean canReadResource(DjfsPrincipal principal, DjfsResourcePath path, Supplier<Option<ShareInfo>> shareInfoProvider) {
        return !isPrincipalBlocked(principal)
                && hasFilesystemPermission(principal, path, shareInfoProvider, FilesystemPermission.READ);
    }

    public void checkReadResource(DjfsPrincipal principal, DjfsResourcePath path) {
        Option<ShareInfo> shareInfo = shareInfoManager.get(path);
        checkReadResource(principal, path, shareInfo);
    }

    public void checkReadResourceWithReadPreference(DjfsPrincipal principal, DjfsResourcePath path, Option<ReadPreference> readPreference) {
        Option<ShareInfo> shareInfo = shareInfoManager.get(path, readPreference);
        checkReadResource(principal, path, shareInfo);
    }

    public void checkReadResource(DjfsPrincipal principal, DjfsResourcePath path, Option<ShareInfo> shareInfo) {
        if (isPrincipalBlocked(principal)) {
            throw new PrincipalBlockedException(principal, FilesystemOperation.READ_RESOURCE, path, shareInfo);
        }
        if (!hasFilesystemPermission(principal, path, shareInfo, FilesystemPermission.READ)) {
            throw new NoPermissionException(principal, FilesystemOperation.READ_RESOURCE, path, shareInfo);
        }
    }

    public boolean canWriteResource(DjfsPrincipal principal, DjfsResourcePath path, Option<ShareInfo> shareInfo) {
        return !isPrincipalBlocked(principal)
                && hasFilesystemPermission(principal, path, shareInfo, FilesystemPermission.WRITE);
    }

    public void checkWriteResource(DjfsPrincipal principal, DjfsResourcePath path) {
        Option<ShareInfo> shareInfo = shareInfoManager.get(path);
        checkWriteResource(principal, path, shareInfo);
    }

    public void checkWriteResource(DjfsPrincipal principal, DjfsResourcePath path, Option<ShareInfo> shareInfo) {
        if (isPrincipalBlocked(principal)) {
            throw new PrincipalBlockedException(principal, FilesystemOperation.WRITE_RESOURCE, path, shareInfo);
        }
        if (!hasFilesystemPermission(principal, path, shareInfo, FilesystemPermission.WRITE)) {
            throw new NoPermissionException(principal, FilesystemOperation.WRITE_RESOURCE, path, shareInfo);
        }
    }

    public void checkCreateFolder(DjfsPrincipal principal, DjfsResourcePath path, Option<ShareInfo> shareInfo) {
        if (isPrincipalBlocked(principal)) {
            throw new PrincipalBlockedException(principal, FilesystemOperation.CREATE_FOLDER, path, shareInfo);
        }
        if (!hasFilesystemPermission(principal, path, shareInfo, FilesystemPermission.WRITE)) {
            throw new NoPermissionException(principal, FilesystemOperation.CREATE_FOLDER, path, shareInfo);
        }
    }

    public void checkCreateFile(DjfsPrincipal principal, DjfsResourcePath path, Option<ShareInfo> shareInfo) {
        if (isPrincipalBlocked(principal)) {
            throw new PrincipalBlockedException(principal, FilesystemOperation.CREATE_FILE, path, shareInfo);
        }
        if (!hasFilesystemPermission(principal, path, shareInfo, FilesystemPermission.WRITE)) {
            throw new NoPermissionException(principal, FilesystemOperation.CREATE_FILE, path, shareInfo);
        }
    }

    public class ShareAccessController {
        private ShareAccessController() {
        }

        private boolean canLeaveImpl(DjfsPrincipal principal, DjfsUid uid, ShareInfo shareInfo) {
            if (principal.isSystem() || principal.isInternalService() || principal.isSupport()) {
                return true;
            }

            if (principal.isUid()) {
                final DjfsUid principalUid = principal.getUid();
                return Objects.equals(principalUid, uid) && !Objects.equals(principalUid, shareInfo.getOwnerUid());
            }

            return false;
        }

        public boolean canLeave(DjfsPrincipal principal, DjfsUid uid, ShareInfo shareInfo) {
            return !FilesystemAccessController.this.isPrincipalBlocked(principal)
                    && canLeaveImpl(principal, uid, shareInfo);
        }

        public void checkLeave(DjfsPrincipal principal, DjfsUid uid, ShareInfo shareInfo) {
            if (FilesystemAccessController.this.isPrincipalBlocked(principal)) {
                throw new PrincipalBlockedException(principal, FilesystemOperation.SHARE_LEAVE_GROUP, uid, shareInfo);
            }

            if (!canLeaveImpl(principal, uid, shareInfo)) {
                throw new NoPermissionException(principal, FilesystemOperation.SHARE_LEAVE_GROUP, uid, shareInfo);
            }
        }

        private boolean canKickImpl(DjfsPrincipal principal, DjfsUid uid, ShareInfo shareInfo) {
            if (principal.isSystem() || principal.isInternalService() || principal.isSupport()) {
                return true;
            }

            if (principal.isUid()) {
                final DjfsUid principalUid = principal.getUid();
                return Objects.equals(principalUid, shareInfo.getOwnerUid()) && !Objects.equals(principalUid, uid);
            }

            return false;
        }

        public boolean canKick(DjfsPrincipal principal, DjfsUid uid, ShareInfo shareInfo) {
            return !FilesystemAccessController.this.isPrincipalBlocked(principal)
                    && canKickImpl(principal, uid, shareInfo);
        }

        public void checkKick(DjfsPrincipal principal, DjfsUid uid, ShareInfo shareInfo) {
            if (FilesystemAccessController.this.isPrincipalBlocked(principal)) {
                throw new PrincipalBlockedException(principal, FilesystemOperation.SHARE_KICK_USER, uid, shareInfo);
            }

            if (!canKickImpl(principal, uid, shareInfo)) {
                throw new NoPermissionException(principal, FilesystemOperation.SHARE_KICK_USER, uid, shareInfo);
            }
        }

        private boolean canRemoveAllSharedFoldersImpl(DjfsPrincipal principal, DjfsUid uid) {
            return principal.isSystem();
        }

        public void checkRemoveAllSharedFolders(DjfsPrincipal principal, DjfsUid uid) {
            if (FilesystemAccessController.this.isPrincipalBlocked(principal)) {
                throw new PrincipalBlockedException(principal, FilesystemOperation.REMOVE_ALL_SHARED_FOLDERS, uid);
            }

            if (!canRemoveAllSharedFoldersImpl(principal, uid)) {
                throw new NoPermissionException(principal, FilesystemOperation.REMOVE_ALL_SHARED_FOLDERS, uid);
            }
        }
    }
}
