package ru.yandex.chemodan.app.djfs.core.filesystem;

import com.mongodb.MongoClient;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Import;
import org.springframework.context.annotation.Primary;

import ru.yandex.chemodan.app.djfs.core.EventManager;
import ru.yandex.chemodan.app.djfs.core.EventManagerContextConfiguration;
import ru.yandex.chemodan.app.djfs.core.billing.BillingManager;
import ru.yandex.chemodan.app.djfs.core.changelog.ChangelogManager;
import ru.yandex.chemodan.app.djfs.core.client.OperationCallbackHttpClient;
import ru.yandex.chemodan.app.djfs.core.db.DaoProxyFactory;
import ru.yandex.chemodan.app.djfs.core.db.DjfsDbContextConfiguration;
import ru.yandex.chemodan.app.djfs.core.db.mongo.MongoShardResolver;
import ru.yandex.chemodan.app.djfs.core.db.pg.PgShardedDaoContext;
import ru.yandex.chemodan.app.djfs.core.db.pg.TransactionUtils;
import ru.yandex.chemodan.app.djfs.core.diskinfo.DiskInfoDao;
import ru.yandex.chemodan.app.djfs.core.filesystem.iteration.IterationContextConfiguration;
import ru.yandex.chemodan.app.djfs.core.filesystem.operation.FilesystemOperationContextConfiguration;
import ru.yandex.chemodan.app.djfs.core.filesystem.operation.postprocess.QuickMovePostProcessTaskSubmitter;
import ru.yandex.chemodan.app.djfs.core.index.SearchPushGenerator;
import ru.yandex.chemodan.app.djfs.core.lastfiles.LastFilesCacheUpdater;
import ru.yandex.chemodan.app.djfs.core.lock.LockManager;
import ru.yandex.chemodan.app.djfs.core.notification.EmailGenerator;
import ru.yandex.chemodan.app.djfs.core.notification.XivaPushGenerator;
import ru.yandex.chemodan.app.djfs.core.publication.LinkDataDao;
import ru.yandex.chemodan.app.djfs.core.publication.MongoBlockingsDao;
import ru.yandex.chemodan.app.djfs.core.share.GroupDao;
import ru.yandex.chemodan.app.djfs.core.share.GroupLinkDao;
import ru.yandex.chemodan.app.djfs.core.share.ShareInfoManager;
import ru.yandex.chemodan.app.djfs.core.user.OrganizationDao;
import ru.yandex.chemodan.app.djfs.core.user.UserDao;
import ru.yandex.chemodan.queller.worker.CeleryTaskManager;
import ru.yandex.commune.dynproperties.DynamicProperty;
import ru.yandex.misc.thread.ParallelStreamSupplier;

/**
 * @author eoshch
 */
@Configuration
@Import({
        DjfsDbContextConfiguration.class,
        EventManagerContextConfiguration.class,
        FilesystemOperationContextConfiguration.class,
        IterationContextConfiguration.class,
})
public class FilesystemContextConfiguration {
    private final DynamicProperty<Boolean> isSupportInPG =
            new DynamicProperty<>("djfs.common.is-support-in-pg", true);
    private final DynamicProperty<Integer> cacheTtl =
            new DynamicProperty<>("filesystem.client-cache-ttl-minutes", 1);

    @Bean
    public MongoDjfsResourceDao mongoDjfsResourceDao(@Qualifier("mongo") ParallelStreamSupplier parallelStreamSupplier,
            MongoShardResolver mongoShardResolver)
    {
        return new MongoDjfsResourceDao(parallelStreamSupplier, mongoShardResolver);
    }

    @Bean
    public PgDjfsResourceDao pgDjfsResourceDao(PgShardedDaoContext context,
         @Value("${bulk-info.inner-threads:-20}") int bulkInfoInnerThreads)
    {
        return new PgDjfsResourceDao(context, bulkInfoInnerThreads);
    }

    @Bean
    @Primary
    public DjfsResourceDao djfsResourceDaoProxy(DaoProxyFactory daoProxyFactory, MongoDjfsResourceDao mongoDao,
            PgDjfsResourceDao pgDao)
    {
        return daoProxyFactory.create(DjfsResourceDao.class, pgDao);
    }

    @Bean
    public MongoSupportBlockedHidsDao mongoSupportBlockedHidsDao(@Qualifier("common") MongoClient mongoClient) {
        return new MongoSupportBlockedHidsDao(mongoClient);
    }

    @Bean
    public PgSupportBlockedHidsDao pgSupportBlockedHidsDao(PgShardedDaoContext context) {
        return new PgSupportBlockedHidsDao(context);
    }

    @Bean
    @Primary
    public SupportBlockedHidsDao supportBlockedHidsDaoProxy(DaoProxyFactory daoProxyFactory,
            MongoSupportBlockedHidsDao mongoDao,
            PgSupportBlockedHidsDao pgDao)
    {
        return daoProxyFactory.create(SupportBlockedHidsDao.class, pgDao, mongoDao, isSupportInPG::get);
    }

    @Bean
    public PgTrashCleanQueueDao pgTrashCleanQueueDao(PgShardedDaoContext context) {
        return new PgTrashCleanQueueDao(context);
    }

    @Bean
    public MongoBlockingsDao mongoBlockingsDao(@Qualifier("blockings") MongoClient mongoClient) {
        return new MongoBlockingsDao(mongoClient);
    }

    @Bean
    public MongoSupportDao mongoSupportDao(@Qualifier("common") MongoClient mongoClient) {
        return new MongoSupportDao(mongoClient);
    }

    @Bean
    public PgSupportDao pgSupportDao(PgShardedDaoContext context) {
        return new PgSupportDao(context);
    }

    @Bean
    @Primary
    public SupportDao supportDaoProxy(DaoProxyFactory daoProxyFactory,
            MongoSupportDao mongoDao,
            PgSupportDao pgDao)
    {
        return daoProxyFactory.create(SupportDao.class, pgDao, mongoDao, isSupportInPG::get);
    }

    @Bean
    public FilesystemAccessController filesystemPermissionChecker(ShareInfoManager shareInfoManager, UserDao userDao) {
        return new FilesystemAccessController(shareInfoManager, userDao);
    }

    @Bean
    public Filesystem filesystem(DjfsResourceDao djfsResourceDao, UserDao userDao,
         LockManager lockManager, GroupDao groupDao, GroupLinkDao groupLinkDao,
         ShareInfoManager shareInfoManager, FilesystemAccessController filesystemPermissionChecker,
         EventManager eventManager, QuotaManager quotaManager, ChangelogManager changelogManager,
         SearchPushGenerator searchPushGenerator, XivaPushGenerator xivaPushGenerator,
         LastFilesCacheUpdater lastFilesCacheUpdater, TransactionUtils transactionUtils,
         LinkDataDao linkDataDao, SupportBlockedHidsDao supportBlockedHidsDao,
         PgTrashCleanQueueDao pgTrashCleanQueueDao,
         QuickMovePostProcessTaskSubmitter quickMovePostProcessTaskSubmitter)
    {
        return new Filesystem(djfsResourceDao, userDao, lockManager, groupDao, groupLinkDao,
                shareInfoManager, filesystemPermissionChecker, eventManager, quotaManager, changelogManager,
                searchPushGenerator, xivaPushGenerator, lastFilesCacheUpdater, transactionUtils, linkDataDao,
                supportBlockedHidsDao, pgTrashCleanQueueDao, quickMovePostProcessTaskSubmitter);
    }

    @Bean
    public FilesystemActions filesystemActions(Filesystem filesystem, SupportDao supportDao) {
        return new FilesystemActions(filesystem, supportDao, cacheTtl::get);
    }

    @Bean
    public QuotaManager quotaManager(UserDao userDao, DiskInfoDao diskInfoDao, GroupDao groupDao,
            OrganizationDao organizationDao, XivaPushGenerator xivaPushGenerator,
            EmailGenerator emailGenerator, BillingManager billingManager)
    {
        return new QuotaManager(userDao, diskInfoDao, groupDao, organizationDao, xivaPushGenerator,
                emailGenerator, billingManager);
    }

    @Bean
    public OperationCallbackHandler operationCallbackHandler(CeleryTaskManager celeryTaskManager,
            OperationCallbackHttpClient operationCallbackHttpClient)
    {
        return new OperationCallbackHandler(operationCallbackHttpClient, celeryTaskManager);
    }
}
