package ru.yandex.chemodan.app.djfs.core.filesystem;

import java.util.Objects;
import java.util.UUID;
import java.util.function.Supplier;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import com.mongodb.ErrorCategory;
import com.mongodb.MongoClient;
import com.mongodb.MongoWriteException;
import com.mongodb.ReadPreference;
import com.mongodb.client.MongoCollection;
import com.mongodb.client.model.Filters;
import com.mongodb.client.model.Projections;
import com.mongodb.client.model.Sorts;
import com.mongodb.client.model.Updates;
import lombok.RequiredArgsConstructor;
import org.bson.BsonDocument;
import org.bson.BsonString;
import org.bson.BsonValue;
import org.bson.conversions.Bson;
import org.joda.time.Instant;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.CollectionF;
import ru.yandex.bolts.collection.CollectorsF;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.bolts.collection.Tuple3;
import ru.yandex.bolts.internal.NotImplementedException;
import ru.yandex.chemodan.app.djfs.core.FilenamePreviewStidMimetypeVersionFileId;
import ru.yandex.chemodan.app.djfs.core.db.DjfsShardInfo;
import ru.yandex.chemodan.app.djfs.core.db.DjfsUidSource;
import ru.yandex.chemodan.app.djfs.core.db.EntityAlreadyExistsException;
import ru.yandex.chemodan.app.djfs.core.db.mongo.MongoShardResolver;
import ru.yandex.chemodan.app.djfs.core.db.mongo.MongoUtil;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.DjfsFileId;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.DjfsResource;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.DjfsResourceArea;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.DjfsResourceId;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.DjfsResourcePath;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.FileDjfsResource;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.FolderDjfsResource;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.StidType;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.mongo.MongoResource;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.mongo.ResourceFactory;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.mongo.parsing.MongoFileDjfsResourceParser;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.mongo.parsing.MongoFileDjfsResourceSerializer;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.mongo.parsing.MongoFolderDjfsResourceParser;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.mongo.parsing.MongoFolderDjfsResourceSerializer;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.mongo.parsing.ResourceFactory2;
import ru.yandex.chemodan.app.djfs.core.user.DjfsUid;
import ru.yandex.chemodan.app.djfs.core.util.UuidUtils;
import ru.yandex.commune.mongo3.MongoCollectionX;
import ru.yandex.misc.geo.Coordinates;
import ru.yandex.misc.image.Dimension;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.thread.ParallelStreamSupplier;

/**
 * @author eoshch
 */
@RequiredArgsConstructor
public class MongoDjfsResourceDao implements DjfsResourceDao {
    private static final Logger logger = LoggerFactory.getLogger(MongoDjfsResourceDao.class);

    private final ParallelStreamSupplier parallelStreamSupplier;
    private final MongoShardResolver mongoShardResolver;

    @Override
    public Option<DjfsResource> find(DjfsUid uid, DjfsResourceArea area, UUID id) {
        Option<MongoResource> r = getCollectionX(uid, area).findOne(
                Filters.eq("_id", UuidUtils.toHexString(id)));
        return r.map(ResourceFactory::create);
    }

    @Override
    public Option<DjfsResource> find(DjfsUid uid, DjfsResourceArea area, UUID id, boolean forUpdate) {
        return find(uid, area, id);
    }

    @Override
    public Option<DjfsResource> find2(DjfsUid uid, DjfsResourceArea area, UUID id) {
        MongoCollection<BsonDocument> collection = getCollection(uid, area);
        BsonDocument r = MongoUtil.retryTimeoutOnce(
                () -> collection.find(
                        Filters.and(
                                Filters.eq("_id", UuidUtils.toHexString(id)),
                                Filters.eq("uid", uid.asString())
                        )).first());
        return Option.ofNullable(r).map(x -> ResourceFactory2.create(area, x));
    }

    @Override
    public Option<DjfsResource> find2(DjfsUid uid, DjfsResourceArea area, UUID id, boolean forUpdate) {
        return find2(uid, area, id);
    }

    @Override
    public Option<FileDjfsResource> find2File(DjfsUid uid, DjfsResourceArea area, UUID id) {
        MongoCollection<BsonDocument> collection = getCollection(uid, area);
        BsonDocument r = MongoUtil.retryTimeoutOnce(
                () -> collection.find(
                        Filters.and(
                                Filters.eq("_id", UuidUtils.toHexString(id)),
                                Filters.eq("uid", uid.asString()),
                                Filters.eq("type", "file")
                        )).first());
        return Option.ofNullable(r).map(MongoFileDjfsResourceParser::create);
    }

    @Override
    public Option<FolderDjfsResource> find2Folder(DjfsResourcePath path) {
        MongoCollection<BsonDocument> collection = getCollection(path);
        BsonDocument r = MongoUtil.retryTimeoutOnce(
                () -> collection.find(
                        Filters.and(
                                Filters.eq("_id", path.getMongoId()),
                                Filters.eq("uid", path.getUid().asString()),
                                Filters.eq("type", "dir")
                        )).first());
        return Option.ofNullable(r).map(x -> MongoFolderDjfsResourceParser.create(path.getArea(), x));
    }

    @Override
    public Option<DjfsResource> find(DjfsResourcePath path) {
        Option<MongoResource> r = getCollectionX(path).findOne(
                Filters.and(Filters.eq("uid", path.getUid().asString()), Filters.eq("_id", path.getMongoId())));
        return r.map(ResourceFactory::create);
    }

    @Override
    public Option<DjfsResource> find2(DjfsResourcePath path) {
        return find2(path.getUid(), path.getArea(), path.getPgId());
    }

    @Override
    public ListF<DjfsResource> find(DjfsResourceId resourceId) {
        /*
        todo: add parallel find

        ListF<MongoCollectionX<String, MongoResource>> collections = getCollectionsX(resourceId, DjfsResourceArea.ALL);
        Stream<Supplier<ListF<DjfsResource>>> stream = collections.stream().map(x ->
                () -> x.find(Filters.and(Filters.eq("uid", resourceId.getUid().asString()),
                        Filters.eq("data.file_id", resourceId.getFileId()))).map(ResourceFactory::create));

        return parallelStreamSupplier.supply(stream).flatMap(CollectionF::stream).collect(CollectorsF.toList());
        */

        ListF<DjfsResource> result = Cf.arrayList();

        ListF<MongoCollectionX<String, MongoResource>> collections = getCollectionsX(resourceId, DjfsResourceArea.ALL);
        for (MongoCollectionX<String, MongoResource> collection : collections) {
            result.addAll(collection.find(Filters.and(Filters.eq("uid", resourceId.getUid().asString()),
                    Filters.eq("data.file_id", resourceId.getFileId().getValue())))
                    .map(ResourceFactory::create)
            );
        }
        return result;
    }

    @Override
    public ListF<DjfsResource> find(DjfsResourceId resourceId, DjfsResourceArea area) {
        return getCollectionX(resourceId, area)
                .find(Filters.and(Filters.eq("uid", resourceId.getUid().asString()),
                        Filters.eq("data.file_id", resourceId.getFileId().getValue())))
                .map(ResourceFactory::create);
    }

    @Override
    public ListF<Tuple2<DjfsFileId, Option<Double>>> getAesthetics(DjfsUid uid, ListF<DjfsFileId> fileIds) {
        throw new NotImplementedException();
    }

    @Override
    public ListF<DjfsFileId> getFileIds(DjfsUid uid, int limit, Option<DjfsFileId> startingFileId) {
        throw new NotImplementedException();
    }

    @Override
    public ListF<DjfsFileId> getFileIdForFilesWithCoordinatesWithoutAesthetics(DjfsUid uid, int limit) {
        throw new NotImplementedException();
    }

    @Override
    public Option<Tuple2<DjfsResource, ListF<UUID>>> findWithParentIds(DjfsResourcePath path)
    {
        Option<DjfsResource> resource = find(path);
        if (!resource.isPresent()) {
            return Option.empty();
        }
        return Option.of(Tuple2.tuple(resource.get(), path.getMongoParentIds()));
    }

    /**
     * Looks for resources by resourceId in specified areas. In case of file_id collisions returns a resource with the lowest _id
     */
    @Override
    public ListF<Tuple2<DjfsResource, ListF<UUID>>> findWithParentIds(DjfsUid uid, ListF<String> fileIds,
            ListF<DjfsResourceArea> areasToSearch)
    {
        ListF<MongoCollectionX<String, MongoResource>> collections = getCollectionsX(uid, areasToSearch);

        Stream<Supplier<ListF<MongoResource>>> stream = collections.stream().map(x ->
                () -> x.find(Filters.and(Filters.eq("uid", uid.asString()),
                        Filters.in("data.file_id", fileIds))));

        return parallelStreamSupplier.supply(stream).flatMap(CollectionF::stream)
                .map(ResourceFactory::create)
                .map(x -> Tuple2.tuple(x, x.getPath().getMongoParentIds()))
                .collect(CollectorsF.toList());
    }

    @Override
    public ListF<FolderDjfsResource> find2ImmediateChildFoldersOrderByFileIdThenId(DjfsUid uid, DjfsResourceArea area,
            UUID parentId, Option<String> startingFileId, Option<UUID> startingId, int limit, boolean forUpdate)
    {
        MongoCollection<BsonDocument> collection = getCollection(uid, area);

        ListF<BsonDocument> r = MongoUtil.retryTimeoutOnce(() -> {
            final ListF<BsonDocument> result = Cf.arrayList();

            ListF<Bson> filters = Cf.list(
                    Filters.eq("uid", uid.asString()),
                    Filters.eq("parent", UuidUtils.toHexString(parentId)),
                    Filters.eq("type", "dir"));
            if (startingFileId.isPresent()) {
                if (startingId.isPresent()) {
                    filters = filters.plus(
                            Filters.or(
                                    Filters.gt("data.file_id", startingFileId.get()),
                                    Filters.and(
                                            Filters.eq("data.file_id", startingFileId.get()),
                                            Filters.gt("_id", startingId.map(UuidUtils::toHexString).get())
                                    )));
                } else {
                    filters = filters.plus(Filters.gt("data.file_id", startingFileId.get()));
                }
            }

            return collection.find(Filters.and(filters)).sort(Sorts.ascending("data.file_id", "_id")).limit(limit)
                    .into(result);
        });

        return r.map(x -> MongoFolderDjfsResourceParser.create(area, x));
    }

    @Override
    public ListF<FolderDjfsResource> find2ImmediateChildFoldersWithoutFileId(DjfsUid uid, DjfsResourceArea area,
            UUID parentId)
    {
        MongoCollection<BsonDocument> collection = getCollection(uid, area);

        ListF<BsonDocument> r = MongoUtil.retryTimeoutOnce(() -> {
            final ListF<BsonDocument> result = Cf.arrayList();
            return collection.find(Filters.and(Cf.list(
                    Filters.eq("uid", uid.asString()),
                    Filters.eq("parent", UuidUtils.toHexString(parentId)),
                    Filters.eq("type", "dir"),
                    Filters.exists("data.file_id", false)))).sort(Sorts.ascending("_id"))
                    .into(result);
        });

        return r.map(x -> MongoFolderDjfsResourceParser.create(area, x));
    }

    @Override
    public ListF<FileDjfsResource> find2ImmediateChildFilesOrderByFileIdThenId(DjfsUid uid, DjfsResourceArea area,
            UUID parentId, Option<String> startingFileId, Option<UUID> startingId, int limit, boolean forUpdate)
    {
        MongoCollection<BsonDocument> collection = getCollection(uid, area);

        ListF<BsonDocument> r = MongoUtil.retryTimeoutOnce(() -> {
            final ListF<BsonDocument> result = Cf.arrayList();

            ListF<Bson> filters = Cf.list(
                    Filters.eq("uid", uid.asString()),
                    Filters.eq("parent", UuidUtils.toHexString(parentId)),
                    Filters.eq("type", "file"));
            if (startingFileId.isPresent()) {
                if (startingId.isPresent()) {
                    filters = filters.plus(
                            Filters.or(
                                    Filters.gt("data.file_id", startingFileId.get()),
                                    Filters.and(
                                            Filters.eq("data.file_id", startingFileId.get()),
                                            Filters.gt("_id", startingId.map(UuidUtils::toHexString).get())
                                    )));
                } else {
                    filters = filters.plus(Filters.gt("data.file_id", startingFileId.get()));
                }
            }

            return collection.find(Filters.and(filters)).sort(Sorts.ascending("data.file_id", "_id")).limit(limit)
                    .into(result);
        });

        return r.map(MongoFileDjfsResourceParser::create);
    }

    @Override
    public Option<FileDjfsResource> find2AdditionalFileFor(FileDjfsResource file) {
        if (!file.isLivePhoto() || !file.getLiveVideoId().isPresent()) {
            return Option.empty();
        }
        return find2File(file.getUid(), DjfsResourceArea.ADDITIONAL_DATA, file.getLiveVideoId().get());
    }

    @Override
    public ListF<FilenamePreviewStidMimetypeVersionFileId> find2FilenamePreviewStidMimetypeVersionFid(
            DjfsUid uid, ListF<DjfsFileId> fileIds)
    {
        ListF<MongoCollection<BsonDocument>> collections = getCollections(uid, DjfsResourceArea.ALL);

        Stream<Supplier<ListF<BsonDocument>>> stream = collections.stream().map(x ->
                () -> {
                    final ListF<BsonDocument> result = Cf.arrayList();
                    return x.find(Filters.and(
                            Filters.eq("uid", uid.asString()), Filters.eq("type", "file"),
                            Filters.in("data.file_id", fileIds.map(DjfsFileId::getValue)))).into(result);
                });

        ListF<FileDjfsResource> resources = Cf.x(
                parallelStreamSupplier.supply(stream).flatMap(CollectionF::stream)
                        .map(MongoFileDjfsResourceParser::create).collect(Collectors.toList())
        );

        return resources.map(FilenamePreviewStidMimetypeVersionFileId::new);
    }

    public ListF<Tuple3<UUID, StidType, String>> findStids(DjfsShardInfo.Mongo shardInfo, DjfsResourceArea area,
            Option<UUID> start, Option<UUID> end, int limit)
    {
        MongoCollection<BsonDocument> collection = getCollection(shardInfo, area);
        ListF<Bson> filters = Cf.arrayList();
        filters.add(Filters.eq("type", "file"));
        if (start.isPresent()) {
            filters.add(Filters.gt("_id", UuidUtils.toHexString(start.get())));
        }
        if (end.isPresent()) {
            filters.add(Filters.lte("_id", UuidUtils.toHexString(end.get())));
        }

        Bson query = filters.length() > 1 ? Filters.and(filters) : filters.first();
        Bson projection = Projections.include("data.stids");
        Bson sort = Sorts.ascending("_id");

        final ListF<BsonDocument> result = Cf.arrayList();
        return MongoUtil.retryTimeoutOnce(() -> collection.withReadPreference(ReadPreference.secondaryPreferred())
                .find(query).projection(projection).sort(sort).limit(limit).into(result))
                .flatMap(x -> {
                    UUID id = UuidUtils.fromHex(x.getString("_id").getValue());
                    ListF<Tuple3<UUID, StidType, String>> r = Cf.arrayList();
                    if (!x.containsKey("data") || !x.getDocument("data").containsKey("stids")) {
                        return Cf.list();
                    }
                    for (BsonValue stid : x.getDocument("data").getArray("stids").getValues()) {
                        String type = ((BsonDocument) stid).getString("type").getValue();
                        if (Objects.equals(type, "file_mid")) {
                            r.add(Tuple3.tuple(id, StidType.FILE, ((BsonDocument) stid).getString("stid").getValue()));
                        } else if (Objects.equals(type, "digest_mid")) {
                            r.add(Tuple3
                                    .tuple(id, StidType.DIGEST, ((BsonDocument) stid).getString("stid").getValue()));
                        } else if (Objects.equals(type, "pmid")) {
                            r.add(Tuple3
                                    .tuple(id, StidType.PREVIEW, ((BsonDocument) stid).getString("stid").getValue()));
                        }
                    }
                    return r;
                });
    }

    @Override
    public ListF<DjfsResource> find2ImmediateChildren(DjfsResourcePath path, int limit) {
        throw new NotImplementedException();
    }

    @Override
    public long countImmediateChildren(DjfsResourcePath path) {
        throw new NotImplementedException();
    }

    @Override
    public ListF<DjfsResource> find2ImmediateChildren(DjfsResourcePath path, int offset, int limit, boolean ordered) {
        throw new NotImplementedException();
    }

    @Override
    public ListF<DjfsResource> find2ImmediateChildren(DjfsResource resource, int offset, int limit, boolean ordered) {
        throw new NotImplementedException();
    }

    @Override
    public ListF<DjfsResource> findAll(DjfsUid uid, DjfsResourceArea partition) {
        return getCollectionX(uid, partition).find(Filters.eq("uid", uid.asString())).map(ResourceFactory::create);
    }

    @Override
    public void checkAlreadyExistsException(Throwable e, DjfsResource resource) throws EntityAlreadyExistsException {
        if (ErrorCategory.fromErrorCode((((MongoWriteException) e).getError().getCode())) == ErrorCategory.DUPLICATE_KEY) {
            logger.warn("MongoDjfsResourceDao.insert(" + resource.getClass().getSimpleName() + ") handled exception for path "
                    + resource.getPath().toString() + " : ", e);
            throw new EntityAlreadyExistsException(UuidUtils.toHexString(resource.getId()), e);
        }
    }

    @Override
    public void insert(FolderDjfsResource resource) {
        try {
            getCollectionX(resource.getPath()).insertOne(ResourceFactory.toMongo(resource));
        } catch (MongoWriteException e) {
            checkAlreadyExistsException(e, resource);
            throw e;
        }
    }

    @Override
    public void insert(FileDjfsResource resource) {
        getCollectionX(resource.getPath()).insertOne(ResourceFactory.toMongo(resource));
    }

    @Override
    public void insert2(FolderDjfsResource resource) {
        BsonDocument document = MongoFolderDjfsResourceSerializer.serialize(resource);
        MongoCollection<BsonDocument> collection = getCollection(resource.getPath());
        try {
            MongoUtil.retryTimeoutOnce(() -> collection.insertOne(document));
        } catch (MongoWriteException e) {
            checkAlreadyExistsException(e, resource);
            throw e;
        }
    }

    @Override
    public void insert2(FileDjfsResource resource) {
        BsonDocument document = MongoFileDjfsResourceSerializer.serialize(resource);
        MongoCollection<BsonDocument> collection = getCollection(resource.getPath());
        try {
            MongoUtil.retryTimeoutOnce(() -> collection.insertOne(document));
        } catch (MongoWriteException e) {
            checkAlreadyExistsException(e, resource);
            throw e;
        }
    }

    @Override
    public void insert2AndLinkTo(FileDjfsResource file, FileDjfsResource link) {
        if (!file.isLivePhoto()) {
            throw new NotImplementedException("insert2AndLinkTo: file is not live photo");
        }
        if (link.getPath().getArea() != DjfsResourceArea.ADDITIONAL_DATA) {
            throw new NotImplementedException("insert2AndLinkTo: link is not in additional data");
        }

        BsonDocument document = MongoFileDjfsResourceSerializer.serialize(file);
        document.getDocument("data").append("live_video_id", new BsonString(UuidUtils.toHexString(link.getId())));
        MongoCollection<BsonDocument> collection = getCollection(file.getPath());
        try {
            MongoUtil.retryTimeoutOnce(() -> collection.insertOne(document));
        } catch (MongoWriteException e) {
            checkAlreadyExistsException(e, file);
            throw e;
        }
    }

    @Override
    public void changeParent(DjfsUid uid, UUID id, UUID newParentId) {
        throw new NotImplementedException();
    }

    @Override
    public void changeParentAndName(DjfsUid uid, UUID id, UUID newParentId, String name) {
        throw new NotImplementedException();
    }

    @Override
    public void setModificationTime(DjfsUid uid, UUID id, Instant time) {
        throw new NotImplementedException();
    }

    @Override
    public void setVersion(DjfsUid uid, UUID id, long version) {
        throw new NotImplementedException();
    }

    @Override
    public void setYarovayaMark(DjfsUid uid, UUID id) {
        throw new NotImplementedException();
    }

    @Override
    public void setTrashAppendTimeAndOriginalPath(DjfsUid uid, UUID id, String path, Instant trashAppendTime) {
        throw new NotImplementedException();
    }

    @Override
    public void setTrashAppendTime(DjfsUid uid, ListF<UUID> ids, Instant trashAppendTime) {
        throw new NotImplementedException();
    }

    @Override
    public void setArea(DjfsUid uid, ListF<UUID> ids, DjfsResourceArea area) {
        throw new NotImplementedException();
    }

    @Override
    public void setHiddenAppendTime(DjfsUid uid, UUID id, Instant hiddenAppendTime) {
        throw new NotImplementedException();
    }

    @Override
    public void setPublishedAndRemovePublic(DjfsUid uid, UUID id) {
        throw new NotImplementedException();
    }

    @Override
    public void addFileId(DjfsResourcePath path, DjfsFileId fileId) {
        getCollectionX(path).updateOne(
                Filters.and(
                        Filters.eq("_id", path.getMongoId()),
                        Filters.eq("uid", path.getUid().asString()),
                        Filters.exists("data.file_id", false)),
                Updates.set("data.file_id", fileId.getValue()));
    }

    @Override
    public void removeFileId(DjfsResourcePath path) {
        getCollectionX(path).updateOne(
                Filters.and(Filters.eq("_id", path.getMongoId()), Filters.eq("uid", path.getUid().asString())),
                Updates.unset("data.file_id"));
    }

    @Override
    public void setAesthetics(DjfsUid uid, UUID id, double aesthetics) {
        throw new NotImplementedException();
    }

    @Override
    public void setCoordinates(DjfsUid uid, UUID id, Coordinates coordinates) {
        throw new NotImplementedException();
    }

    @Override
    public void setAesthetics(DjfsUid uid, Tuple2List<DjfsFileId, Double> fileIdWithAesthetics, boolean skipExisting) {
        throw new NotImplementedException();
    }

    @Override
    public void setCoordinates(DjfsUid uid, Tuple2List<DjfsFileId, Coordinates> fileIdWithCoordinates,
            boolean skipExisting) {
        throw new NotImplementedException();
    }

    @Override
    public void setDimensionsWithAngle(DjfsUid uid, UUID id, Dimension dimension, int angle) {
        throw new NotImplementedException();
    }

    @Override
    public boolean hasAnyParentWithYarovayaMark(DjfsResourcePath path) {
        throw new NotImplementedException();
    }

    @Override
    public void delete(DjfsResourcePath path) {
        Bson filter = Filters.and(
                Filters.eq("_id", path.getMongoId()),
                Filters.eq("uid", path.getUid().asString()),
                Filters.eq("key", path.getPath()));
        getCollectionX(path).deleteOne(filter);
    }

    @Override
    public ListF<FileDjfsResource> getUserPhotosWithGeoCoordinates(DjfsUid uid) {
        throw new NotImplementedException();
    }

    @Override
    public long countAllFiles(DjfsUid uid) {
        throw new NotImplementedException();
    }

    @Override
    public ListF<DjfsResource> findByPaths(DjfsUid uid, ListF<DjfsResourcePath> paths) {
        throw new NotImplementedException("Not implemented for mongoDB");
    }

    private MongoCollection<BsonDocument> getCollection(MongoClient mongoClient, String name) {
        return mongoClient.getDatabase(name).getCollection(name, BsonDocument.class);
    }

    private MongoCollection<BsonDocument> getCollection(DjfsUid uid, DjfsResourceArea area) {
        MongoClient mongoClient = mongoShardResolver.resolve(uid);
        return getCollection(mongoClient, area.mongoCollectionName);
    }

    private ListF<MongoCollection<BsonDocument>> getCollections(DjfsUid uid, ListF<DjfsResourceArea> areas) {
        MongoClient mongoClient = mongoShardResolver.resolve(uid);
        return areas.map(x -> getCollection(mongoClient, x.mongoCollectionName));
    }

    private MongoCollection<BsonDocument> getCollection(DjfsShardInfo.Mongo shardInfo, DjfsResourceArea area) {
        MongoClient mongoClient = mongoShardResolver.resolve(shardInfo);
        return getCollection(mongoClient, area.mongoCollectionName);
    }

    private MongoCollection<BsonDocument> getCollection(DjfsUidSource uidSource, DjfsResourceArea area) {
        return getCollection(uidSource.getUid(), area);
    }

    private MongoCollection<BsonDocument> getCollection(DjfsResourcePath path) {
        return getCollection(path.getUid(), path.getArea());
    }

    private MongoCollectionX<String, MongoResource> getCollectionX(MongoClient mongoClient, String name) {
        return new MongoCollectionX<>(
                mongoClient.getDatabase(name).getCollection(name, BsonDocument.class),
                MongoResource.B);
    }

    private MongoCollectionX<String, MongoResource> getCollectionX(DjfsUidSource uidSource, DjfsResourceArea area) {
        return getCollectionX(uidSource.getUid(), area);
    }

    private MongoCollectionX<String, MongoResource> getCollectionX(DjfsUid uid, DjfsResourceArea area) {
        MongoClient mongoClient = mongoShardResolver.resolve(uid);
        return getCollectionX(mongoClient, area.mongoCollectionName);
    }

    private MongoCollectionX<String, MongoResource> getCollectionX(DjfsResourcePath path) {
        return getCollectionX(path.getUid(), path.getArea());
    }

    private ListF<MongoCollectionX<String, MongoResource>> getCollectionsX(DjfsUidSource uidSource,
            ListF<DjfsResourceArea> areas)
    {
        return getCollectionsX(uidSource.getUid(), areas);
    }

    private ListF<MongoCollectionX<String, MongoResource>> getCollectionsX(DjfsUid uid, ListF<DjfsResourceArea> areas) {
        MongoClient mongoClient = mongoShardResolver.resolve(uid);
        return areas.map(x -> getCollectionX(mongoClient, x.mongoCollectionName));
    }

    private ListF<MongoCollectionX<String, MongoResource>> getCollectionsX(DjfsUid uid) {
        return getCollectionsX(uid, Cf.wrap(DjfsResourceArea.values()));
    }
}
