package ru.yandex.chemodan.app.djfs.core.filesystem;

import java.nio.charset.StandardCharsets;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.Map;
import java.util.Objects;
import java.util.UUID;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;

import com.fasterxml.jackson.databind.ObjectMapper;
import org.apache.commons.lang3.ArrayUtils;
import org.joda.time.Instant;
import org.postgresql.geometric.PGpoint;
import org.postgresql.util.PSQLException;
import org.postgresql.util.ServerErrorMessage;
import org.springframework.dao.DataIntegrityViolationException;
import org.springframework.jdbc.core.RowMapper;
import org.springframework.jdbc.core.SingleColumnRowMapper;
import org.springframework.jdbc.core.namedparam.MapSqlParameterSource;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.bolts.collection.Tuple3;
import ru.yandex.bolts.function.Function3;
import ru.yandex.bolts.internal.NotImplementedException;
import ru.yandex.chemodan.app.djfs.core.FilenamePreviewStidMimetypeVersionFileId;
import ru.yandex.chemodan.app.djfs.core.db.DjfsShardInfo;
import ru.yandex.chemodan.app.djfs.core.db.EntityAlreadyExistsException;
import ru.yandex.chemodan.app.djfs.core.db.pg.PgArray;
import ru.yandex.chemodan.app.djfs.core.db.pg.PgShardedDao;
import ru.yandex.chemodan.app.djfs.core.db.pg.PgShardedDaoContext;
import ru.yandex.chemodan.app.djfs.core.db.pg.ResultSetUtils;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.AntiVirusScanStatus;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.DjfsFileId;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.DjfsResource;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.DjfsResourceArea;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.DjfsResourceId;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.DjfsResourcePath;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.FileDjfsResource;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.FolderDjfsResource;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.FolderType;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.MediaType;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.PhotosliceAlbumType;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.StidType;
import ru.yandex.chemodan.app.djfs.core.user.DjfsUid;
import ru.yandex.chemodan.app.djfs.core.util.InstantUtils;
import ru.yandex.chemodan.app.djfs.core.util.UuidUtils;
import ru.yandex.chemodan.http.YandexCloudRequestIdHolder;
import ru.yandex.commune.dynproperties.DynamicProperty;
import ru.yandex.commune.json.jackson.ObjectMapperX;
import ru.yandex.misc.ExceptionUtils;
import ru.yandex.misc.codec.Hex;
import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.misc.db.masterSlave.MasterSlaveContextHolder;
import ru.yandex.misc.db.masterSlave.MasterSlavePolicy;
import ru.yandex.misc.db.postgres.PSQLState;
import ru.yandex.misc.geo.Coordinates;
import ru.yandex.misc.image.Dimension;
import ru.yandex.misc.lang.CharsetUtils;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

/**
 * @author eoshch
 */
public class PgDjfsResourceDao extends PgShardedDao implements DjfsResourceDao {
    private static final Logger logger = LoggerFactory.getLogger(PgDjfsResourceDao.class);
    private static final ObjectMapperX mapper = new ObjectMapperX(new ObjectMapper());

    private final DynamicProperty<Integer> bulkInfoMaxInnerQueries =
            new DynamicProperty<>("bulk-info-max-inner-queries", 10);

    private final DynamicProperty<Boolean> bulkInfoParallelInnerQueries =
            new DynamicProperty<>("bulk-info-parallel-inner-queries", true);

    private final DynamicProperty<ListF<String>> photoMediaTypes =
            new DynamicProperty<>("disk-djfs-albums-photo-media-types", Cf.list("image"));

    private final String[] RESOURCE_FIELDS_ARRAY = {"id", "uid", "fid", "short_url", "visible", "blocked",
            "custom_properties", "yarovaya_mark", "public", "path_before_remove", "date_removed", "version", "symlink",
            "modify_uid", "public_hash", "custom_setprop_fields", "name", "parent_fid", "date_modified",
            "download_counter", "folder_url", "date_uploaded", "published", "date_created", "date_hidden_data",
            "area"};

    private final String[] FILE_ONLY_FIELDS_ARRAY = {"is_live_photo", "source_uid", "date_exif", "source", "storage_id",
            "media_type", "ext_aesthetics", "mime_type", "photoslice_album_type", "albums_exclusions",
            "ext_coordinates"};

    private final String[] STORAGE_FILE_FIELDS_ARRAY = {"storage_id", "stid", "digest_stid", "preview_stid",
            "date_origin", "size", "md5_sum", "sha256_sum", "av_scan_status", "height", "width", "angle", "video_data"};

    private final String[] FOLDER_ONLY_FIELDS_ARRAY = {"folder_type", "folders_count", "files_count"};

    private final String[] FOLDER_FIELDS_ARRAY = ArrayUtils.addAll(RESOURCE_FIELDS_ARRAY, FOLDER_ONLY_FIELDS_ARRAY);

    private final String FOLDER_FIELDS_PREFIXED = String.join(", ", Cf.wrap(FOLDER_FIELDS_ARRAY)
            .map(x -> "folders." + x));

    private final String[] FILE_FIELDS_ARRAY = ArrayUtils.addAll(RESOURCE_FIELDS_ARRAY, FILE_ONLY_FIELDS_ARRAY);

    private final String FILE_FIELDS_PREFIXED = String.join(", ", Cf.wrap(FILE_FIELDS_ARRAY).map(x -> "files." + x));

    private final String STORAGE_FILE_FIELDS_PREFIXED = String.join(", ", Cf.wrap(STORAGE_FILE_FIELDS_ARRAY)
            .map(x -> "storage_files." + x));

    private final String FILE_FIELDS_PREFIXED_WITH_ALIASES = String.join(", ",
            Cf.wrap(FILE_FIELDS_ARRAY).map(x -> "files." + x + " as files_" + x)
    );
    private final String STORAGE_FILE_FIELDS_PREFIXED_WITH_ALIASES = String.join(", ",
            Cf.wrap(STORAGE_FILE_FIELDS_ARRAY).map(x -> "storage_files." + x + " as files_" + x)
    );
    private final String FOLDER_FIELDS_PREFIXED_WITH_ALIASES = String.join(", ",
            Cf.wrap(FOLDER_FIELDS_ARRAY).map(x -> "folders." + x + " as folders_" + x)
    );

    private final String FOLDER_NULLS_WITH_ALIASES = String.join(", ",
            Cf.wrap(FOLDER_FIELDS_ARRAY).map(x -> "null as folders_" + x)
    );
    private final String FILE_NULLS_WITH_ALIASES = String.join(", ",
            Cf.wrap(FILE_FIELDS_ARRAY).map(x -> "null as files_" + x)
    );
    private final String STORAGE_FILE_NULLS_WITH_ALIASES = String.join(", ",
            Cf.wrap(STORAGE_FILE_FIELDS_ARRAY).map(x -> "null as storage_files_" + x)
    );
    private final static Function3<ResultSet, Integer, String, FileDjfsResource> fileMapperWithPrefix =
            (rs, rowNum, prefix) -> {
                try {
                    FileDjfsResource.Builder builder = FileDjfsResource.builder();
                    builder.id(ResultSetUtils.getUuid(rs, prefix + "fid"));
                    builder.parentId(ResultSetUtils.getUuidO(rs, prefix + "parent_fid"));

                    Option<String> fileId = Option.ofNullable(rs.getBytes(prefix + "id")).map(Hex::encode);
                    builder.fileId(fileId.map(DjfsFileId::cons));

                    builder.path(DjfsResourcePath.cons(rs.getLong(prefix + "uid"), rs.getString("path")));
                    builder.modifyUid(ResultSetUtils.getLongO(rs, prefix + "modify_uid").map(DjfsUid::cons));
                    builder.version(ResultSetUtils.getLongO(rs, prefix + "version"));

                    builder.uploadTime(ResultSetUtils.getInstantO(rs, prefix + "date_uploaded"));
                    builder.trashAppendTime(ResultSetUtils.getInstantO(rs, prefix + "date_removed"));
                    builder.hiddenAppendTime(ResultSetUtils.getInstantO(rs, prefix + "date_hidden_data"));

                    builder.trashAppendOriginalPath(rs.getString(prefix + "path_before_remove"));

                    builder.isVisible(rs.getBoolean(prefix + "visible"));
                    builder.isPublic(rs.getBoolean(prefix + "public"));
                    builder.isBlocked(rs.getBoolean(prefix + "blocked"));
                    builder.isPublished(rs.getBoolean(prefix + "published"));
                    builder.hasYarovayaMark(rs.getBoolean(prefix + "yarovaya_mark"));

                    builder.publicHash(
                            Option.ofNullable(rs.getBytes(prefix + "public_hash")).map(CharsetUtils::decodeUtf8));
                    builder.shortUrl(rs.getString(prefix + "short_url"));
                    builder.symlink(rs.getString(prefix + "symlink"));
                    builder.folderUrl(rs.getString(prefix + "folder_url"));
                    // todo: long?
                    builder.downloadCounter(ResultSetUtils.getIntO(rs, prefix + "download_counter"));
                    builder.customProperties(rs.getString(prefix + "custom_properties"));

                    builder.size(rs.getLong(prefix + "size"));
                    builder.md5(rs.getString(prefix + "md5_sum").replace("-", ""));
                    builder.sha256(Hex.encode((rs.getBytes(prefix + "sha256_sum"))));
                    builder.hid(rs.getString(prefix + "storage_id").replace("-", ""));

                    builder.fileStid(rs.getString(prefix + "stid"));
                    builder.digestStid(rs.getString(prefix + "digest_stid"));
                    builder.previewStid(rs.getString(prefix + "preview_stid"));

                    builder.creationTime(ResultSetUtils.getInstant(rs, prefix + "date_created"));
                    builder.modificationTime(ResultSetUtils.getInstant(rs, prefix + "date_modified"));
                    builder.exifTime(ResultSetUtils.getInstantO(rs, prefix + "date_exif"));

                    builder.aesthetics(ResultSetUtils.getDoubleO(rs, prefix + "ext_aesthetics"));

                    builder.photosliceAlbumType(
                            PhotosliceAlbumType.R.valueOfO(rs.getString(prefix + "photoslice_album_type")));
                    builder.albumsExclusions(
                            ResultSetUtils.getArrayO(rs, prefix + "albums_exclusions", String.class).map(Cf::list));

                    builder.antiVirusScanStatus(
                            AntiVirusScanStatus.R.valueOfO(rs.getString(prefix + "av_scan_status")));
                    builder.source(rs.getString(prefix + "source"));
                    builder.mimetype(rs.getString(prefix + "mime_type"));
                    builder.mediaType(MediaType.R.valueOfO(rs.getString(prefix + "media_type")));

                    builder.height(ResultSetUtils.getIntO(rs, prefix + "height"));
                    builder.width(ResultSetUtils.getIntO(rs, prefix + "width"));
                    builder.angle(ResultSetUtils.getIntO(rs, prefix + "angle"));
                    builder.videoInfo(rs.getString(prefix + "video_data"));
                    builder.albumsExclusions(ResultSetUtils.getStringArrayO(rs, prefix + "albums_exclusions"));

                    builder.coordinates(ResultSetUtils.getCoordinatesO(rs, prefix + "ext_coordinates"));

                    builder.area(DjfsResourceArea.R.fromValueO(rs.getString(prefix + "area")));

                    String rawCustomSetpropFields = rs.getString(prefix + "custom_setprop_fields");
                    if (!StringUtils.isEmpty(rawCustomSetpropFields)) {
                        //noinspection unchecked
                        MapF<String, String> customSetpropFields =
                                Cf.wrap(mapper.readValue(Map.class, rawCustomSetpropFields));
                        customSetpropFields.removeTs("file_id_zipped");
                        customSetpropFields.removeTs("total_results_count");
                        customSetpropFields.removeTs("revision");
                        customSetpropFields.removeTs("preview_sizes");
                        customSetpropFields.removeTs("page_blocked_items_num");
                        builder.fotkiTags(customSetpropFields.removeO("fotki_tags"));
                        builder.externalUrl(customSetpropFields.removeO("external_url"));
                        builder.fsSymbolicLink(customSetpropFields.removeO("fs_symbolic_link"));
                        builder.externalProperties(customSetpropFields);
                    }

                    // false for NULL
                    builder.isLivePhoto(rs.getBoolean(prefix + "is_live_photo"));

                    return builder.build();
                } catch (SQLException e) {
                    throw ExceptionUtils.translate(e);
                }
            };

    private final static RowMapper<FileDjfsResource> fileMapper =
            (rs, rowNum) -> fileMapperWithPrefix.apply(rs, rowNum, "");

    private final static Function3<ResultSet, Integer, String, FolderDjfsResource> folderMapperWithPrefix =
            (rs, rowNum, prefix) -> {
                try {
                    FolderDjfsResource.Builder builder = FolderDjfsResource.builder();

                    if (DjfsResourcePath.ROOT.equals(rs.getString("path"))) {
                        // todo: get area from method call
                        builder.path(DjfsResourcePath.root(rs.getLong(prefix + "uid"), DjfsResourceArea.DISK));
                    } else {
                        builder.path(DjfsResourcePath.cons(rs.getLong(prefix + "uid"), rs.getString("path")));
                    }

                    builder.id(ResultSetUtils.getUuid(rs, prefix + "fid"));
                    builder.parentId(ResultSetUtils.getUuidO(rs, prefix + "parent_fid"));
                    builder.fileId(
                            Option.ofNullable(rs.getBytes(prefix + "id")).map(Hex::encode).map(DjfsFileId::cons));

                    builder.modifyUid(ResultSetUtils.getLongO(rs, prefix + "modify_uid").map(DjfsUid::cons));
                    builder.version(ResultSetUtils.getLongO(rs, prefix + "version"));

                    builder.creationTime(ResultSetUtils.getInstantO(rs, prefix + "date_created"));
                    builder.modificationTime(ResultSetUtils.getInstantO(rs, prefix + "date_modified"));
                    builder.uploadTime(ResultSetUtils.getInstantO(rs, prefix + "date_uploaded"));
                    builder.trashAppendTime(ResultSetUtils.getInstantO(rs, prefix + "date_removed"));
                    builder.hiddenAppendTime(ResultSetUtils.getInstantO(rs, prefix + "date_hidden_data"));

                    builder.trashAppendOriginalPath(rs.getString(prefix + "path_before_remove"));

                    builder.isVisible(rs.getBoolean(prefix + "visible"));
                    builder.isPublic(rs.getBoolean(prefix + "public"));
                    builder.isBlocked(rs.getBoolean(prefix + "blocked"));
                    builder.isPublished(rs.getBoolean(prefix + "published"));
                    builder.hasYarovayaMark(rs.getBoolean(prefix + "yarovaya_mark"));

                    builder.publicHash(
                            Option.ofNullable(rs.getBytes(prefix + "public_hash")).map(CharsetUtils::decodeUtf8));
                    builder.shortUrl(rs.getString(prefix + "short_url"));
                    builder.symlink(rs.getString(prefix + "symlink"));

                    builder.folderUrl(rs.getString(prefix + "folder_url"));
                    builder.folderType(FolderType.R.fromValueO(rs.getString(prefix + "folder_type")));
                    // todo: long?
                    builder.downloadCounter(ResultSetUtils.getIntO(rs, prefix + "download_counter"));
                    builder.customProperties(rs.getString(prefix + "custom_properties"));

                    builder.area(DjfsResourceArea.R.fromValueO(rs.getString(prefix + "area")));

                    String rawCustomSetpropFields = rs.getString(prefix + "custom_setprop_fields");
                    if (!StringUtils.isEmpty(rawCustomSetpropFields)) {
                        //noinspection unchecked
                        MapF<String, String> customSetpropFields =
                                Cf.wrap(mapper.readValue(Map.class, rawCustomSetpropFields));
                        customSetpropFields.removeTs("file_id_zipped");
                        customSetpropFields.removeTs("total_results_count");
                        customSetpropFields.removeTs("revision");
                        customSetpropFields.removeTs("folder_id");
                        customSetpropFields.removeTs("lenta_block_id");
                        builder.lastImportTime(customSetpropFields.removeO("last_import_time").cast(Integer.class)
                                .map(InstantUtils::fromSeconds));
                        builder.externalProperties(customSetpropFields);
                    }

                    return builder.build();
                } catch (SQLException e) {
                    throw ExceptionUtils.translate(e);
                }
            };

    private final static RowMapper<FolderDjfsResource> folderMapper =
            (rs, rowNum) -> folderMapperWithPrefix.apply(rs, rowNum, "");

    private final static RowMapper<DjfsResource> resourceMapper = (rs, rowNum) -> {
        String type = rs.getString("type");
        if (type.equals("file")) {
            FileDjfsResource file = fileMapperWithPrefix.apply(rs, rowNum, "files_");
            return file;
        } else if (type.equals("dir")) {
            FolderDjfsResource folder = folderMapperWithPrefix.apply(rs, rowNum, "folders_");
            return folder;
        }
        throw new NotImplementedException();
    };


    private final static RowMapper<Tuple2<DjfsResource, ListF<UUID>>> resourceMapperWithParentIds = (rs, rowNum) -> {
        ListF<UUID> parentFids = Cf.wrap((UUID[]) rs.getArray("parent_fids").getArray());
        String type = rs.getString("type");

        if (type.equals("file")) {
            FileDjfsResource file = fileMapperWithPrefix.apply(rs, rowNum, "files_");
            return Tuple2.tuple(file, parentFids);
        } else if (type.equals("dir")) {
            FolderDjfsResource folder = folderMapperWithPrefix.apply(rs, rowNum, "folders_");
            return Tuple2.tuple(folder, parentFids);
        }

        throw new NotImplementedException();
    };

    private final ExecutorService bulkInfoInnerExecutor;

    public PgDjfsResourceDao(PgShardedDaoContext context, int bulkInfoInnerThreads) {
        super(context);
        this.bulkInfoInnerExecutor = new ThreadPoolExecutor(bulkInfoInnerThreads, bulkInfoInnerThreads,
                0L, TimeUnit.MILLISECONDS, new LinkedBlockingQueue<>(bulkInfoInnerThreads),
                new ThreadPoolExecutor.CallerRunsPolicy());
    }

    @Override
    public Option<DjfsResource> find(DjfsUid uid, DjfsResourceArea area, UUID id, boolean forUpdate) {
        // todo: optimize: parallel queries?

        String s = collectStats(uid) + " SELECT " + FILE_FIELDS_PREFIXED + ", " + STORAGE_FILE_FIELDS_PREFIXED + ","
                + " (SELECT path FROM code.fid_to_path(files.fid, files.uid)) as path"
                + " FROM disk.files files"
                + " JOIN disk.storage_files storage_files ON files.storage_id = storage_files.storage_id"
                + " WHERE files.uid = :uid AND files.fid = :fid";

        Map<String, Object> parameters = Cf.map("uid", uid.asLong(), "fid", id);

        Option<FileDjfsResource> file = jdbcTemplate(uid).queryForOption(s, fileMapper, parameters);

        if (file.isPresent()) {
            return file.cast();
        }

        String q = collectStats(uid) + " SELECT " + FOLDER_FIELDS_PREFIXED + ","
                + " (SELECT path FROM code.fid_to_path(folders.fid, folders.uid)) as path"
                + " FROM disk.folders folders"
                + " WHERE folders.uid = :uid AND folders.fid = :fid";

        if (forUpdate) {
            q += " FOR UPDATE";
        }

        Option<FolderDjfsResource> folder = jdbcTemplate(uid).queryForOption(q, folderMapper, parameters);

        return folder.cast();
    }

    @Override
    public Option<DjfsResource> find(DjfsUid uid, DjfsResourceArea area, UUID id) {
        return find(uid, area, id, false);
    }

    @Override
    public Option<DjfsResource> find2(DjfsUid uid, DjfsResourceArea area, UUID id, boolean forUpdate) {
        return find(uid, area, id, forUpdate);
    }

    @Override
    public Option<DjfsResource> find2(DjfsUid uid, DjfsResourceArea area, UUID id) {
        return find(uid, area, id);
    }

    @Override
    public Option<FileDjfsResource> find2File(DjfsUid uid, DjfsResourceArea area, UUID id) {
        String s = collectStats(uid) + " SELECT " + FILE_FIELDS_PREFIXED + ", " + STORAGE_FILE_FIELDS_PREFIXED + ","
                + " (SELECT path FROM code.fid_to_path(files.fid, files.uid)) as path"
                + " FROM disk.files files"
                + " JOIN disk.storage_files storage_files ON files.storage_id = storage_files.storage_id"
                + " WHERE files.uid = :uid AND files.fid = :fid";

        Map<String, Object> parameters = Cf.map("uid", uid.asLong(), "fid", id);

        return jdbcTemplate(uid).queryForOption(s, fileMapper, parameters);
    }

    @Override
    public Option<FolderDjfsResource> find2Folder(DjfsResourcePath path) {
        String q = collectStats(path) + " SELECT " + FOLDER_FIELDS_PREFIXED + ", :path as path"
                + " FROM disk.folders folders"
                + " JOIN (SELECT fid FROM code.path_to_fid(:path, :uid)) path_to_fid USING (fid)"
                + " WHERE folders.uid = :uid";

        Map<String, Object> parameters = Cf.map("uid", path.getUid(), "path", path.getPath());

        return jdbcTemplate(path).queryForOption(q, folderMapper, parameters);
    }

    @Override
    public Option<DjfsResource> find(DjfsResourcePath path) {
        return findWithParentIds(path).map(Tuple2::get1);
    }

    @Override
    public Option<DjfsResource> find2(DjfsResourcePath path) {
        return find(path);
    }

    @Override
    public ListF<DjfsResource> find(DjfsResourceId resourceId) {
        Map<String, Object> parameters = Cf.map(
                "uid", resourceId.getUid(),
                "file_id", Hex.decode(resourceId.getFileId().getValue()));

        String s = collectStats(resourceId) + " SELECT " + FILE_FIELDS_PREFIXED + ", " + STORAGE_FILE_FIELDS_PREFIXED
                + ", path.path "
                + " FROM disk.files files JOIN disk.storage_files storage_files USING (storage_id),"
                + " LATERAL(SELECT path FROM code.fid_to_path(files.fid, files.uid)) as path"
                + " WHERE files.uid = :uid AND files.id = :file_id";

        ListF<FileDjfsResource> files = jdbcTemplate(resourceId).query(s, fileMapper, parameters);

        String q = collectStats(resourceId) + " SELECT " + FOLDER_FIELDS_PREFIXED + ", path.path"
                + " FROM disk.folders folders,"
                + " LATERAL(SELECT path FROM code.fid_to_path(folders.fid, folders.uid)) as path"
                + " WHERE folders.uid = :uid AND folders.id = :file_id";

        ListF<FolderDjfsResource> folders = jdbcTemplate(resourceId).query(q, folderMapper, parameters);

        return files.<DjfsResource>cast().plus(folders.cast());
    }

    @Override
    public ListF<DjfsResource> find(DjfsResourceId resourceId, DjfsResourceArea area) {
        throw new NotImplementedException();
    }

    @Override
    public ListF<Tuple2<DjfsFileId, Option<Double>>> getAesthetics(DjfsUid uid, ListF<DjfsFileId> fileIds) {
        String sql = collectStats(uid) + " SELECT id,ext_aesthetics FROM disk.files WHERE uid = :uid AND id IN (:ids)";
        MapF<String, Object> params = Cf.map(
                "uid", uid,
                "ids", fileIds.map(DjfsFileId::getValue).map(Hex::decode)
        );
        return jdbcTemplate(uid).query(sql, (rs, rowNum) -> {
            DjfsFileId fileId = DjfsFileId.cons(Hex.encode(rs.getBytes("id")));
            Option<Double> aesthetics = ResultSetUtils.getDoubleO(rs, "ext_aesthetics");
            return Tuple2.tuple(fileId, aesthetics);
        }, params);
    }

    @Override
    public ListF<DjfsFileId> getFileIds(DjfsUid uid, int limit, Option<DjfsFileId> startingFileId) {
        Tuple2List<String, Object> params = Tuple2List.fromPairs(
                "uid", uid,
                "limit", limit
        );

        String sql = collectStats(uid) + " SELECT id FROM disk.files WHERE uid = :uid";
        if (startingFileId.isPresent()) {
            sql += " AND id >= :id";
            params.add("id", Hex.decode(startingFileId.get().getValue()));
        }
        sql += " ORDER BY id LIMIT :limit";

        return jdbcTemplate(uid).query(sql, (rs, rowNum) -> DjfsFileId.cons(Hex.encode(rs.getBytes("id"))),
                params.toMap());
    }

    @Override
    public ListF<DjfsFileId> getFileIdForFilesWithCoordinatesWithoutAesthetics(DjfsUid uid, int limit) {
        Tuple2List<String, Object> params = Tuple2List.fromPairs(
                "uid", uid,
                "limit", limit
        );

        String sql = collectStats(uid)
                + " SELECT id FROM disk.files"
                + " WHERE uid = :uid AND ext_coordinates IS NOT NULL AND ext_aesthetics IS NULL"
                + " LIMIT :limit";

        return jdbcTemplate(uid).query(sql, (rs, rowNum) -> DjfsFileId.cons(Hex.encode(rs.getBytes("id"))),
                params.toMap());
    }

    @Override
    public Option<Tuple2<DjfsResource, ListF<UUID>>> findWithParentIds(DjfsResourcePath path)
    {
        if (path.isRoot()) {
            String q = collectStats(path) + " SELECT " + FOLDER_FIELDS_PREFIXED + ", '/' as path"
                    + " FROM disk.folders folders"
                    + " WHERE folders.uid = :uid AND folders.parent_fid IS NULL AND folders.name = ''";

            MapF<String, Object> parameters = Cf.map(
                    "uid", path.getUid()
            );

            ListF<FolderDjfsResource> resources = jdbcTemplate(path.getUid()).query(q, folderMapper, parameters);

            if (resources.isEmpty()) {
                return Option.empty();
            }

            return Option.of(Tuple2.tuple(resources.first(), Cf.arrayList()));
        }

        String sqlQueryRootPart = "WITH r AS ("
                + "     SELECT"
                + "         1 AS idx,"
                + "         dir.name,"
                + "         dir.parent_fid,"
                + "         dir.fid,"
                + "         ARRAY[dir.fid] AS parent_fids"
                + "     FROM"
                + "         disk.folders dir"
                + "     WHERE"
                + "         dir.uid = :uid"
                + "         AND dir.parent_fid IS NULL"
                + "         AND dir.name = ''"
                + " ) ";

        String sqlQueryRecursivePart = "WITH RECURSIVE r AS ("
                + "     SELECT"
                + "         1 AS idx,"
                + "         dir.name,"
                + "         dir.parent_fid,"
                + "         dir.fid,"
                + "         ARRAY[dir.fid] AS parent_fids"
                + "     FROM"
                + "         disk.folders dir"
                + "     WHERE"
                + "         dir.uid = :uid"
                + "         AND dir.parent_fid IS NULL"
                + "         AND dir.name = ''"
                + " UNION"
                + "     SELECT"
                + "         idx + 1,"
                + "         child.name,"
                + "         child.parent_fid,"
                + "         child.fid,"
                + "         parent.parent_fids || child.fid AS parent_fids"
                + "     FROM"
                + "         r parent join disk.folders child on parent.fid = child.parent_fid"
                + "     WHERE"
                + "         uid = :uid"
                + "         AND child.name = (ARRAY[ :dir_names ])[idx]"
                + " ) ";

        String sqlQueryEnding = " SELECT "
                + FILE_FIELDS_PREFIXED_WITH_ALIASES + ", " + STORAGE_FILE_FIELDS_PREFIXED_WITH_ALIASES + ", "
                + FOLDER_NULLS_WITH_ALIASES + ", "
                + "     (SELECT parent_fids FROM r WHERE idx=:parents_count LIMIT 1) as parent_fids,"
                + "     :path as path, 'file' as type"
                + " FROM"
                + "     disk.files files JOIN disk.storage_files storage_files USING (storage_id)"
                + " WHERE"
                + "     files.uid = :uid"
                + "     AND files.parent_fid = (SELECT fid FROM r WHERE idx=:parents_count LIMIT 1)"
                + "     AND files.name = :resource_name"
                + " UNION ALL"
                + " SELECT "
                + FILE_NULLS_WITH_ALIASES + ", " + STORAGE_FILE_NULLS_WITH_ALIASES + ", "
                + FOLDER_FIELDS_PREFIXED_WITH_ALIASES + ", "
                + "     (SELECT parent_fids FROM r WHERE idx=:parents_count LIMIT 1) as parent_fids,"
                + "     :path as path, 'dir' as type"
                + " FROM"
                + "     disk.folders folders"
                + " WHERE"
                + "     folders.uid = :uid"
                + "     AND folders.parent_fid = (SELECT fid FROM r WHERE idx=:parents_count LIMIT 1)"
                + "     AND folders.name = :resource_name";

        String sqlQuery = collectStats(path)
                + (path.isAreaRoot() ? sqlQueryRootPart + sqlQueryEnding : sqlQueryRecursivePart + sqlQueryEnding);
        ListF<String> parentDirNames = path.getAllParents().filter(x -> !x.isRoot()).map(DjfsResourcePath::getName);

        MapF<String, Object> parameters = Tuple2List.<String, Object>fromPairs(
                "uid", path.getUid(),
                "dir_names", parentDirNames,
                "resource_name", path.getName(),
                "path", path.getPath(),
                "parents_count", parentDirNames.length() + 1
        ).toMap();

        ListF<Tuple2<DjfsResource, ListF<UUID>>> resources =
                jdbcTemplate(path.getUid()).query(sqlQuery, resourceMapperWithParentIds, parameters);

        if (resources.isNotEmpty()) {
            return Option.of(resources.get(0));
        }

        return Option.empty();
    }

    private ListF<DjfsResource> findByNamesInOneFolder(
            DjfsUid uid, String parentPath, ListF<String> resourceNames, boolean uniqueOnly)
    {
        MapF<String, Object> parameters = Cf.hashMap();
        parameters.put("uid", uid.asLong());
        parameters.put("parent_path", parentPath.endsWith(DjfsResourcePath.DELIMITER)
                ? parentPath
                : parentPath + DjfsResourcePath.DELIMITER);
        parameters.put("names", resourceNames);

        String query = collectStats(uid) + " SELECT "
                + FILE_FIELDS_PREFIXED_WITH_ALIASES + ", " + STORAGE_FILE_FIELDS_PREFIXED_WITH_ALIASES + ", "
                + FOLDER_NULLS_WITH_ALIASES + ", "
                + "     :parent_path || name as path, 'file' as type"
                + " FROM"
                + "     disk.files files JOIN disk.storage_files storage_files USING (storage_id)"
                + " WHERE"
                + "     files.uid = :uid"
                + "     AND files.parent_fid = (SELECT fid FROM code.path_to_fid(:parent_path, :uid))"
                + "     AND files.name IN (:names)"
                + " UNION ALL"
                + " SELECT "
                + FILE_NULLS_WITH_ALIASES + ", " + STORAGE_FILE_NULLS_WITH_ALIASES + ", "
                + FOLDER_FIELDS_PREFIXED_WITH_ALIASES + ", "
                + "     :parent_path || name as path, 'dir' as type"
                + " FROM"
                + "     disk.folders folders"
                + " WHERE"
                + "     folders.uid = :uid"
                + "     AND folders.parent_fid = (SELECT fid FROM code.path_to_fid(:parent_path, :uid))"
                + "     AND folders.name IN (:names)";

        ListF<DjfsResource> uniqueResources = jdbcTemplate(uid).query(query, resourceMapper, parameters);

        if (uniqueOnly) {
            return uniqueResources;
        }
        // support duplicating resourceNames
        MapF<String, DjfsResource> resourcesByName = uniqueResources
                .toMap(resource -> resource.getPath().getName(), resource -> resource);

        return resourceNames.flatMap(resourcesByName::getO);
    }

    @Override
    public ListF<DjfsResource> findByPaths(DjfsUid uid,
            ListF<DjfsResourcePath> paths)
    {

        if (paths.isEmpty()) {
            return Cf.list();
        }

        ListF<String> rawPaths = paths.map(DjfsResourcePath::getPath);

        MapF<String, ListF<String>> groupedPaths = paths.groupBy(path -> path.getParent().getPath())
                .mapValues(pathsInOneFolder -> pathsInOneFolder.map(DjfsResourcePath::getName));

        /*
            if we have more then bulkInfoMaxInnerQueries parent folders:
                use sophisticated general query below
            otherwise for one parent folder:
                use findByNamesInOneFolder in same thread
            otherwise:
                run dedicated queries in executor or same thread, regarding bulkInfoParallelInnerQueries
        */

        if (groupedPaths.size() <= bulkInfoMaxInnerQueries.get()) {
            if (groupedPaths.size() == 1) {
                String parentPath = groupedPaths.keys().single();
                ListF<String> names = groupedPaths.values().single();
                return findByNamesInOneFolder(uid, parentPath, names, false);
            }

            ListF<DjfsResource> uniqueResources;

            if (bulkInfoParallelInnerQueries.get()) {
                uniqueResources = CompletableFutures.allOf(
                        groupedPaths.mapEntries((parentPath, names) -> CompletableFuture.supplyAsync(
                                MasterSlaveContextHolder.supplyWithStandardThreadLocal(
                                        YandexCloudRequestIdHolder.supplyWithYcrid(
                                                () -> findByNamesInOneFolder(uid, parentPath, names, true))),
                                bulkInfoInnerExecutor)
                        )
                ).join().flatMap(l -> l);
            } else {
                uniqueResources = groupedPaths.mapEntries(
                        (parentPath, names) -> findByNamesInOneFolder(uid, parentPath, names, true)).flatMap(l -> l);
            }
            MapF<String, DjfsResource> resourcesByPath = uniqueResources
                    .toMap(resource -> resource.getPath().getPath(), resource -> resource);
            return rawPaths.flatMap(resourcesByPath::getO);
        }

        MapF<String, Object> parameters = Cf.hashMap();
        parameters.put("uid", uid.asLong());
        parameters.put("paths", rawPaths);

        /*
        Описание хитрого запроса ниже

        raw_paths - просто CTE, чтобы засунуть исходные пути в запрос

        preprocessed_paths - сплитятся пути по разделителю '/' и из них выделяется 2 части:
        в первой части находится массив родительских папок, чтобы вычитать их из базы отдельно
        во второй находиться имя ресурса (файла/папки)


        uniq_parents - собираем из preprocessed_paths уникальных родителей
        пример:
        пути на входе (в raw_paths):

        '/disk/test/d/1.jpg'
        '/disk/test/252.jpg'
        '/disk/test/242.jpg'
        '/disk/test/232.jpg'
        '/disk/test/d/2.jpg'
        '/disk/test/d/ddd'
        '/disk/test/d/ddd/1.txt'

        результат:

         level |       path
        -------+-------------------
             1 | {disk}
             2 | {disk,test}
             3 | {disk,test,d}
             4 | {disk,test,d,ddd}

        dir_hierarchy - берем рекурсивно из базы все папки содержащиеся в uniq_parents, попутно собирая пути
        */

        String query = collectStats(uid) + "\n"
                + "WITH RECURSIVE raw_paths (path) AS (\n"
                + "    SELECT\n"
                + "        unnest(ARRAY[ :paths ]::text[])\n"
                + "), preprocessed_paths AS (\n"
                + "    SELECT\n"
                + "        path,\n"
                + "        array_length(splitted_path, 1) AS path_len,\n"
                + "        CASE WHEN array_length(splitted_path, 1) = 1\n"
                + "            THEN ARRAY[]::text[]\n"
                + "            ELSE splitted_path[\\:array_length(splitted_path, 1) - 1]\n"
                + "        END AS path_head,\n"
                + "        (splitted_path[array_length(splitted_path, 1)\\:])[1] AS path_tail\n"
                + "    FROM (\n"
                + "        SELECT\n"
                + "            path,\n"
                + "            string_to_array(trim(both '/' from path), '/') as splitted_path\n"
                + "        FROM raw_paths\n"
                + "    ) paths\n"
                + "), uniq_parents AS (\n"
                + "    SELECT DISTINCT\n"
                + "        1 AS level,\n"
                + "        path_head[\\:1] AS path\n"
                + "    FROM preprocessed_paths\n"
                + "    WHERE path_head IS NOT null\n"
                + "    UNION ALL\n"
                + "    SELECT DISTINCT\n"
                + "        level + 1,\n"
                + "        path_head[\\:level + 1]\n"
                + "    FROM uniq_parents AS up\n"
                + "    JOIN preprocessed_paths pp ON up.path = pp.path_head[\\:level]\n"
                + "    WHERE pp.path_len - 1 >= level + 1\n"
                + "), dir_hierarchy AS  (\n"
                + "    SELECT\n"
                + "        0 AS level,\n"
                + "        dir.name,\n"
                + "        dir.parent_fid,\n"
                + "        dir.fid,\n"
                + "        ARRAY[dir.fid] AS parent_fids,\n"
                + "        ARRAY[]::text[] AS path\n"
                + "    FROM\n"
                + "        disk.folders dir\n"
                + "    WHERE dir.uid = :uid\n"
                + "        AND dir.parent_fid IS NULL\n"
                + "        AND dir.name = ''\n"
                + "    UNION ALL\n"
                + "    SELECT\n"
                + "        parent.level + 1,\n"
                + "        child.name,\n"
                + "        child.parent_fid,\n"
                + "        child.fid,\n"
                + "        parent.parent_fids || child.fid AS parent_fids,\n"
                + "        parent.path || child.name as path\n"
                + "    FROM dir_hierarchy parent\n"
                + "        JOIN disk.folders child on parent.fid = child.parent_fid\n"
                + "        JOIN uniq_parents up on parent.path || child.name = up.path\n"
                + "    WHERE uid = :uid\n"
                + ")\n"
                + "SELECT\n"
                + "    'file'::text AS type,\n"
                + "    dh.parent_fids,\n"
                + "    pp.path,\n"
                + "    " + FILE_FIELDS_PREFIXED_WITH_ALIASES + ", " + STORAGE_FILE_FIELDS_PREFIXED_WITH_ALIASES + ", "
                + FOLDER_NULLS_WITH_ALIASES + "\n"
                + "FROM dir_hierarchy dh\n"
                + "JOIN preprocessed_paths pp ON dh.path = pp.path_head\n"
                + "JOIN disk.files files ON dh.fid = files.parent_fid and pp.path_tail = files.name\n"
                + "JOIN disk.storage_files storage_files on files.storage_id = storage_files.storage_id\n"
                + "WHERE files.uid = :uid\n"
                + "UNION ALL\n"
                + "SELECT\n"
                + "    'dir'::text as type,\n"
                + "    dh.parent_fids,\n"
                + "    pp.path,\n"
                + "    " + FILE_NULLS_WITH_ALIASES + ", " + STORAGE_FILE_NULLS_WITH_ALIASES + ", "
                + FOLDER_FIELDS_PREFIXED + "\n"
                + "FROM dir_hierarchy dh \n"
                + "JOIN preprocessed_paths pp ON dh.path = pp.path_head\n"
                + "JOIN disk.folders folders ON dh.fid = folders.parent_fid and pp.path_tail = folders.name\n"
                + "WHERE folders.uid = :uid";

        return jdbcTemplate(uid).query(query, resourceMapper, parameters);
    }

    @Override
    public ListF<Tuple2<DjfsResource, ListF<UUID>>> findWithParentIds(DjfsUid uid, ListF<String> fileIds,
            ListF<DjfsResourceArea> areasToSearch)
    {
        if (areasToSearch.isEmpty()) {
            return Option.empty();
        }

        String sqlFilesBeginPart = "WITH file_data AS ("
                + "    SELECT " + FILE_FIELDS_PREFIXED + ", " + STORAGE_FILE_FIELDS_PREFIXED
                + "    FROM disk.files files JOIN disk.storage_files storage_files USING (storage_id)"
                + "    WHERE files.uid = :uid AND files.id = ANY(ARRAY[ :file_ids ])"
                + "),";

        String sqlFoldersBeginPart = "WITH file_data AS ("
                + "    SELECT " + FOLDER_FIELDS_PREFIXED
                + "    FROM disk.folders folders"
                + "    WHERE folders.uid = :uid AND folders.id = ANY(ARRAY[ :file_ids ])"
                + "),";

        String sqlCommonPart = " distinct_parents AS ("
                + "     SELECT DISTINCT parent_fid, uid FROM file_data"
                + " ), parent_paths AS ("
                + "     SELECT * FROM ("
                + "         WITH RECURSIVE r AS ("
                + "             SELECT 1 AS idx,"
                + "                 dir.name AS relative_path,"
                + "                 dir.name AS name,"
                + "                 dir.parent_fid AS parent_fid,"
                + "                 ARRAY[dir.fid] AS parent_fids,"
                + "                 dp.parent_fid AS original_parent_fid,"
                + "                 dp.uid AS uid"
                + "             FROM disk.folders AS dir JOIN distinct_parents AS dp"
                + "                 ON dir.uid = dp.uid AND dir.fid = dp.parent_fid"
                + "             UNION ALL"
                + "             SELECT idx + 1 AS idx,"
                + "                 parent.name || '/' || child.relative_path AS relative_path,"
                + "                 parent.name AS name,"
                + "                 parent.parent_fid AS parent_fid,"
                + "                 parent.fid || child.parent_fids AS parent_fids,"
                + "                 child.original_parent_fid,"
                + "                 child.uid"
                + "             FROM r child JOIN disk.folders parent ON child.parent_fid = parent.fid"
                + "                 AND parent.uid = child.uid"
                + "         )"
                +
                "         SELECT row_number() over w as row_num, relative_path AS path, original_parent_fid, parent_fids, uid, idx"
                + "         FROM r"
                + "         WINDOW w AS (PARTITION BY original_parent_fid, uid ORDER BY idx DESC)"
                + "     ) AS q"
                + "     WHERE row_num = 1"
                + " )"
                + " SELECT fd.*, parent_paths.path || '/' || fd.name AS path, parent_paths.parent_fids"
                +
                " FROM file_data AS fd JOIN parent_paths ON fd.uid = parent_paths.uid AND fd.parent_fid = parent_paths.original_parent_fid";

        String storageLikeClause = String.join(" OR ",
                areasToSearch
                        .map(x -> "parent_paths.path LIKE '/" + x.value() + "/%' OR parent_paths.path = '/" + x.value()
                                + "' OR parent_paths.path || '/' || fd.name = '/" + x.value() + "'"));

        String sqlLikeEndPart = " WHERE (" + storageLikeClause + ")";

        MapF<String, Object> parameters = Cf.map(
                "uid", uid,
                "file_ids", fileIds.map(Hex::decode)
        );

        String sqlFilesQuery = collectStats(uid) + sqlFilesBeginPart + sqlCommonPart + sqlLikeEndPart;
        String sqlFoldersQuery = collectStats(uid) + sqlFoldersBeginPart + sqlCommonPart + sqlLikeEndPart;

        RowMapper<Tuple2<DjfsResource, ListF<UUID>>> fileWithParentIdsMapper = (rs, rowNum) -> {
            ListF<UUID> parentIds = Cf.wrap((UUID[]) rs.getArray("parent_fids").getArray());
            FileDjfsResource file = fileMapper.mapRow(rs, rowNum);
            return Tuple2.tuple(file, parentIds);
        };

        RowMapper<Tuple2<DjfsResource, ListF<UUID>>> folderWithParentIdsMapper = (rs, rowNum) -> {
            ListF<UUID> parentIds = Cf.wrap((UUID[]) rs.getArray("parent_fids").getArray());
            FolderDjfsResource folder = folderMapper.mapRow(rs, rowNum);
            return Tuple2.tuple(folder, parentIds);
        };

        ListF<Tuple2<DjfsResource, ListF<UUID>>> files =
                jdbcTemplate(uid).query(sqlFilesQuery, fileWithParentIdsMapper, parameters);
        ListF<Tuple2<DjfsResource, ListF<UUID>>> folders =
                jdbcTemplate(uid).query(sqlFoldersQuery, folderWithParentIdsMapper, parameters);

        return Cf.<Tuple2<DjfsResource, ListF<UUID>>>list().plus(files).plus(folders);
    }

    @Override
    public ListF<FolderDjfsResource> find2ImmediateChildFoldersOrderByFileIdThenId(DjfsUid uid, DjfsResourceArea area,
            UUID parentId, Option<String> startingFileId, Option<UUID> startingId, int limit, boolean forUpdate)
    {
        String sql = collectStats(uid) + "SELECT " + FOLDER_FIELDS_PREFIXED + ","
                + "     (SELECT path FROM code.fid_to_path(folders.fid, folders.uid)) as path"
                + " FROM disk.folders folders"
                + " WHERE folders.uid = :uid AND folders.parent_fid = :parent_fid";
        MapF<String, Object> parameters = Cf.map("uid", uid.asLong(), "parent_fid", parentId);
        if (startingFileId.isPresent()) {
            if (startingId.isPresent()) {
                sql += " AND ( "
                        + " folders.id > :starting_file_id "
                        + " OR ( folders.id = :starting_file_id AND folders.fid > :starting_id ) ) ";
                parameters = parameters
                        .plus1("starting_file_id", startingFileId.map(Hex::decode).get())
                        .plus1("starting_id", startingId.get());
            } else {
                sql += " AND folders.id > :starting_file_id ";
                parameters = parameters.plus1("starting_file_id", startingFileId.map(Hex::decode).get());
            }
        }

        sql += " ORDER BY folders.id, folders.fid"
                + " LIMIT " + limit;

        if (forUpdate) {
            sql += " FOR UPDATE";
        }

        return jdbcTemplate(uid).query(sql, folderMapper, parameters);
    }

    @Override
    public ListF<FolderDjfsResource> find2ImmediateChildFoldersWithoutFileId(DjfsUid uid, DjfsResourceArea area,
            UUID parentId)
    {
        // FileId can't be absent, this violates the constraint
        // do nothing
        return Cf.list();
    }

    @Override
    public ListF<FileDjfsResource> find2ImmediateChildFilesOrderByFileIdThenId(DjfsUid uid, DjfsResourceArea area,
            UUID parentId, Option<String> startingFileId, Option<UUID> startingId, int limit, boolean forUpdate)
    {
        String sql = collectStats(uid) + "SELECT " + FILE_FIELDS_PREFIXED + ", " + STORAGE_FILE_FIELDS_PREFIXED + ", "
                + "     (SELECT path FROM code.fid_to_path(files.fid, files.uid)) as path"
                + " FROM disk.files files"
                + " JOIN disk.storage_files storage_files USING (storage_id)"
                + " WHERE files.uid = :uid AND files.parent_fid = :parent_fid";
        MapF<String, Object> parameters = Cf.map("uid", uid.asLong(), "parent_fid", parentId);
        if (startingFileId.isPresent()) {
            if (startingId.isPresent()) {
                sql += " AND ( "
                        + " files.id > :starting_file_id "
                        + " OR ( files.id = :starting_file_id AND files.fid > :starting_id ) ) ";
                parameters = parameters
                        .plus1("starting_file_id", startingFileId.map(Hex::decode).get())
                        .plus1("starting_id", startingId.get());
            } else {
                sql += " AND files.id > :starting_file_id ";
                parameters = parameters.plus1("starting_file_id", startingFileId.map(Hex::decode).get());
            }
        }

        sql += " ORDER BY files.id, files.fid"
                + " LIMIT " + limit;

        if (forUpdate) {
            sql += " FOR UPDATE";
        }

        return jdbcTemplate(uid).query(sql, fileMapper, parameters);
    }

    @Override
    public Option<FileDjfsResource> find2AdditionalFileFor(FileDjfsResource file) {
        String sql =
                collectStats(file) + "SELECT " + FILE_FIELDS_PREFIXED + ", " + STORAGE_FILE_FIELDS_PREFIXED + ", "
                        + "     (SELECT path FROM code.fid_to_path(files.fid, files.uid)) as path"
                        + " FROM disk.files files"
                        + " JOIN disk.storage_files storage_files USING (storage_id)"
                        +
                        " JOIN disk.additional_file_links additional_file_links ON files.fid = additional_file_links.additional_file_fid"
                        + " WHERE files.uid = :uid AND type = 'live_video' "
                        + "     AND additional_file_links.uid = :uid AND additional_file_links.main_file_fid = :fid";
        MapF<String, Object> parameters = Cf.map(
                "uid", file.getUid().asLong(),
                "fid", file.getId());
        return jdbcTemplate(file.getUid()).queryForOption(sql, fileMapper, parameters);
    }

    @Override
    public ListF<FilenamePreviewStidMimetypeVersionFileId> find2FilenamePreviewStidMimetypeVersionFid(
            DjfsUid uid, ListF<DjfsFileId> fileIds)
    {
        String q = collectStats(uid) +
                " SELECT files.name, files.mime_type, files.version, files.fid, files.id, storage_files.preview_stid"
                + " FROM disk.files files JOIN disk.storage_files storage_files USING (storage_id) "
                + " WHERE files.uid = :uid AND files.id IN (:file_ids)";

        MapF<String, Object> parameters = Cf.map(
                "uid", uid.asLong(),
                "file_ids", fileIds.map(DjfsFileId::getValue).map(Hex::decode));

        return jdbcTemplate(uid).query(q, (rs, rowNum) ->
                new FilenamePreviewStidMimetypeVersionFileId(
                        ResultSetUtils.getStringO(rs, "preview_stid"),
                        ResultSetUtils.getStringO(rs, "mime_type"),
                        rs.getString("name"),
                        ResultSetUtils.getLongO(rs, "version"),
                        ResultSetUtils.getUuid(rs, "fid"),
                        Option.ofNullable(rs.getBytes("id")).map(Hex::encode).map(DjfsFileId::cons)
                ), parameters
        );
    }

    public ListF<Tuple3<UUID, StidType, String>> findStids(DjfsShardInfo.Pg shardInfo, Option<UUID> start,
            Option<UUID> end, int limit)
    {
        String where;
        if (start.isPresent() && end.isPresent()) {
            where = " WHERE storage_id > :storage_id_start AND storage_id <= :storage_id_end ";
        } else if (start.isPresent()) {
            where = " WHERE storage_id > :storage_id_start ";
        } else if (end.isPresent()) {
            where = " WHERE storage_id <= :storage_id_end ";
        } else {
            where = "";
        }

        String sql = "SELECT storage_id, stid, digest_stid, preview_stid FROM disk.storage_files "
                + where
                + " ORDER BY storage_id ASC LIMIT " + limit;

        MapF<String, Object> parameters = Cf.map(
                "storage_id_start", start.getOrNull(),
                "storage_id_end", end.getOrNull());

        return MasterSlaveContextHolder.withPolicy(MasterSlavePolicy.R_SM, () ->
                jdbcTemplate(shardInfo).query(sql, (x, y) -> {
                    ListF<Tuple3<UUID, StidType, String>> r = Cf.arrayList();
                    UUID id = UuidUtils.from(x.getString("storage_id"));
                    if (x.getString("stid") != null) {
                        r.add(Tuple3.tuple(id, StidType.FILE, x.getString("stid")));
                    }
                    if (x.getString("digest_stid") != null) {
                        r.add(Tuple3.tuple(id, StidType.DIGEST, x.getString("digest_stid")));
                    }
                    if (x.getString("preview_stid") != null) {
                        r.add(Tuple3.tuple(id, StidType.PREVIEW, x.getString("preview_stid")));
                    }
                    return r;
                }, parameters).flatten());
    }

    @Override
    public ListF<DjfsResource> find2ImmediateChildren(DjfsResourcePath path, int limit) {
        return find2ImmediateChildren(path, 0, limit, false);
    }

    @Override
    public long countImmediateChildren(DjfsResourcePath path) {
        String s = collectStats(path)
                + " SELECT SUM(CNT) FROM ("
                + " SELECT COUNT(*) CNT "
                + " FROM"
                + "     disk.files files"
                + " WHERE"
                + "     files.uid = :uid"
                + "     AND files.parent_fid = (SELECT fid FROM code.path_to_fid(:path, :uid))"
                + " UNION ALL"
                + " SELECT COUNT(*) CNT"
                + " FROM"
                + "     disk.folders folders"
                + " WHERE"
                + "     folders.uid = :uid"
                + "     AND folders.parent_fid = (SELECT fid FROM code.path_to_fid(:path, :uid))"
                + ") t";
        MapF<String, Object> parameters = Tuple2List.fromPairs(
                "uid", path.getUid(),
                "path", path.getPath())
                .toMap();

        return jdbcTemplate(path.getUid()).query(s, new SingleColumnRowMapper<>(Long.class), parameters).single();

    }

    @Override
    public ListF<DjfsResource> find2ImmediateChildren(DjfsResourcePath path, int offset, int limit, boolean ordered) {
        final String orderedBy = ordered ? " ORDER BY type, path" : "";
        String s = collectStats(path)
                + " ("
                + " SELECT "
                + FILE_FIELDS_PREFIXED_WITH_ALIASES + ", " + STORAGE_FILE_FIELDS_PREFIXED_WITH_ALIASES + ", "
                + FOLDER_NULLS_WITH_ALIASES + ", :path || '/' || files.name as path, 'file' as type"
                + " FROM"
                + "     disk.files files JOIN disk.storage_files storage_files USING (storage_id)"
                + " WHERE"
                + "     files.uid = :uid"
                + "     AND files.parent_fid = (SELECT fid FROM code.path_to_fid(:path, :uid))"
                + " UNION ALL"
                + " SELECT "
                + FILE_NULLS_WITH_ALIASES + ", " + STORAGE_FILE_NULLS_WITH_ALIASES + ", "
                + FOLDER_FIELDS_PREFIXED_WITH_ALIASES + ", :path || '/' || folders.name as path, 'dir' as type"
                + " FROM"
                + "     disk.folders folders"
                + " WHERE"
                + "     folders.uid = :uid"
                + "     AND folders.parent_fid = (SELECT fid FROM code.path_to_fid(:path, :uid))"
                + orderedBy
                + " OFFSET :offset"
                + " LIMIT :limit"
                + ")";

        MapF<String, Object> parameters = Tuple2List.fromPairs(
                "uid", path.getUid(),
                "path", path.getPath(),
                "offset", offset)
                .plus1("limit", limit)
                .toMap();

        return jdbcTemplate(path.getUid()).query(s, resourceMapper, parameters).take(limit);
    }

    @Override
    public ListF<DjfsResource> find2ImmediateChildren(DjfsResource resource, int offset, int limit, boolean ordered) {
        final String orderedBy = ordered ? " ORDER BY name" : "";

        String foldersCountQuery = "SELECT COUNT(*)"
                + " FROM"
                + "     disk.folders folders"
                + " WHERE"
                + "     folders.uid = ?"
                + "     AND folders.parent_fid = ?";
        final int foldersCount =
                jdbcTemplate(resource.getUid()).queryForInt(foldersCountQuery, resource.getUid(), resource.getId());

        MapF<String, Object> parameters = Cf.map(
                "uid", resource.getUid(),
                "fid", resource.getId(),
                "path", resource.getPath().getPath());

        ListF<DjfsResource> resources = Cf.list();
        if (foldersCount > offset) {
            String foldersQuery = collectStats(resource.getPath())
                    + " ("
                    + " SELECT "
                    + FILE_NULLS_WITH_ALIASES + ", " + STORAGE_FILE_NULLS_WITH_ALIASES + ", "
                    + FOLDER_FIELDS_PREFIXED_WITH_ALIASES + ", :path || '/' || folders.name as path, 'dir' as type"
                    + " FROM"
                    + "     disk.folders folders"
                    + " WHERE"
                    + "     folders.uid = :uid"
                    + "     AND folders.parent_fid = :fid"
                    + orderedBy
                    + " OFFSET :offset"
                    + " LIMIT :limit"
                    + ")";

            resources = resources.plus(jdbcTemplate(resource.getUid())
                    .query(foldersQuery, resourceMapper, parameters.plus1("offset", offset).plus1("limit", limit)));
        }

        final int filesOffset = offset > foldersCount ? offset - foldersCount : 0;
        final int filesLimit = offset > foldersCount ? limit : limit - (foldersCount - offset);
        if (filesLimit > 0) {
            String filesCountQuery = "SELECT COUNT(*)"
                    + " FROM"
                    + "     disk.files files"
                    + " WHERE"
                    + "     files.uid = ?"
                    + "     AND files.parent_fid = ?";
            final int filesCount =
                    jdbcTemplate(resource.getUid()).queryForInt(filesCountQuery, resource.getUid(), resource.getId());
            if (filesCount > filesOffset) {
                String filesQuery = collectStats(resource.getPath())
                        + " ("
                        + " SELECT "
                        + FILE_FIELDS_PREFIXED_WITH_ALIASES + ", " + STORAGE_FILE_FIELDS_PREFIXED_WITH_ALIASES + ", "
                        + FOLDER_NULLS_WITH_ALIASES + ", :path || '/' || files.name as path, 'file' as type"
                        + " FROM"
                        + "     disk.files files JOIN disk.storage_files storage_files USING (storage_id)"
                        + " WHERE"
                        + "     files.uid = :uid"
                        + "     AND files.parent_fid = :fid"
                        + orderedBy
                        + " OFFSET :offset"
                        + " LIMIT :limit"
                        + ")";

                resources = resources.plus(jdbcTemplate(resource.getUid())
                        .query(filesQuery, resourceMapper,
                                parameters.plus1("offset", filesOffset).plus1("limit", filesLimit)));
            }
        }
        return resources;
    }

    @Override
    public ListF<DjfsResource> findAll(DjfsUid uid, DjfsResourceArea partition) {
        String s = collectStats(uid) + " SELECT " + FILE_FIELDS_PREFIXED + ", " + STORAGE_FILE_FIELDS_PREFIXED + ", "
                + "    (SELECT path FROM code.fid_to_path(files.fid, files.uid)) as path"
                + " FROM disk.files files"
                + " JOIN disk.storage_files storage_files USING (storage_id)"
                + " WHERE files.uid = :uid";

        String q = collectStats(uid) + " SELECT " + FOLDER_FIELDS_PREFIXED + ", "
                + "     (SELECT path FROM code.fid_to_path(folders.fid, folders.uid)) as path"
                + " FROM disk.folders folders"
                + " WHERE folders.uid = :uid";

        ListF<FileDjfsResource> files = jdbcTemplate(uid).query(s, fileMapper, Cf.map("uid", uid));
        ListF<FolderDjfsResource> folders = jdbcTemplate(uid).query(q, folderMapper, fileMapper, Cf.map("uid", uid));
        return folders.<DjfsResource>cast().plus(files.cast());
    }

    @Override
    public void insert(FolderDjfsResource resource) {
        String sql = collectStats(resource)
                + " INSERT INTO disk.folders (uid, fid, parent_fid, id, name, visible, version, public, public_hash, "
                + "symlink, short_url, blocked, published, files_count, folders_count, folder_type, date_removed, "
                + "date_uploaded, date_created, date_modified, date_hidden_data, path_before_remove, modify_uid, "
                + "download_counter, folder_url, custom_properties, custom_setprop_fields, yarovaya_mark, area) "

                + " VALUES (:uid, :fid, :parent_fid, :id, :name, :visible, :version, :public, :public_hash, "
                +
                ":symlink, :short_url, :blocked, :published, :files_count, :folders_count, :folder_type, :date_removed, "
                + ":date_uploaded, :date_created, :date_modified, :date_hidden_data, :path_before_remove, :modify_uid, "
                + ":download_counter, :folder_url, :custom_properties, :custom_setprop_fields::json, :yarovaya_mark, "
                + ":area)";

        MapF<String, Object> customSetpropFields = Cf.hashMap();
        customSetpropFields.putAll(resource.getExternalProperties());
        if (resource.getLastImportTime().isPresent()) {
            customSetpropFields.put("last_import_time", InstantUtils.toSecondsLong(resource.getLastImportTime().get()));
        }

        MapF<String, Object> parameters = Cf.toMap(Tuple2List.fromPairs(
                "uid", resource.getUid(),
                "fid", resource.getId(),
                "parent_fid", resource.getParentId().getOrNull(),
                "id", resource.getFileId().map(DjfsFileId::getValue).map(Hex::decode).getOrNull(),
                "name", resource.getPath().getName(),
                "visible", resource.isVisible(),
                "version", resource.getVersion().getOrElse(0L),
                "public", resource.isPublic(),
                "public_hash", resource.getPublicHash().map(CharsetUtils::encodeUtf8ToArray).getOrNull(),
                "symlink", resource.getSymlink().getOrNull(),
                "short_url", resource.getShortUrl().getOrNull(),
                "blocked", resource.isBlocked(),
                "published", resource.isPublished(),
                "files_count", 0,
                "folders_count", 0,
                "folder_type", resource.getFolderType().getOrNull(),
                "date_removed", resource.getTrashAppendTime().getOrNull(),
                "date_uploaded", resource.getUploadTime().getOrNull(),
                "date_created", resource.getCreationTime().getOrNull(),
                "date_modified", resource.getModificationTime().getOrNull(),
                "date_hidden_data", resource.getHiddenAppendTime().getOrNull(),
                "path_before_remove", resource.getTrashAppendOriginalPath().getOrNull(),
                "modify_uid", resource.getModifyUid().getOrNull(),
                "download_counter", resource.getDownloadCounter().getOrNull(),
                "folder_url", resource.getFolderUrl().getOrNull(),
                "custom_properties", resource.getCustomProperties().getOrNull(),
                "custom_setprop_fields", mapper.writeValueAsString(customSetpropFields),
                "yarovaya_mark", resource.hasYarovayaMark(),
                "area", resource.getArea().getOrNull()
        ));

        try {
            jdbcTemplate(resource.getUid()).update(sql, parameters);
        } catch (DataIntegrityViolationException e) {
            checkAlreadyExistsException(e, resource);
            throw e;
        }
    }

    @Override
    public void checkAlreadyExistsException(Throwable e, DjfsResource resource) throws EntityAlreadyExistsException {
        Throwable cause = e.getCause();
        if (cause instanceof PSQLException) {
            ServerErrorMessage error = ((PSQLException) cause).getServerErrorMessage();
            String pkConstraint, ukConstraint;
            if (FileDjfsResource.class.equals(resource.getClass())) {
                pkConstraint = "pk_files";
                ukConstraint = "uk_files_uid_parent_fid";
            } else if (FolderDjfsResource.class.equals(resource.getClass())) {
                pkConstraint = "pk_folders";
                ukConstraint = "uk_folders_uid_parent_fid_name";
            } else {
                return;
            }
            if (error != null && (Objects.equals(error.getConstraint(), pkConstraint)
                    || Objects.equals(error.getConstraint(), ukConstraint)
                    || Objects.equals(error.getConstraint(), "uk_resource_parent_fid_name")
                    || Objects.equals(error.getSQLState(), PSQLState.UNIQUE_VIOLATION.getCode())
            )) {
                logger.info("PgDjfsResourceDao.insert(" + resource.getClass().getSimpleName() +
                        ") handled exception for path "
                        + resource.getPath().toString() + " : ", e);
                throw new EntityAlreadyExistsException(UuidUtils.toHexString(resource.getId()), e);
            }
        }
    }

    @Override
    public void insert2(FolderDjfsResource folder) {
        insert(folder);
    }

    @Override
    public void insert(FileDjfsResource resource) {
        String sql2 = collectStats(resource)
                + " INSERT INTO disk.storage_files (storage_id, stid, digest_stid, preview_stid, date_origin, "
                + "size, md5_sum, sha256_sum, av_scan_status, height, width, angle, video_data) "

                + " VALUES (:storage_id, :stid, :digest_stid, :preview_stid, :date_origin, "
                + ":size, :md5_sum, :sha256_sum, :av_scan_status, :height, :width, :angle, :video_data::json) "
                + " ON CONFLICT ON CONSTRAINT pk_storage_files DO NOTHING";

        MapF<String, Object> parameters2 = Cf.toMap(Tuple2List.fromPairs(
                "storage_id", UuidUtils.fromHex(resource.getHid()),
                "stid", resource.getFileStid(),
                "digest_stid", resource.getDigestStid(),
                "preview_stid", resource.getPreviewStid().getOrNull(),
                "date_origin", Instant.now(),
                "size", resource.getSize(),
                "md5_sum", UuidUtils.from(Hex.decode(resource.getMd5())),
                "sha256_sum", Hex.decode(resource.getSha256()),
                "av_scan_status", resource.getAntiVirusScanStatus().getOrNull(),
                "height", resource.getHeight().getOrNull(),
                "width", resource.getWidth().getOrNull(),
                "angle", resource.getAngle().getOrNull(),
                "video_data", resource.getVideoInfo().getOrNull()
        ));

        String sql = collectStats(resource)
                + "INSERT INTO disk.files (uid, fid, parent_fid, storage_id, id, name, visible, version, public, "
                + "public_hash, symlink, short_url, download_counter, folder_url, blocked, source_uid, published, "
                +
                "date_created, date_uploaded, date_modified, date_removed, date_hidden_data, date_exif, path_before_remove, "
                + "modify_uid, media_type, mime_type, source, custom_properties, custom_setprop_fields, "
                + "is_live_photo, yarovaya_mark, ext_aesthetics, photoslice_album_type, albums_exclusions, area, "
                + "ext_coordinates) "

                + " VALUES (:uid, :fid, :parent_fid, :storage_id, :id, :name, :visible, :version, :public, "
                +
                ":public_hash, :symlink, :short_url, :download_counter, :folder_url, :blocked, :source_uid, :published, "
                +
                ":date_created, :date_uploaded, :date_modified, :date_removed, :date_hidden_data, :date_exif, :path_before_remove, "
                + ":modify_uid, :media_type, :mime_type, :source, :custom_properties, :custom_setprop_fields::json, "
                + ":is_live_photo, :yarovaya_mark, :ext_aesthetics, :photoslice_album_type, :albums_exclusions, :area,"
                + ":ext_coordinates)";

        MapF<String, String> customSetpropFields = Cf.hashMap();
        customSetpropFields.putAll(resource.getExternalProperties());
        if (resource.getFotkiTags().isPresent()) {
            customSetpropFields.put("fotki_tags", resource.getFotkiTags().get());
        }
        if (resource.getExternalUrl().isPresent()) {
            customSetpropFields.put("external_url", resource.getExternalUrl().get());
        }

        MapF<String, Object> parameters = Cf.toMap(Tuple2List.fromPairs(
                "uid", resource.getUid().asLong(),
                "fid", resource.getId(),
                "parent_fid", resource.getParentId().get(),
                "storage_id", UuidUtils.fromHex(resource.getHid()),
                "id", resource.getFileId().map(DjfsFileId::getValue).map(Hex::decode).getOrNull(),
                "name", resource.getPath().getName(),
                "visible", resource.isVisible(),
                "version", resource.getVersion().getOrNull(),
                "public", resource.isPublic(),
                "public_hash", resource.getPublicHash().map(x -> x.getBytes(StandardCharsets.UTF_8)).getOrNull(),
                "symlink", resource.getSymlink().getOrNull(),
                "short_url", resource.getShortUrl().getOrNull(),
                "download_counter", resource.getDownloadCounter().getOrNull(),
                "folder_url", resource.getFolderUrl().getOrNull(),
                "blocked", resource.isBlocked(),
                "source_uid", null,
                "published", resource.isPublished(),
                "date_created", resource.getCreationTime(),
                "date_uploaded", resource.getUploadTime().getOrNull(),
                "date_modified", resource.getModificationTime(),
                "date_removed", resource.getTrashAppendTime().getOrNull(),
                "date_hidden_data", resource.getHiddenAppendTime().getOrNull(),
                "date_exif", resource.getExifTime().getOrNull(),
                "path_before_remove", resource.getTrashAppendOriginalPath().getOrNull(),
                "modify_uid", resource.getModifyUid().map(DjfsUid::asLong).getOrNull(),
                "media_type", resource.getMediaType().getOrNull(),
                "mime_type", resource.getMimetype().getOrNull(),
                "source", resource.getSource().getOrNull(),
                "custom_properties", resource.getCustomProperties().getOrNull(),
                "custom_setprop_fields", mapper.writeValueAsString(customSetpropFields),
                "is_live_photo", resource.isLivePhoto(),
                "yarovaya_mark", resource.hasYarovayaMark(),
                "ext_aesthetics", resource.getAesthetics().getOrNull(),
                "photoslice_album_type", resource.getPhotosliceAlbumType().getOrNull(),
                "albums_exclusions",
                resource.getAlbumsExclusions().map(p -> PgArray.textArray(p.toArray(String.class))).getOrNull(),
                "area", resource.getArea().getOrNull(),
                "ext_coordinates",
                resource.getCoordinates().map(x -> new PGpoint(x.getLatitude(), x.getLongitude())).getOrNull()
        ));

        jdbcTemplate(resource.getUid()).update(sql2, parameters2);
        try {
            jdbcTemplate(resource.getUid()).update(sql, parameters);
        } catch (DataIntegrityViolationException e) {
            checkAlreadyExistsException(e, resource);
            throw e;
        }
    }

    @Override
    public void insert2(FileDjfsResource file) {
        insert(file);
    }

    @Override
    public void insert2AndLinkTo(FileDjfsResource file, FileDjfsResource link) {
        if (!file.isLivePhoto()) {
            throw new NotImplementedException("insert2AndLinkTo: file is not live photo");
        }
        if (link.getPath().getArea() != DjfsResourceArea.ADDITIONAL_DATA) {
            throw new NotImplementedException("insert2AndLinkTo: link is not in additional data");
        }

        insert2(file);
        String sql = collectStats(file)
                + " INSERT INTO disk.additional_file_links (uid, main_file_fid, additional_file_fid, type) "
                + "   VALUES (:uid, :main_file_fid, :additional_file_fid, 'live_video')";
        Map<String, Object> parameters = Cf.map(
                "uid", file.getUid(),
                "main_file_fid", file.getId(),
                "additional_file_fid", link.getId());
        jdbcTemplate(file.getUid()).update(sql, parameters);
    }

    @Override
    public void changeParent(DjfsUid uid, UUID id, UUID newParentId) {
        String s = collectStats(uid)
                + " UPDATE disk.folders SET parent_fid = :parent_fid WHERE uid = :uid and fid = :fid;";
        String q = collectStats(uid)
                + " UPDATE disk.files SET parent_fid = :parent_fid WHERE uid = :uid and fid = :fid;";
        Map<String, Object> parameters = Cf.map("uid", uid, "fid", id, "parent_fid", newParentId);
        jdbcTemplate(uid).update(s, parameters);
        jdbcTemplate(uid).update(q, parameters);
    }

    @Override
    public void changeParentAndName(DjfsUid uid, UUID id, UUID newParentId, String name) {
        String s = collectStats(uid)
                + " UPDATE disk.folders SET name = :name, parent_fid = :parent_fid WHERE uid = :uid and fid = :fid;";
        String q = collectStats(uid)
                + " UPDATE disk.files SET name = :name, parent_fid = :parent_fid WHERE uid = :uid and fid = :fid;";
        Map<String, Object> parameters = Cf.map(
                "uid", uid,
                "fid", id,
                "name", name,
                "parent_fid", newParentId
        );
        jdbcTemplate(uid).update(s, parameters);
        jdbcTemplate(uid).update(q, parameters);
    }

    @Override
    public void setVersion(DjfsUid uid, UUID id, long version) {
        String s = collectStats(uid)
                + " UPDATE disk.folders SET version = :version WHERE uid = :uid and fid = :fid;";
        String q = collectStats(uid)
                + " UPDATE disk.files SET version = :version WHERE uid = :uid and fid = :fid;";
        Map<String, Object> parameters = Cf.map("uid", uid, "fid", id, "version", version);
        jdbcTemplate(uid).update(s, parameters);
        jdbcTemplate(uid).update(q, parameters);
    }

    @Override
    public void setModificationTime(DjfsUid uid, UUID id, Instant time) {
        String s = collectStats(uid)
                + " UPDATE disk.folders SET date_modified = :date_modified WHERE uid = :uid and fid = :fid;";
        String q = collectStats(uid)
                + " UPDATE disk.files SET date_modified = :date_modified WHERE uid = :uid and fid = :fid;";
        Map<String, Object> parameters = Cf.map("uid", uid, "fid", id, "date_modified", time);
        jdbcTemplate(uid).update(s, parameters);
        jdbcTemplate(uid).update(q, parameters);
    }

    @Override
    public void setYarovayaMark(DjfsUid uid, UUID id) {
        String s = collectStats(uid)
                + " UPDATE disk.folders SET yarovaya_mark = true WHERE uid = :uid and fid = :fid;";
        String q = collectStats(uid)
                + " UPDATE disk.files SET yarovaya_mark = true WHERE uid = :uid and fid = :fid;";
        Map<String, Object> parameters = Cf.map("uid", uid, "fid", id);
        jdbcTemplate(uid).update(s, parameters);
        jdbcTemplate(uid).update(q, parameters);
    }

    @Override
    public void setTrashAppendTimeAndOriginalPath(DjfsUid uid, UUID id, String path, Instant trashAppendTime) {
        String s = collectStats(uid)
                + " UPDATE disk.folders"
                + " SET path_before_remove = :path_before_remove, date_removed=:date_removed"
                + " WHERE uid = :uid AND fid = :fid";
        String q = collectStats(uid)
                + " UPDATE disk.files"
                + " SET path_before_remove = :path_before_remove, date_removed=:date_removed"
                + " WHERE uid = :uid AND fid = :fid";
        Map<String, Object> parameters = Cf.map(
                "uid", uid,
                "fid", id,
                "path_before_remove", path,
                "date_removed", trashAppendTime
        );
        jdbcTemplate(uid).update(s, parameters);
        jdbcTemplate(uid).update(q, parameters);
    }

    @Override
    public void setTrashAppendTime(DjfsUid uid, ListF<UUID> ids, Instant trashAppendTime) {
        String s = collectStats(uid)
                + " UPDATE disk.folders SET date_removed=:date_removed WHERE uid = :uid AND fid IN (:fids)";
        String q = collectStats(uid)
                + " UPDATE disk.files SET date_removed=:date_removed WHERE uid = :uid AND fid IN (:fids)";
        Map<String, Object> parameters = Cf.map(
                "uid", uid,
                "fids", ids,
                "date_removed", trashAppendTime
        );
        jdbcTemplate(uid).update(s, parameters);
        jdbcTemplate(uid).update(q, parameters);
    }

    @Override
    public void setArea(DjfsUid uid, ListF<UUID> ids, DjfsResourceArea area) {
        String s = collectStats(uid)
                + " UPDATE disk.folders SET area=:area::disk.area_type WHERE uid = :uid AND fid IN (:fids)";
        String q = collectStats(uid)
                + " UPDATE disk.files SET area=:area::disk.area_type WHERE uid = :uid AND fid IN (:fids)";
        Map<String, Object> parameters = Cf.map(
                "uid", uid,
                "fids", ids,
                "area", area.value()
        );
        jdbcTemplate(uid).update(s, parameters);
        jdbcTemplate(uid).update(q, parameters);
    }

    @Override
    public void setHiddenAppendTime(DjfsUid uid, UUID id, Instant hiddenAppendTime) {
        String s = collectStats(uid)
                + " UPDATE disk.folders SET date_hidden_data = :date_hidden_data WHERE uid = :uid and fid = :fid;";
        String q = collectStats(uid)
                + " UPDATE disk.files SET date_hidden_data = :date_hidden_data WHERE uid = :uid and fid = :fid;";
        Map<String, Object> parameters = Cf.map("uid", uid, "fid", id, "date_hidden_data", hiddenAppendTime);
        jdbcTemplate(uid).update(s, parameters);
        jdbcTemplate(uid).update(q, parameters);
    }

    @Override
    public void setPublishedAndRemovePublic(DjfsUid uid, UUID id) {
        String s = collectStats(uid)
                + " UPDATE disk.folders SET published = true, public = false, download_counter = null"
                + " WHERE uid = :uid and fid = :fid";
        String q = collectStats(uid)
                + " UPDATE disk.files SET published = true, public = false, download_counter = null"
                + " WHERE uid = :uid and fid = :fid";

        Map<String, Object> parameters = Cf.map("uid", uid, "fid", id);
        jdbcTemplate(uid).update(s, parameters);
        jdbcTemplate(uid).update(q, parameters);
    }

    @Override
    public void addFileId(DjfsResourcePath path, DjfsFileId fileId) {
        // FileId can't be absent, this violates the constraint
        // do nothing
    }

    @Override
    public void removeFileId(DjfsResourcePath path) {
        // FileId can't be absent, this violates the constraint
        // do nothing
        // should only be used in tests
    }

    @Override
    public void setAesthetics(DjfsUid uid, UUID id, double aesthetics) {
        String q = collectStats(uid)
                + " UPDATE disk.files SET ext_aesthetics = :aesthetics WHERE uid = :uid AND fid = :fid";
        Map<String, Object> parameters = Cf.map("uid", uid, "fid", id, "aesthetics", aesthetics);
        jdbcTemplate(uid).update(q, parameters);
    }

    @Override
    public void setCoordinates(DjfsUid uid, UUID id, Coordinates coordinates) {
        String q = collectStats(uid)
                + " UPDATE disk.files SET ext_coordinates = :ext_coordinates WHERE uid = :uid AND fid = :fid";
        Map<String, Object> parameters = Cf.map(
                "uid", uid,
                "fid", id,
                "ext_coordinates", new PGpoint(coordinates.getLatitude(), coordinates.getLongitude())
        );
        jdbcTemplate(uid).update(q, parameters);
    }

    @Override
    public void setAesthetics(DjfsUid uid, Tuple2List<DjfsFileId, Double> fileIdWithAesthetics, boolean skipExisting) {
        MapSqlParameterSource parameters = new MapSqlParameterSource();
        String template = "(:id_%1$s,:aesthetics_%1$s)";

        parameters.addValue("uid", uid);
        StringBuilder values = new StringBuilder();
        for (int i = 0; i < fileIdWithAesthetics.size(); i++) {
            if (i != 0) {
                values.append(",");
            }

            values.append(String.format(template, i + 1));
            parameters.addValue("id_" + (i + 1), Hex.decode(fileIdWithAesthetics.get(i)._1.getValue()));
            parameters.addValue("aesthetics_" + (i + 1), fileIdWithAesthetics.get(i)._2);
        }

        String query = collectStats(uid)
                + " UPDATE disk.files AS f SET"
                + "     ext_aesthetics = v.aesthetics"
                + " FROM (VALUES "
                + values
                + " ) AS v(id, aesthetics)"
                + " WHERE f.uid = :uid AND f.id = v.id";
        if (skipExisting) {
            query += " AND f.ext_aesthetics IS NULL";
        }

        jdbcTemplate(uid).update(query, parameters);
    }

    @Override
    public void setCoordinates(DjfsUid uid, Tuple2List<DjfsFileId, Coordinates> fileIdWithCoordinates,
            boolean skipExisting)
    {
        MapSqlParameterSource parameters = new MapSqlParameterSource();
        String template = "(:id_%1$s,:coord_%1$s)";

        parameters.addValue("uid", uid);
        StringBuilder values = new StringBuilder();
        for (int i = 0; i < fileIdWithCoordinates.size(); i++) {
            if (i != 0) {
                values.append(",");
            }

            values.append(String.format(template, i + 1));
            parameters.addValue("id_" + (i + 1), Hex.decode(fileIdWithCoordinates.get(i)._1.getValue()));
            parameters.addValue("coord_" + (i + 1), new PGpoint(
                    fileIdWithCoordinates.get(i)._2.getLatitude(), fileIdWithCoordinates.get(i)._2.getLongitude()));
        }

        String query = collectStats(uid)
                + " UPDATE disk.files AS f SET"
                + "     ext_coordinates = v.coordinates"
                + " FROM (VALUES "
                + values
                + " ) AS v(id, coordinates)"
                + " WHERE f.uid = :uid AND f.id = v.id";
        if (skipExisting) {
            query += " AND f.ext_coordinates IS NULL";
        }

        jdbcTemplate(uid).update(query, parameters);
    }

    @Override
    public void setDimensionsWithAngle(DjfsUid uid, UUID id, Dimension dimension, int angle) {
        String q = collectStats(uid)
                + " UPDATE disk.storage_files"
                + "   SET width = :width, height = :height, angle = :angle"
                + " WHERE storage_id = (SELECT storage_id FROM disk.files WHERE uid = :uid AND fid = :fid )";
        Map<String, Object> parameters = Cf.toMap(Tuple2List.fromPairs(
                "uid", uid,
                "fid", id,
                "width", dimension.getWidth(),
                "height", dimension.getHeight(),
                "angle", angle
        ));
        jdbcTemplate(uid).update(q, parameters);
    }

    @Override
    public boolean hasAnyParentWithYarovayaMark(DjfsResourcePath path) {
        String q = collectStats(path.getUid())
                + " WITH RECURSIVE recurse AS (\n"
                + "     SELECT\n"
                + "         1 AS idx,\n"
                + "         dir.name,\n"
                + "         dir.parent_fid,\n"
                + "         dir.fid,\n"
                + "         dir.yarovaya_mark\n"
                + "     FROM\n"
                + "         disk.folders dir\n"
                + "     WHERE\n"
                + "         dir.parent_fid IS NULL\n"
                + "         AND dir.name = (ARRAY[ :all_parent_names ])[1]\n"
                + "         AND dir.uid = :uid\n"
                + "     UNION\n"
                + "     SELECT\n"
                + "         idx + 1,\n"
                + "         child.name,\n"
                + "         child.parent_fid,\n"
                + "         child.fid,\n"
                + "         child.yarovaya_mark\n"
                + "     FROM\n"
                + "         recurse parent JOIN disk.folders child ON parent.fid = child.parent_fid\n"
                + "     WHERE\n"
                + "         uid = :uid\n"
                + "         AND child.name = (ARRAY[ :all_parent_names ])[idx + 1]\n"
                + " )\n"
                + " SELECT 1 as \"found\"\n"
                + " FROM recurse\n"
                + " WHERE yarovaya_mark is TRUE\n"
                + " LIMIT 1";

        Map<String, Object> parameters = Cf.toMap(Tuple2List.fromPairs(
                "uid", path.getUid(),
                "all_parent_names", path.getAllParents().map(DjfsResourcePath::getName)
        ));

        ListF<Integer> found = jdbcTemplate(path.getUid()).query(
                q, (rs, rowNum) -> rs.getInt("found"), parameters);
        return !found.isEmpty() && found.first() == 1;
    }

    @Override
    public void delete(DjfsResourcePath path) {
        String s = collectStats(path)
                + " DELETE FROM disk.folders WHERE uid=:uid AND fid=(SELECT fid FROM code.path_to_fid(:path, :uid))";
        String q = collectStats(path)
                + " DELETE FROM disk.files WHERE uid=:uid AND fid=(SELECT fid FROM code.path_to_fid(:path, :uid))";
        MapF<String, Object> parameters = Cf.map("uid", path.getUid(), "path", path.getPath());
        jdbcTemplate(path.getUid()).update(s, parameters);
        jdbcTemplate(path.getUid()).update(q, parameters);
    }

    @Override
    public ListF<FileDjfsResource> getUserPhotosWithGeoCoordinates(DjfsUid uid) {
        String query = collectStats(uid) +
                " SELECT " + FILE_FIELDS_PREFIXED + ", " + STORAGE_FILE_FIELDS_PREFIXED + ","
                + " (SELECT path FROM code.fid_to_path(files.fid, files.uid)) as path"
                + " FROM disk.files files"
                + " JOIN disk.storage_files storage_files ON files.storage_id = storage_files.storage_id"
                +
                " WHERE files.uid=:uid AND files.media_type::text IN (:mediaTypes) AND files.ext_coordinates IS NOT NULL";
        return jdbcTemplate(uid).query(query, fileMapper, Cf.map("uid", uid,
                "mediaTypes", photoMediaTypes.get()));
    }

    @Override
    public long countAllFiles(DjfsUid uid) {
        String query = collectStats(uid) + " SELECT COUNT(*) FROM disk.files WHERE uid = ?";
        return jdbcTemplate(uid).queryForInt(query, uid.asLong());
    }
}
