package ru.yandex.chemodan.app.djfs.core.filesystem;

import java.util.UUID;

import org.bson.types.ObjectId;
import org.joda.time.Instant;
import org.springframework.jdbc.core.RowMapper;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.djfs.core.db.pg.PgShardedDao;
import ru.yandex.chemodan.app.djfs.core.db.pg.PgShardedDaoContext;
import ru.yandex.chemodan.app.djfs.core.user.DjfsUid;
import ru.yandex.chemodan.app.djfs.core.util.UuidUtils;

public class PgSupportBlockedHidsDao extends PgShardedDao implements SupportBlockedHidsDao {
    public PgSupportBlockedHidsDao(PgShardedDaoContext context) {
        super(context);
    }

    private final static RowMapper<SupportBlockedHid> M = (rs, rowNum) -> SupportBlockedHid.builder()
            .id(new ObjectId(rs.getBytes("id")))
            .block_type(Option.ofNullable(rs.getString("block_type")))
            .hid(UuidUtils.from(rs.getString("storage_id")))
            .ctime(Option.ofNullable(rs.getTimestamp("ctime")).map(Instant::new))
            .build();

    @Override
    public Option<SupportBlockedHid> find(String hid) {
        String sql = collectStats(DjfsUid.COMMON_UID) + " SELECT * FROM disk.support_blocked_hids WHERE storage_id = :hid";
        return jdbcTemplate(DjfsUid.COMMON_UID).queryForOption(sql, M, Cf.map("hid", UuidUtils.fromHex(hid)));
    }

    @Override
    public ListF<SupportBlockedHid> find(ListF<String> hids) {
        ListF<UUID> uuidHids = hids.map(UuidUtils::fromHex);
        if (uuidHids.length() < 1) {
            return Option.empty();
        }
        String sql = collectStats(DjfsUid.COMMON_UID)
                + " SELECT * FROM disk.support_blocked_hids WHERE storage_id IN (:hids)";
        return jdbcTemplate(DjfsUid.COMMON_UID).query(sql, M, Cf.map("hids", uuidHids));
    }
}
