package ru.yandex.chemodan.app.djfs.core.filesystem;

import org.bson.types.ObjectId;
import org.springframework.jdbc.core.RowMapper;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.chemodan.app.djfs.core.db.pg.PgShardedDao;
import ru.yandex.chemodan.app.djfs.core.db.pg.PgShardedDaoContext;
import ru.yandex.chemodan.app.djfs.core.db.pg.ResultSetUtils;
import ru.yandex.chemodan.app.djfs.core.user.DjfsUid;
import ru.yandex.misc.bender.Bender;
import ru.yandex.misc.bender.serialize.BenderJsonSerializer;

public class PgSupportDao extends PgShardedDao implements SupportDao {
    public PgSupportDao(PgShardedDaoContext context) {
        super(context);
    }

    private final static RowMapper<SupportComment> M = (rs, rowNum) -> SupportComment.builder()
            .id(new ObjectId(rs.getBytes("id")))
            .data(ResultSetUtils.getObjectO(rs, "data", JsonSupportData.class).get())
            .dataId(ResultSetUtils.getStringO(rs, "data_id"))
            .dataHash(ResultSetUtils.getStringO(rs, "data_hash"))
            .dataStids(ResultSetUtils.getStringArrayO(rs, "data_stids").getOrNull())
            .uid(DjfsUid.cons(rs.getLong("uid")))
            .build();

    private final static RowMapper<ListF<String>> stidsMapper =
            (rs, rowNum) -> ResultSetUtils.getStringArrayO(rs, "data_stids").getOrNull();

    private final static BenderJsonSerializer<JsonSupportData> jsonSupportDataSerializer = Bender.jsonSerializer(JsonSupportData.class);

    @Override
    public Option<SupportComment> find(String hash) {
        String sql = collectStats(DjfsUid.COMMON_UID) + " SELECT * FROM disk.support_mpfs WHERE data_hash = :hash";
        return jdbcTemplate(DjfsUid.COMMON_UID).queryForOption(sql, M, Cf.map("hash", hash));
    }

    @Override
    public SetF<String> getAllBlockedStids() {
        SetF<String> blockedStids = Cf.hashSet();
        String sql = collectStats(DjfsUid.COMMON_UID)
                + " SELECT data_stids FROM disk.support_mpfs WHERE data_stids IS NOT NULL";
        Iterable<ListF<String>> cursor = jdbcTemplate(DjfsUid.COMMON_UID).query(sql, stidsMapper);
        cursor.forEach(blockedStids::addAll);
        return blockedStids;
    }

    @Override
    public void insert(SupportComment supportComment) {
        String sql = collectStats(supportComment.getUid())
                + " INSERT INTO disk.support_mpfs (id, data, data_id, data_hash, data_stids, uid)"
                + " VALUES(:id, :data::json, :data_id, :data_hash, :data_stids::text[], :uid)"
                + " ON CONFLICT (id) DO NOTHING";

        jdbcTemplate(DjfsUid.COMMON_UID).update(
                sql, Cf.toMap(Tuple2List.fromPairs(
                    "id", supportComment.getId().toByteArray(),
                    "data", new String(jsonSupportDataSerializer.serializeJson(supportComment.getData())),
                    "data_id", supportComment.getDataId(),
                    "data_hash", supportComment.getDataHash(),
                    "data_stids", supportComment.getDataStids().mkString("{", ",", "}"),
                    "uid", supportComment.getUid()
                ))
        );
    }
}
