package ru.yandex.chemodan.app.djfs.core.filesystem;

import java.time.Instant;
import java.time.temporal.ChronoUnit;

import com.mongodb.ReadPreference;
import lombok.RequiredArgsConstructor;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.bolts.internal.NotImplementedException;
import ru.yandex.chemodan.app.djfs.core.billing.BillingManager;
import ru.yandex.chemodan.app.djfs.core.billing.BillingProduct;
import ru.yandex.chemodan.app.djfs.core.db.DjfsUidSource;
import ru.yandex.chemodan.app.djfs.core.diskinfo.DiskInfoDao;
import ru.yandex.chemodan.app.djfs.core.diskinfo.DownloadTraffic;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.DjfsResourceArea;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.DjfsResourcePath;
import ru.yandex.chemodan.app.djfs.core.legacy.exception.LegacyUserIsReadOnlyException;
import ru.yandex.chemodan.app.djfs.core.notification.EmailGenerator;
import ru.yandex.chemodan.app.djfs.core.notification.XivaPushGenerator;
import ru.yandex.chemodan.app.djfs.core.share.GroupDao;
import ru.yandex.chemodan.app.djfs.core.share.ShareInfo;
import ru.yandex.chemodan.app.djfs.core.user.DjfsUid;
import ru.yandex.chemodan.app.djfs.core.user.Organization;
import ru.yandex.chemodan.app.djfs.core.user.OrganizationDao;
import ru.yandex.chemodan.app.djfs.core.user.UserDao;
import ru.yandex.chemodan.app.djfs.core.user.UserData;
import ru.yandex.chemodan.app.djfs.core.util.ByteConstants;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

/**
 * @author eoshch
 */
@RequiredArgsConstructor
public class QuotaManager {
    private static final Logger logger = LoggerFactory.getLogger(QuotaManager.class);
    private static final SetF<String> B2B_IGNORED_PRODUCTS = Cf.set("initial_10gb", "b2b_10gb", "initial_3gb",
            "promo_shared", "file_uploaded", "app_install");

    private final UserDao userDao;
    private final DiskInfoDao diskInfoDao;
    private final GroupDao groupDao;
    private final OrganizationDao organizationDao;
    private final XivaPushGenerator xivaPushGenerator;
    private final EmailGenerator emailGenerator;
    private final BillingManager billingManager;

    public long getTotalUsed(DjfsUid uid) {
        return Math.max(0, diskInfoDao.findTotalUsed(uid).getOrElse(0L));
    }

    public long getTrashUsed(DjfsUid uid) {
        return Math.max(0, diskInfoDao.findTrashUsed(uid).getOrElse(0L));
    }

    public long getLimit(DjfsUid uid, Option<ReadPreference> readPreference) {
        // b2b organization users have shared limit
        // https://st.yandex-team.ru/CHE-122

        Option<String> organizationId = userDao.find(uid).filterMap(UserData::getB2bKey);
        if (organizationId.isPresent()) {
            Option<Organization> organization = organizationDao.find(organizationId.get());
            if (organization.isPresent() && organization.get().isPaid) {
                long used = diskInfoDao.findTotalUsed(uid).getOrElse(0L);
                long bought = billingManager.findProducts(uid, readPreference)
                        .filter(x -> !B2B_IGNORED_PRODUCTS.containsTs(x.getProductId()))
                        .map(BillingProduct::getSpaceAmount).reduceLeftO(Long::sum).getOrElse(0L);

                return Math.max(organization.get().quotaFree, 0) + Math.max(bought - used, 0) + used;
            }
        }

        return diskInfoDao.findLimit(uid).getOrElse(0L);
    }

    // todo: move to test project
    public void setLimit(DjfsUid uid, long limit) {
        diskInfoDao.setLimit(uid, limit);
    }

    public long getFree(DjfsUid uid, Option<ReadPreference> readPreference) {
        return getLimit(uid, readPreference) - getTotalUsed(uid);
    }

    public long getFree(DjfsUidSource uidSource, Option<ReadPreference> readPreference) {
        return getFree(uidSource.getUid(), readPreference);
    }

    public void incrementAreaCounter(DjfsUid uid, DjfsResourceArea area, long delta) {
        if (area == DjfsResourceArea.TRASH) {
            diskInfoDao.incrementTrashUsed(uid, delta);
            return;
        }
        throw new NotImplementedException();
    }

    public void incrementUsed(DjfsResourcePath path, Option<ShareInfo> shareInfo, long delta) {
        if (path.getArea() == DjfsResourceArea.TRASH) {
            diskInfoDao.incrementTrashUsed(path.getUid(), delta);
        }

        if (path.getArea() == DjfsResourceArea.DISK || path.getArea() == DjfsResourceArea.TRASH) {
            diskInfoDao.incrementTotalUsed(path.getUid(), delta);
        }

        if (shareInfo.isPresent()) {
            // Сейчас честный шаринг и размер группы хранится только для статистики по историческим причинам
            try {
                groupDao.increaseSize(shareInfo.get().getGroupId(), delta);
            } catch (LegacyUserIsReadOnlyException e) {
                // https://st.yandex-team.ru/CHEMODAN-75397 Не ломать операции перемещения, если группы в ридонли
                logger.error("groups.size increment failed (gid: " + shareInfo.get().getGroupId() + ")", e);
            }
        }
    }

    public void notifyIfLowSpace(DjfsUid uid, Option<ShareInfo> shareInfo) {
        if (shareInfo.isPresent()) {
            xivaPushGenerator.sendSpaceGroupPush(uid, shareInfo.get());
            return;
        }

        long free = getFree(uid, Option.empty());

        if (free < ByteConstants.GiB) {
            long limit = getLimit(uid, Option.empty());

            // for most users limit is at least 10GiB so this is almost always false
            if (free > limit * 0.1) {
                return;
            }

            long used = getTotalUsed(uid);
            QuotaInfo quotaInfo = new QuotaInfo(limit, used, getTrashUsed(uid), free);
            long fileCount = diskInfoDao.findFileCount(uid).getOrElse(0L);

            if (free < ByteConstants.MiB) {
                xivaPushGenerator.sendSpaceIsFullPush(uid, quotaInfo);
                if (userDao.find(uid).map(UserData::isB2b).getOrElse(false)) {
                    emailGenerator.sendNoFreeSpaceB2bMail(uid, quotaInfo, fileCount);
                } else {
                    emailGenerator.sendNoFreeSpaceMail(uid, quotaInfo, fileCount);
                }
            } else {
                xivaPushGenerator.sendSpaceIsLowPush(uid, quotaInfo);
                emailGenerator.sendLowFreeSpaceMail(uid, quotaInfo, fileCount);
            }
        }
    }

    public Option<Long> getDownloadSpeedLimitEndTime(DjfsUid uid) {
        final Option<DownloadTraffic> downloadTraffic = diskInfoDao.findDownloadTraffic(uid);
        Option<Long> result = Option.empty();
        if (downloadTraffic.isPresent()) {
            final long limit = getLimit(uid, Option.of(ReadPreference.secondaryPreferred()));
            final long dayBefore = Instant.now().minus(1L, ChronoUnit.DAYS).getEpochSecond();
            final long ctime = downloadTraffic.get().ctime;
            final long size = downloadTraffic.get().bytes;
            if (ctime > dayBefore && size > 2 * limit) {
                result = Option.of(Instant.ofEpochSecond(ctime).plus(1L, ChronoUnit.DAYS).getEpochSecond());
            }
        }
        return result;
    }
}
