package ru.yandex.chemodan.app.djfs.core.filesystem.iteration;

import java.util.UUID;

import lombok.AccessLevel;
import lombok.AllArgsConstructor;
import lombok.Builder;
import lombok.Getter;
import lombok.Value;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.bolts.internal.NotImplementedException;
import ru.yandex.chemodan.app.djfs.core.filesystem.DjfsResourceDao;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.DjfsFileId;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.DjfsResource;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.DjfsResourceArea;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.DjfsResourcePath;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.FileDjfsResource;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.FolderDjfsResource;
import ru.yandex.chemodan.app.djfs.core.user.DjfsUid;
import ru.yandex.misc.bender.annotation.BenderBindAllFields;
import ru.yandex.misc.lang.Check;
import ru.yandex.misc.lang.Validate;

/**
 * Traverse filesystem.
 * <p>
 * Does not support shared folders.
 *
 * @author eoshch
 */
public class ResourceIterator {
    private final DjfsUid uid;
    private final DjfsResourceArea area;
    private final DjfsResourcePath iterationRootPath;
    private final TraversalType traversalType;

    private final DjfsResourceDao djfsResourceDao;

    private ResourceIterator(DjfsResourcePath iterationRootPath, TraversalType traversalType,
            DjfsResourceDao djfsResourceDao)
    {
        this.iterationRootPath = iterationRootPath;
        this.uid = iterationRootPath.getUid();
        this.area = iterationRootPath.getArea();
        this.traversalType = traversalType;
        this.djfsResourceDao = djfsResourceDao;
    }

    public ResourceIteratorState initialize() {
        Option<DjfsResource> iterationRootO = djfsResourceDao.find(iterationRootPath);
        if (!iterationRootO.isPresent()) {
            return ResourceIteratorState.finished();
        }

        DjfsResource iterationRoot = iterationRootO.get();
        if (!(iterationRoot instanceof FolderDjfsResource)) {
            return ResourceIteratorState.finished();
        }

        return ResourceIteratorState.builder()
                .stack(Cf.list(ResourceIteratorState.FolderData.cons((FolderDjfsResource) iterationRoot)))
                .build();
    }

    public Tuple2<ResourceIteratorState, ListF<DjfsResource>> next(ResourceIteratorState state, int limit) {
        return next(state, limit, false);
    }

    /**
     * Tries to return the desired count of resources but it is not always possible.
     * Iteration is done based on non-unique fileId field.
     * All files with the same fileId are always returned in the same group.
     */
    public Tuple2<ResourceIteratorState, ListF<DjfsResource>> next(ResourceIteratorState state, int limit,
            boolean lockForUpdate)
    {
        Validate.gt(limit, 0);

        // todo: cache shared folders info within one request/task handler for at least 1 minute

        // todo: cache found folders for iteration?
        // todo: smarter iteration - do not return empty results when state is changed?

        if (state.isFinished()) {
            return Tuple2.tuple(state, Cf.list());
        }

        ListF<ResourceIteratorState.FolderData> stack = state.getStack();
        // todo: stack empty: iterate from start or do not iterate at all?
        if (stack.isEmpty()) {
            return Tuple2.tuple(ResourceIteratorState.finished(), Cf.list());
        }

        ResourceIteratorState nextState;
        ListF<DjfsResource> result;

        ResourceIteratorState.FolderData current = stack.last();

        switch (current.getIterationType()) {
            case FOLDER:
                ListF<FolderDjfsResource> childFolders = djfsResourceDao.find2ImmediateChildFoldersOrderByFileIdThenId(
                        uid, area, current.getId(), current.getIterationFileId(), current.getIterationId(), 1,
                        lockForUpdate);

                Check.le(childFolders.length(), 1, "limit is set to 1 so this should be impossible");

                // if there are folders without FileId, set it
                if (childFolders.length() > 0 && !childFolders.first().getFileId().isPresent()) {
                    ensureAllChildFoldersHaveFileId(uid, area, current.getId());
                    childFolders = djfsResourceDao.find2ImmediateChildFoldersOrderByFileIdThenId(
                            uid, area, current.getId(), current.getIterationFileId(), current.getIterationId(), 1,
                            lockForUpdate);

                    Check.le(childFolders.length(), 1, "limit is set to 1 so this should be impossible");
                }

                if (childFolders.length() > 0) {
                    FolderDjfsResource nextChild = childFolders.first();
                    nextState = state.startIteratingSubfolder(nextChild);
                    result = traversalType == TraversalType.TOP_DOWN ? Cf.list(nextChild) : Cf.list();
                    break;
                } else {
                    nextState = state.startIteratingFiles();
                    result = Cf.list();
                    break;
                }
            case FILE:
                ListF<FileDjfsResource> childFiles = djfsResourceDao.find2ImmediateChildFilesOrderByFileIdThenId(
                        uid, area, current.getId(), current.getIterationFileId(), current.getIterationId(), limit,
                        lockForUpdate);

                Check.le(childFiles.length(), limit, "query returned more than limit entities");

                if (childFiles.length() == limit) {
                    nextState = state.continueIteratingFiles(childFiles.last());
                    result = childFiles.cast();
                    break;
                } else {
                    nextState = state.finishIteratingFiles();
                    result = childFiles.cast();
                    if (traversalType == TraversalType.BOTTOM_UP && !nextState.isFinished()) {
                        Option<DjfsResource> traversedFolder = djfsResourceDao.find2(uid, area, current.getId(),
                                lockForUpdate);
                        if (traversedFolder.isPresent()) {
                            result = childFiles.<DjfsResource>cast().plus(traversedFolder.get());
                        }
                    }
                    break;
                }
            default:
                throw new NotImplementedException();
        }

        return Tuple2.tuple(nextState, result);
    }

    public boolean hasNext(ResourceIteratorState state) {
        return !state.isFinished();
    }

    private void ensureAllChildFoldersHaveFileId(DjfsUid uid, DjfsResourceArea area, UUID parentId) {
        ListF<FolderDjfsResource> resources = djfsResourceDao.find2ImmediateChildFoldersWithoutFileId(uid, area,
                parentId);
        for (FolderDjfsResource resource : resources) {
            djfsResourceDao.addFileId(resource.getPath(), DjfsFileId.random());
        }
    }

    public enum TraversalType {
        BOTTOM_UP, TOP_DOWN
    }

    @AllArgsConstructor
    public static class Factory {
        private final DjfsResourceDao djfsResourceDao;

        public ResourceIterator create(DjfsResourcePath iterationRoot, TraversalType traversalType) {
            return new ResourceIterator(iterationRoot, traversalType, djfsResourceDao);
        }
    }

    @Builder(toBuilder = true)
    @Getter(AccessLevel.PRIVATE)
    @BenderBindAllFields
    public static class ResourceIteratorState {
        @Builder.Default
        private final ListF<FolderData> stack = Cf.list();
        @Builder.Default
        private final ListF<FolderData> iterationRootParents = Cf.list();
        @Builder.Default
        private final boolean finished = false;

        private ResourceIteratorState startIteratingSubfolder(FolderDjfsResource folder) {
            UUID id = folder.getId();
            String fileId = folder.getResourceId().get().getFileId().getValue();
            FolderData currentData = stack.last().toBuilder()
                    .iterationType(DjfsResource.Type.FOLDER)
                    .iterationFileId(Option.of(fileId))
                    .iterationId(Option.of(id))
                    .build();
            FolderData childData = ResourceIteratorState.FolderData.cons(folder);

            ListF<ResourceIteratorState.FolderData> nextStack = stack.subList(0, stack.length() - 1)
                    .plus(currentData)
                    .plus(childData);

            return toBuilder().stack(nextStack).build();
        }

        ResourceIteratorState startIteratingFiles() {
            FolderData currentData = stack.last().toBuilder()
                    .iterationType(DjfsResource.Type.FILE)
                    .iterationId(Option.empty())
                    .iterationFileId(Option.empty())
                    .build();

            ListF<ResourceIteratorState.FolderData> nextStack = stack.subList(0, stack.length() - 1)
                    .plus(currentData);

            return toBuilder().stack(nextStack).build();
        }

        ResourceIteratorState continueIteratingFiles(FileDjfsResource file) {
            String fileId = file.getResourceId().get().getFileId().getValue();
            FolderData currentData = stack.last().toBuilder()
                    .iterationType(DjfsResource.Type.FILE)
                    .iterationId(Option.of(file.getId()))
                    .iterationFileId(Option.of(fileId))
                    .build();

            ListF<ResourceIteratorState.FolderData> nextStack = stack.subList(0, stack.length() - 1)
                    .plus(currentData);

            return toBuilder().stack(nextStack).build();
        }

        ResourceIteratorState finishIteratingFiles() {
            if (stack.length() == 1) {
                return ResourceIteratorState.finished();
            }

            FolderData currentData = stack.last();
            FolderData penultimate = stack.get(stack.length() - 2);
            FolderData nextData;
            if (penultimate.getIterationId().isPresent()) {
                nextData = penultimate.toBuilder()
                        .iterationId(Option.of(currentData.getId()))
                        .build();
            } else {
                nextData = penultimate.toBuilder()
                        .iterationFileId(Option.of(currentData.getFileId()))
                        .build();
            }
            ListF<ResourceIteratorState.FolderData> nextStack = stack.subList(0, stack.length() - 2)
                    .plus(nextData);

            return toBuilder().stack(nextStack).build();
        }

        private static ResourceIteratorState finished() {
            return ResourceIteratorState.builder().finished(true).build();
        }

        @Value
        @Builder(toBuilder = true)
        @BenderBindAllFields
        private static class FolderData {
            private final UUID id;
            private final UUID parentId;
            private final String name;
            private final String fileId;

            private final DjfsResource.Type iterationType;
            @Builder.Default
            private final Option<String> iterationFileId = Option.empty();
            @Builder.Default
            private final Option<UUID> iterationId = Option.empty();

            private static FolderData cons(FolderDjfsResource resource) {
                return FolderData.builder()
                        .id(resource.getId())
                        .parentId(resource.getParentId().get())
                        .fileId(resource.getResourceId().get().getFileId().getValue())
                        .name(resource.getPath().getName())
                        .iterationType(DjfsResource.Type.FOLDER)
                        .build();
            }
        }
    }
}
