package ru.yandex.chemodan.app.djfs.core.filesystem.model;

import lombok.Value;

import ru.yandex.bolts.internal.NotImplementedException;
import ru.yandex.chemodan.app.djfs.core.ActionContext;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.exception.InvalidClientInputDjfsFileIdException;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.exception.InvalidDjfsFileIdException;
import ru.yandex.chemodan.app.djfs.core.util.UuidUtils;
import ru.yandex.misc.lang.Assume;
import ru.yandex.misc.lang.StringUtils;

/**
 * @author eoshch
 */
@Value
public class DjfsFileId {
    private static final int FILE_ID_LENGTH = 64;

    private static final Assume INTERNAL_ASSUME = new Assume() {
        @Override
        public void fail(String message) {
            throw new InvalidDjfsFileIdException(message);
        }
    };

    private static final Assume CLIENT_ASSUME = new Assume() {
        @Override
        public void fail(String message) {
            throw new InvalidClientInputDjfsFileIdException(message);
        }
    };

    private final String value;

    private DjfsFileId(String fileId) {
        this.value = fileId;
    }

    public static DjfsFileId random() {
        return new DjfsFileId(UuidUtils.randomToHexString() + UuidUtils.randomToHexString());
    }

    public static DjfsFileId cons(String fileId) {
        return cons(fileId, INTERNAL_ASSUME);
    }

    public static DjfsFileId cons(String fileId, ActionContext context) {
        switch (context) {
            case INTERNAL:
                return cons(fileId, INTERNAL_ASSUME);
            case CLIENT_INPUT:
                return cons(fileId, CLIENT_ASSUME);
        }
        throw new NotImplementedException();
    }

    public static DjfsFileId cons(String fileId, Assume assume) {
        assume.notNull(fileId);
        assume.equals(fileId.length(), FILE_ID_LENGTH, "Incorrect value length: " + fileId);
        assume.isTrue(StringUtils.isAlphanumeric(fileId), "Wrong value format: " + fileId);
        return new DjfsFileId(fileId);
    }
}
