package ru.yandex.chemodan.app.djfs.core.filesystem.model;

import java.util.UUID;

import lombok.Getter;
import lombok.experimental.Accessors;
import org.joda.time.Instant;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.bolts.internal.NotImplementedException;
import ru.yandex.chemodan.app.djfs.core.db.DjfsUidSource;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.exception.InvalidDjfsResourceException;
import ru.yandex.chemodan.app.djfs.core.legacy.formatting.Blockings;
import ru.yandex.chemodan.app.djfs.core.user.DjfsUid;
import ru.yandex.misc.lang.Assume;
import ru.yandex.misc.lang.DefaultObject;

/**
 * Именования в разных системах:
 * <p>
 * |   Java   |    Mongo     |     Pg
 * +----------+--------------+------------
 * |    id    |     _id      |    fid
 * | parentId |    parent    | parent_fid
 * |  fileId  | data.file_id |     id
 *
 * @author eoshch
 */
@Getter
public abstract class DjfsResource extends DefaultObject implements DjfsUidSource {
    private final UUID id;
    private final Option<UUID> parentId;
    private final Option<DjfsFileId> fileId;

    private final DjfsResourcePath path;

    private final Option<DjfsUid> modifyUid;
    private final Option<Long> version;

    private final Option<Instant> uploadTime;
    private final Option<Instant> trashAppendTime;
    private final Option<Instant> hiddenAppendTime;

    private final Option<String> trashAppendOriginalPath;

    private final boolean isVisible;
    private final boolean isPublic;
    private final boolean isBlocked;
    private final boolean isPublished;
    @Accessors(fluent = true)
    private final boolean hasYarovayaMark;

    private final Option<String> publicHash;
    private final Option<String> shortUrl;
    private final Option<String> symlink;

    private final Option<String> folderUrl;
    private final Option<Integer> downloadCounter;
    private final Option<String> customProperties;
    private final MapF<String, String> externalProperties;

    private final Option<Integer> pageBlockedItemsNum;
    private final Option<Blockings> blockings;

    private final Option<DjfsResourceArea> area;

    private final Option<String> fsSymbolycLink;

    private static final Assume A = new Assume() {
        @Override
        public void fail(String message) {
            throw new InvalidDjfsResourceException(message);
        }
    };

    protected DjfsResource(Builder<? extends Builder, ? extends DjfsResource> builder) {
        A.notNull(builder.id, "id is null");
        A.notNull(builder.parentId, "parentId is null");
        A.notNull(builder.fileId, "fileId is null");
        A.notNull(builder.path, "path is null");
        A.notNull(builder.modifyUid, "modifyUid is null");
        A.notNull(builder.version, "version is null");
        A.notNull(builder.uploadTime, "uploadTime is null");
        A.notNull(builder.trashAppendTime, "trashAppendTime is null");
        A.notNull(builder.hiddenAppendTime, "hiddenAppendTime is null");
        A.notNull(builder.trashAppendOriginalPath, "trashAppendOriginalPath is null");
        A.notNull(builder.publicHash, "publicHash is null");
        A.notNull(builder.shortUrl, "shortUrl is null");
        A.notNull(builder.symlink, "symlink is null");
        A.notNull(builder.folderUrl, "folderUrl is null");
        A.notNull(builder.downloadCounter, "downloadCounter is null");
        A.notNull(builder.customProperties, "customProperties is null");
        A.notNull(builder.externalProperties, "externalProperties is null");
        A.notNull(builder.fsSymbolycLink, "fsSymbolycLink is null");

        this.id = builder.id;
        this.parentId = builder.parentId;
        this.fileId = builder.fileId;
        this.path = builder.path;
        this.modifyUid = builder.modifyUid;
        this.version = builder.version;
        this.uploadTime = builder.uploadTime;
        this.trashAppendTime = builder.trashAppendTime;
        this.hiddenAppendTime = builder.hiddenAppendTime;
        this.trashAppendOriginalPath = builder.trashAppendOriginalPath;
        this.isVisible = builder.isVisible;
        this.isPublic = builder.isPublic;
        this.isBlocked = builder.isBlocked;
        this.isPublished = builder.isPublished;
        this.hasYarovayaMark = builder.hasYarovayaMark;
        this.publicHash = builder.publicHash;
        this.shortUrl = builder.shortUrl;
        this.symlink = builder.symlink;
        this.folderUrl = builder.folderUrl;
        this.downloadCounter = builder.downloadCounter;
        this.customProperties = builder.customProperties;
        this.externalProperties = builder.externalProperties.unmodifiable();
        this.pageBlockedItemsNum = builder.pageBlockedItemsNum;
        this.blockings = builder.blockings;
        this.area = builder.area;
        this.fsSymbolycLink = builder.fsSymbolycLink;
    }

    protected <T extends Builder<T, U>, U extends DjfsResource> T initializeBuilder(T builder) {
        return builder.id(this.id)
                .parentId(this.parentId)
                .fileId(this.fileId)
                .path(this.path)
                .modifyUid(this.modifyUid)
                .version(this.version)
                .uploadTime(this.uploadTime)
                .trashAppendTime(this.trashAppendTime)
                .hiddenAppendTime(this.hiddenAppendTime)
                .trashAppendOriginalPath(this.trashAppendOriginalPath)
                .isVisible(this.isVisible)
                .isPublic(this.isPublic)
                .isBlocked(this.isBlocked)
                .isPublished(this.isPublished)
                .hasYarovayaMark(this.hasYarovayaMark)
                .publicHash(this.publicHash)
                .shortUrl(this.shortUrl)
                .symlink(this.symlink)
                .folderUrl(this.folderUrl)
                .downloadCounter(this.downloadCounter)
                .customProperties(this.customProperties)
                .externalProperties(Cf.toHashMap(this.externalProperties))
                .pageBlockedItemsNum(this.pageBlockedItemsNum)
                .blockings(this.blockings)
                .area(this.area)
                .fsSymbolicLink(this.fsSymbolycLink);
    }

    public Option<DjfsResourceId> getResourceId() {
        return getFileId().map(x -> DjfsResourceId.cons(getUid(), x));
    }

    public abstract Builder<? extends Builder, ? extends DjfsResource> toBuilder();

    @Override
    public DjfsUid getUid() {
        return path.getUid();
    }

    public boolean isFullyPublic() {
        return isPublic() && getSymlink().isPresent() && getPublicHash().isPresent() && getShortUrl().isPresent();
    }

    public boolean hasAnyPublicField() {
        return getSymlink().isPresent() || isPublic() || getPublicHash().isPresent() || getShortUrl().isPresent();
    }

    public abstract Option<Instant> getCreationTimeO();
    public abstract Option<Instant> getModificationTimeO();
    public abstract boolean hasPhotosliceTime();

    public enum Type {
        FILE,
        FOLDER,
        ;

        public static Type cons(DjfsResource resource) {
            if (resource instanceof FolderDjfsResource) {
                return FOLDER;
            } else if (resource instanceof FileDjfsResource) {
                return FILE;
            } else {
                throw new NotImplementedException();
            }
        }
    }

    @Getter
    public static abstract class Builder<T extends Builder, U extends DjfsResource> {
        private UUID id;
        private Option<UUID> parentId = Option.empty();
        private Option<DjfsFileId> fileId = Option.empty();
        private DjfsResourcePath path;
        private Option<DjfsUid> modifyUid = Option.empty();
        private Option<Long> version = Option.empty();
        private Option<Instant> uploadTime = Option.empty();
        private Option<Instant> trashAppendTime = Option.empty();
        private Option<Instant> hiddenAppendTime = Option.empty();
        private Option<String> trashAppendOriginalPath = Option.empty();
        private boolean isVisible = true;
        private boolean isPublic = false;
        private boolean isBlocked = false;
        private boolean isPublished = false;
        private boolean hasYarovayaMark = false;
        private Option<String> publicHash = Option.empty();
        private Option<String> shortUrl = Option.empty();
        private Option<String> symlink = Option.empty();
        private Option<String> folderUrl = Option.empty();
        private Option<Integer> downloadCounter = Option.empty();
        private Option<String> customProperties = Option.empty();
        private MapF<String, String> externalProperties = Cf.hashMap();
        private Option<Integer> pageBlockedItemsNum = Option.empty();
        private Option<Blockings> blockings = Option.empty();
        private Option<DjfsResourceArea> area = Option.empty();
        private Option<String> fsSymbolycLink = Option.empty();

        public T id(UUID id) {
            this.id = id;
            return getThis();
        }

        public T parentId(Option<UUID> parentId) {
            this.parentId = parentId;
            return getThis();
        }

        public T parentId(UUID parentId) {
            this.parentId = Option.of(parentId);
            return getThis();
        }

        public Option<UUID> getParentId() {
            return this.parentId;
        }

        public T fileId(DjfsFileId fileId) {
            this.fileId = Option.of(fileId);
            return getThis();
        }

        public T fileId(Option<DjfsFileId> fileId) {
            this.fileId = fileId;
            return getThis();
        }

        public T path(DjfsResourcePath path) {
            this.path = path;
            return getThis();
        }

        public DjfsResourcePath getPath() {
            return this.path;
        }

        public T modifyUid(Option<DjfsUid> modifyUid) {
            this.modifyUid = modifyUid;
            return getThis();
        }

        public T modifyUid(DjfsUid modifyUid) {
            this.modifyUid = Option.of(modifyUid);
            return getThis();
        }

        public T version(Option<Long> version) {
            this.version = version;
            return getThis();
        }

        public T version(long version) {
            this.version = Option.of(version);
            return getThis();
        }

        public abstract T creationTime(Instant creationTime);
        public abstract T modificationTime(Instant modificationTime);

        public T uploadTime(Option<Instant> uploadTime) {
            this.uploadTime = uploadTime;
            return getThis();
        }

        public T uploadTime(Instant uploadTime) {
            this.uploadTime = Option.of(uploadTime);
            return getThis();
        }

        public T trashAppendTime(Option<Instant> trashAppendTime) {
            this.trashAppendTime = trashAppendTime;
            return getThis();
        }

        public T trashAppendTime(Instant trashAppendTime) {
            this.trashAppendTime = Option.of(trashAppendTime);
            return getThis();
        }

        public T hiddenAppendTime(Option<Instant> hiddenAppendTime) {
            this.hiddenAppendTime = hiddenAppendTime;
            return getThis();
        }

        public T hiddenAppendTime(Instant hiddenAppendTime) {
            this.hiddenAppendTime = Option.of(hiddenAppendTime);
            return getThis();
        }

        public T trashAppendOriginalPath(Option<String> trashAppendOriginalPath) {
            this.trashAppendOriginalPath = trashAppendOriginalPath;
            return getThis();
        }

        public T trashAppendOriginalPath(String trashAppendOriginalPath) {
            this.trashAppendOriginalPath = Option.ofNullable(trashAppendOriginalPath);
            return getThis();
        }

        public T isVisible(boolean isVisible) {
            this.isVisible = isVisible;
            return getThis();
        }

        public T isPublic(boolean isPublic) {
            this.isPublic = isPublic;
            return getThis();
        }

        public T isBlocked(boolean isBlocked) {
            this.isBlocked = isBlocked;
            return getThis();
        }

        public T isPublished(boolean isPublished) {
            this.isPublished = isPublished;
            return getThis();
        }

        public T hasYarovayaMark(boolean hasYarovayaMark) {
            this.hasYarovayaMark = hasYarovayaMark;
            return getThis();
        }

        public T publicHash(Option<String> publicHash) {
            this.publicHash = publicHash;
            return getThis();
        }

        public T publicHash(String publicHash) {
            this.publicHash = Option.ofNullable(publicHash);
            return getThis();
        }

        public T shortUrl(Option<String> shortUrl) {
            this.shortUrl = shortUrl;
            return getThis();
        }

        public T shortUrl(String shortUrl) {
            this.shortUrl = Option.ofNullable(shortUrl);
            return getThis();
        }

        public T symlink(Option<String> symlink) {
            this.symlink = symlink;
            return getThis();
        }

        public T symlink(String symlink) {
            this.symlink = Option.ofNullable(symlink);
            return getThis();
        }

        public T folderUrl(Option<String> folderUrl) {
            this.folderUrl = folderUrl;
            return getThis();
        }

        public T folderUrl(String folderUrl) {
            this.folderUrl = Option.ofNullable(folderUrl);
            return getThis();
        }

        public T area(Option<DjfsResourceArea> area) {
            this.area = area;
            return getThis();
        }

        public T area(DjfsResourceArea area) {
            this.area = Option.ofNullable(area);
            return getThis();
        }

        public T downloadCounter(Option<Integer> downloadCounter) {
            this.downloadCounter = downloadCounter;
            return getThis();
        }

        public T downloadCounter(int downloadCounter) {
            this.downloadCounter = Option.of(downloadCounter);
            return getThis();
        }

        public T pageBlockedItemsNum(Option<Integer> pageBlockedItemsNum) {
            this.pageBlockedItemsNum = pageBlockedItemsNum;
            return getThis();
        }

        public T pageBlockedItemsNum(int pageBlockedItemsNum) {
            this.pageBlockedItemsNum = Option.of(pageBlockedItemsNum);
            return getThis();
        }

        public T blockings(Option<Blockings> blockings) {
            this.blockings = blockings;
            return getThis();
        }

        public T blockings(Blockings blockings) {
            this.blockings = Option.of(blockings);
            return getThis();
        }

        public T customProperties(Option<String> customProperties) {
            this.customProperties = customProperties;
            return getThis();
        }

        public T customProperties(String customProperties) {
            this.customProperties = Option.ofNullable(customProperties);
            return getThis();
        }

        public T externalProperty(String name, String value) {
            this.externalProperties.put(name, value);
            return getThis();
        }

        public T externalProperty(String name, Option<String> value) {
            if (value.isPresent()) {
                this.externalProperties.put(name, value.get());
            }
            return getThis();
        }

        public T externalProperty(Tuple2<String, String> entry) {
            this.externalProperties.put(entry);
            return getThis();
        }

        public T externalProperties(MapF<String, String> externalProperties) {
            this.externalProperties = externalProperties;
            return getThis();
        }

        public T fsSymbolicLink(Option<String> fsSymbolycLink) {
            this.fsSymbolycLink = fsSymbolycLink;
            return getThis();
        }

        public abstract T getThis();

        public abstract U build();
    }

    public String getDisplayName() {
        Option<String> originalName = getExternalProperties().getO("original_name");
        if (getPath().getArea() == DjfsResourceArea.ATTACH && originalName.isPresent()) {
            return originalName.get();
        }

        if (getPath().getArea() == DjfsResourceArea.TRASH && trashAppendOriginalPath.isPresent()) {
            return DjfsResourcePath.cons(getUid(), trashAppendOriginalPath.get()).getName();
        }

        return getPath().getName();
    }

}
