package ru.yandex.chemodan.app.djfs.core.filesystem.model;

import java.util.function.Supplier;

import com.mongodb.ReadPreference;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.internal.NotImplementedException;
import ru.yandex.chemodan.app.djfs.core.filesystem.Activity;
import ru.yandex.chemodan.app.djfs.core.filesystem.CopyIntent;
import ru.yandex.chemodan.app.djfs.core.filesystem.DjfsPrincipal;
import ru.yandex.chemodan.app.djfs.core.filesystem.DjfsResourceDao;
import ru.yandex.chemodan.app.djfs.core.filesystem.FilesystemAccessController;
import ru.yandex.chemodan.app.djfs.core.filesystem.MoveIntent;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.area.FolderAreaDiskMetaProvider;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.area.FolderAreaMetaProvider;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.area.FolderAreaTrashMetaProvider;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.area.ResourceAreaDiskMetaProvider;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.area.ResourceAreaMetaProvider;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.exception.InvalidDjfsResourceAreaException;
import ru.yandex.chemodan.app.djfs.core.share.ShareInfo;
import ru.yandex.chemodan.app.djfs.core.share.ShareInfoManager;
import ru.yandex.chemodan.app.djfs.core.user.DjfsUid;
import ru.yandex.chemodan.app.djfs.core.util.ByteConstants;
import ru.yandex.misc.enums.StringEnum;
import ru.yandex.misc.enums.StringEnumResolver;
import ru.yandex.misc.lang.StringUtils;

/**
 * @author eoshch
 */
public enum DjfsResourceArea implements StringEnum {
    DISK("disk", "user_data", new ResourceAreaDiskMetaProvider(), new FolderAreaDiskMetaProvider()) {
        @Override
        public boolean isChangelogWritten() {
            return true;
        }

        @Override
        public boolean isUserIndexVersionUpdated() {
            return true;
        }

        @Override
        public Option<DjfsResource> find(DjfsPrincipal principal, DjfsResourcePath path, ShareInfoManager shareInfoManager,
                                         FilesystemAccessController filesystemAccessController, DjfsResourceDao djfsResourceDao,
                                         Option<ReadPreference> readPreference) {
            Option<ShareInfo> shareInfoO = shareInfoManager.get(path, readPreference);
            filesystemAccessController.checkReadResource(principal, path, shareInfoO);
            DjfsResourcePath correctPath = shareInfoO.filterMap(x -> x.participantPathToOwnerPath(path)).getOrElse(path);
            return djfsResourceDao.find(correctPath);
        }

        @Override
        public DjfsResourcePath getOutputPathForResource(DjfsResource resource, DjfsUid uid,
                                                         Supplier<Option<ShareInfo>> shareInfoProvider)
        {
            Option<ShareInfo> shareInfo = shareInfoProvider.get();
            if (shareInfo.isPresent()) {
                return shareInfo.get().ownerPathToParticipantPath(resource.getPath(), uid)
                        .get();
            }
            return resource.getPath();
        }

        @Override
        public DjfsResourcePath getCorrectPath(DjfsPrincipal principal, DjfsResourcePath sourcePath,
                                               FilesystemAccessController filesystemAccessController, ShareInfoManager shareInfoManager,
                                               Option<ReadPreference> readPreference)
        {
            filesystemAccessController.checkReadResourceWithReadPreference(principal, sourcePath, readPreference);
            Option<ShareInfo> shareInfoO = shareInfoManager.get(sourcePath, readPreference);
            return shareInfoO.filterMap(x -> x.participantPathToOwnerPath(sourcePath)).getOrElse(sourcePath);
        }

        @Override
        public DjfsResource getSharedResource(DjfsResource resource, ShareInfoManager shareInfoManager,
                                              FilesystemAccessController filesystemAccessController,
                                              DjfsResourceDao djfsResourceDao,
                                              Option<ReadPreference> readPreference)
        {
            final Option<ShareInfo> shareInfoO = shareInfoManager.getWithoutParent(resource.getPath());
            return shareInfoO.filterMap(
                    shareInfo -> shareInfo.getGroupPath().getArea().find(DjfsPrincipal.cons(shareInfo.getOwnerUid()), shareInfo.getGroupPath(),
                            shareInfoManager, filesystemAccessController, djfsResourceDao, readPreference)
            ).getOrElse(resource);
        }

        @Override
        public boolean shouldCheckGroupPathPermissions() {
            return true;
        }

        @Override
        public Option<ShareInfo> getShareInfoWithRoot(DjfsResource resource, ShareInfoManager shareInfoManager,
                                                      Option<ReadPreference> readPreference,
                                                      Option<DjfsUid> participantUid)
        {
            return shareInfoManager.getWithRoot(resource, participantUid);
        }

    },

    TRASH("trash", "trash", new ResourceAreaDiskMetaProvider(), new FolderAreaTrashMetaProvider()),

    HIDDEN("hidden", "hidden_data", new ResourceAreaMetaProvider(), new FolderAreaMetaProvider()) {
        @Override
        public boolean isSearchable() {
            return false;
        }

        @Override
        protected boolean isActivityPermitted(CopyIntent activity) {
            return activity.getPrincipal().isSystem();
        }
    },

    ATTACH("attach", "attach_data", new ResourceAreaMetaProvider(), new FolderAreaMetaProvider()) {
        @Override
        public boolean isXivaPushNotificationSent() {
            return false;
        }

        @Override
        public boolean isEventHistoryLogged() {
            return false;
        }

        @Override
        protected boolean isActivityPermitted(CopyIntent activity) {
            if (activity.getPrincipal().isSystem()) {
                return true;
            }

            return activity.getDestination().getArea() != this;
        }

        @Override
        public boolean isOperationPermitted(DjfsPrincipal principal, DjfsResourcePath path,
                FilesystemOperation operation)
        {
            if (operation == FilesystemOperation.CREATE_FOLDER) {
                if (principal.isSystem()) {
                    return true;
                }
                DjfsResourcePath allowedFolder = DjfsResourcePath.cons(path.getUid(), "/attach/YaFotki");
                return path.equals(allowedFolder) || allowedFolder.isParentFor(path);
            }
            return true;
        }
    },
    NAROD("lnarod", "narod_data", new ResourceAreaMetaProvider(), new FolderAreaMetaProvider()) {
        @Override
        protected boolean isActivityPermitted(CopyIntent activity) {
            if (activity.getPrincipal().isSystem()) {
                return true;
            }

            return activity.getDestination().getArea() != this;
        }
    },

    MISC("misc", "misc_data", new ResourceAreaMetaProvider(), new FolderAreaMetaProvider()) {
        @Override
        public boolean isSearchable() {
            return false;
        }

        @Override
        protected boolean isActivityPermitted(CopyIntent activity) {
            return activity.getPrincipal().isSystem();
        }
    },

    PHOTOUNLIM("photounlim", "photounlim_data", new ResourceAreaDiskMetaProvider(), new FolderAreaMetaProvider()) {
        @Override
        protected boolean isActivityPermitted(CopyIntent activity) {
            if (activity.getPrincipal().isSystem()) {
                return true;
            }

            return activity.getDestination().getArea() != this;
        }
    },

    NOTES("notes", "notes_data", new ResourceAreaMetaProvider(), new FolderAreaMetaProvider()) {
        @Override
        public boolean isSearchable() {
            return false;
        }

        @Override
        public boolean isXivaPushNotificationSent() {
            return false;
        }

        @Override
        public boolean isEventHistoryLogged() {
            return false;
        }

        @Override
        protected boolean isActivityPermitted(CopyIntent activity) {
            return activity.getPrincipal().isSystem();
        }
    },

    ADDITIONAL_DATA("additional", "additional_data", new ResourceAreaMetaProvider(), new FolderAreaMetaProvider()) {
        @Override
        public boolean isSearchable() {
            return false;
        }

        @Override
        public boolean isXivaPushNotificationSent() {
            return false;
        }

        @Override
        public boolean isEventHistoryLogged() {
            return false;
        }

        @Override
        protected boolean isActivityPermitted(CopyIntent activity) {
            return activity.getPrincipal().isSystem();
        }
    },
    SHARE("share", "share", new ResourceAreaMetaProvider(), new FolderAreaMetaProvider()) {
        @Override
        public boolean isSearchable() {
            return false;
        }

        @Override
        public boolean isXivaPushNotificationSent() {
            return false;
        }

        @Override
        public boolean isEventHistoryLogged() {
            return false;
        }

        @Override
        public boolean isOperationPermitted(DjfsPrincipal principal, DjfsResourcePath path, FilesystemOperation operation) {
            return false;
        }

        @Override
        protected boolean isActivityPermitted(CopyIntent activity) {
            return false;
        }

        @Override
        protected boolean isActivityPermitted(MoveIntent activity) {
            return false;
        }
    }
    ;

    public final String mongoCollectionName;
    public final String rootFolderName;
    private final ResourceAreaMetaProvider resourceAreaMetaProvider;
    private final FolderAreaMetaProvider folderAreaMetaProvider;

    DjfsResourceArea(String rootFolderName, String mongoCollectionName, ResourceAreaMetaProvider resourceAreaMetaProvider,
                     FolderAreaMetaProvider folderAreaMetaProvider) {
        this.rootFolderName = rootFolderName;
        this.mongoCollectionName = mongoCollectionName;
        this.resourceAreaMetaProvider = resourceAreaMetaProvider;
        this.folderAreaMetaProvider = folderAreaMetaProvider;
    }

    @Override
    public String value() {
        return rootFolderName;
    }

    public boolean isSearchable() {
        return true;
    }

    public boolean isChangelogWritten() {
        return false;
    }

    public boolean isUserIndexVersionUpdated() {
        return false;
    }

    public boolean isXivaPushNotificationSent() {
        return true;
    }

    public boolean isEventHistoryLogged() {
        return true;
    }

    public boolean isOperationPermitted(DjfsPrincipal principal, DjfsResourcePath path, FilesystemOperation operation) {
        return true;
    }

    protected boolean isActivityPermitted(CopyIntent activity) {
        return true;
    }

    protected boolean isActivityPermitted(MoveIntent activity) {
        DjfsResourceArea sourceArea = activity.getSource().getArea();
        DjfsResourceArea destinationArea = activity.getDestination().getArea();

        if (sourceArea == destinationArea && sourceArea != TRASH) {
            // allow all moves except move inside trash (desktop client doesn't support moved deltas inside trash)
            return true;
        } else return (sourceArea != TRASH && sourceArea != HIDDEN) && destinationArea == TRASH;
    }

    public boolean isActivityPermitted(Activity activity) {
        if (activity instanceof CopyIntent) {
            return isActivityPermitted((CopyIntent) activity);
        } else if (activity instanceof MoveIntent) {
            return isActivityPermitted((MoveIntent) activity);
        }
        throw new NotImplementedException();
    }

    public long singleFileSizeLimit() {
        return 50 * ByteConstants.GiB;
    }

    public static final ListF<DjfsResourceArea> ALL = Cf.wrap(DjfsResourceArea.values());
    public static final StringEnumResolver<DjfsResourceArea> R = StringEnumResolver.r(DjfsResourceArea.class);

    public static DjfsResourceArea fromLegacyServiceId(String legacyServiceId) {
        String rawArea = StringUtils.removeStart(legacyServiceId, "/");
        return R.fromValueO(rawArea).getOrThrow(() -> new InvalidDjfsResourceAreaException(rawArea));
    }

    public ResourceAreaMetaProvider getResourceAreaMetaProvider() {
        return resourceAreaMetaProvider;
    }

    public FolderAreaMetaProvider getFolderAreaMetaProvider() {
        return folderAreaMetaProvider;
    }

    public Option<DjfsResource> find(DjfsPrincipal principal, DjfsResourcePath path, ShareInfoManager shareInfoManager,
                                     FilesystemAccessController filesystemAccessController, DjfsResourceDao djfsResourceDao,
                                     Option<ReadPreference> readPreference)
    {
        return djfsResourceDao.find(path);
    }

    public DjfsResourcePath getVisiblePathForResource(DjfsResource resource, DjfsUid uid, ShareInfoManager shareInfoManager,
                                                      Option<ReadPreference> readPreference)
    {
        Option<ShareInfo> shareInfoO = getShareInfoWithRoot(resource, shareInfoManager, readPreference, Option.of(uid));
        if (shareInfoO.map(shareInfo -> !shareInfo.isRoot(resource.getPath())).getOrElse(Boolean.FALSE)) {
            return shareInfoO.get().ownerPathToParticipantPath(resource.getPath(), uid).get();
        }
        return resource.getPath();
    }

    public DjfsResourcePath getOutputPathForResource(DjfsResource resource, DjfsUid uid,
                                                     Supplier<Option<ShareInfo>> shareInfoProvider)
    {
        return resource.getPath();
    }

    public DjfsResourcePath getCorrectPath(DjfsPrincipal principal, DjfsResourcePath sourcePath,
                                           FilesystemAccessController filesystemAccessController, ShareInfoManager shareInfoManager,
                                           Option<ReadPreference> readPreference)
    {
        return sourcePath;
    }

    public DjfsResource getSharedResource(DjfsResource resource, ShareInfoManager shareInfoManager,
                                          FilesystemAccessController filesystemAccessController,
                                          DjfsResourceDao djfsResourceDao,
                                          Option<ReadPreference> readPreference)
    {
        return resource;
    }

    public boolean shouldCheckGroupPathPermissions() {
        return false;
    }

    public Option<ShareInfo> getShareInfoWithRoot(DjfsResource resource, ShareInfoManager shareInfoManager,
                                                  Option<ReadPreference> readPreference,
                                                  Option<DjfsUid> participantUid)
    {
        return Option.empty();
    }
}
