package ru.yandex.chemodan.app.djfs.core.filesystem.model;

import lombok.Getter;

import ru.yandex.bolts.internal.NotImplementedException;
import ru.yandex.chemodan.app.djfs.core.ActionContext;
import ru.yandex.chemodan.app.djfs.core.db.DjfsUidSource;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.exception.InvalidClientInputDjfsResourceIdException;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.exception.InvalidDjfsResourceIdException;
import ru.yandex.chemodan.app.djfs.core.user.DjfsUid;
import ru.yandex.misc.lang.Assume;

@Getter
public class DjfsResourceId implements DjfsUidSource {
    private final DjfsUid uid;
    private final DjfsFileId fileId;

    private static final String SEPARATOR = ":";

    private static final Assume INTERNAL_ASSUME = new Assume() {
        @Override
        public void fail(String message) {
            throw new InvalidDjfsResourceIdException(message);
        }
    };

    private static final Assume CLIENT_ASSUME = new Assume() {
        @Override
        public void fail(String message) {
            throw new InvalidClientInputDjfsResourceIdException(message);
        }
    };

    private DjfsResourceId(DjfsUid uid, DjfsFileId fileId) {
        this.uid = uid;
        this.fileId = fileId;
    }

    public static DjfsResourceId cons(String rawResourceId) {
        return cons(rawResourceId, INTERNAL_ASSUME);
    }

    public static DjfsResourceId cons(String rawResourceId, ActionContext context) {
        switch (context) {
            case INTERNAL:
                return cons(rawResourceId, INTERNAL_ASSUME);
            case CLIENT_INPUT:
                return cons(rawResourceId, CLIENT_ASSUME);
        }
        throw new NotImplementedException();
    }

    public static DjfsResourceId cons(String rawResourceId, Assume assume) {
        assume.notNull(rawResourceId);
        assume.notBlank(rawResourceId);
        String[] parts = rawResourceId.split(SEPARATOR, 2);
        assume.equals(2, parts.length, "resource_id missing separator '" + SEPARATOR + "'");
        return new DjfsResourceId(DjfsUid.cons(parts[0], assume), DjfsFileId.cons(parts[1], assume));
    }

    public static DjfsResourceId cons(String uid, String fileId) {
        return cons(uid, fileId, INTERNAL_ASSUME);
    }

    public static DjfsResourceId cons(String uid, String fileId, Assume assume) {
        return new DjfsResourceId(DjfsUid.cons(uid, assume), DjfsFileId.cons(fileId, assume));
    }

    public static DjfsResourceId cons(DjfsUid uid, String fileId) {
        return cons(uid, fileId, INTERNAL_ASSUME);
    }

    public static DjfsResourceId cons(DjfsUid uid, String fileId, Assume assume) {
        return new DjfsResourceId(uid, DjfsFileId.cons(fileId, assume));
    }

    public static DjfsResourceId cons(DjfsUid uid, DjfsFileId fileId) {
        return new DjfsResourceId(uid, fileId);
    }

    public DjfsFileId getFileId() {
        return fileId;
    }

    @Override
    public DjfsUid getUid() {
        return uid;
    }

    public String getValue() {
        return uid + SEPARATOR + fileId.getValue();
    }

    @Override
    public String toString() {
        return getValue();
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }

        if (!(o instanceof DjfsResourceId)) {
            return false;
        }

        DjfsResourceId obj = (DjfsResourceId) o;
        return uid.equals(obj.uid) && fileId.equals(obj.fileId);
    }

    @Override
    public int hashCode() {
        return toString().hashCode();
    }
}
