package ru.yandex.chemodan.app.djfs.core.filesystem.model;

import java.util.Objects;
import java.util.UUID;
import java.util.function.Supplier;

import lombok.Getter;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.bolts.function.Function;
import ru.yandex.chemodan.app.djfs.core.DjfsException;
import ru.yandex.chemodan.app.djfs.core.db.DjfsUidSource;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.exception.InvalidDjfsResourceAreaException;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.exception.InvalidDjfsResourcePathException;
import ru.yandex.chemodan.app.djfs.core.user.DjfsUid;
import ru.yandex.chemodan.app.djfs.core.util.UuidUtils;
import ru.yandex.misc.digest.Md5;
import ru.yandex.misc.lang.Assume;
import ru.yandex.misc.lang.DefaultObject;
import ru.yandex.misc.lang.StringUtils;

/**
 * @author eoshch
 */
@Getter
public class DjfsResourcePath extends DefaultObject implements DjfsUidSource {
    private final DjfsUid uid;
    private final DjfsResourceArea area;
    private final String path;
    private final Option<String> name;

    public static final String DELIMITER = "/";
    private static final String DOUBLE_DELIMITER = "//";
    public static final String ROOT = "/";
    public static final String EXTENSION_DELIMITER = ".";

    public static final int MAX_NAME_LENGTH = 255;
    public static final int MAX_PATH_LENGTH = 32760;
    public static final SetF<String> FORBIDDEN_NAMES = Cf.set(".", "..");

    private static final Assume A = new Assume() {
        @Override
        public void fail(String message) {
            throw new InvalidDjfsResourcePathException(message);
        }
    };

    private DjfsResourcePath(DjfsUid uid, DjfsResourceArea area, String path) {
        this.uid = uid;
        this.area = area;
        this.path = path;
        this.name = Option.empty();
    }

    private DjfsResourcePath(DjfsUid uid, DjfsResourceArea area, String path, Option<String> name) {
        this.uid = uid;
        this.area = area;
        this.path = path;
        this.name = name;
    }

    public String getFullPath() {
        return uid.asString() + ":" + path;
    }

    public String getMongoId() {
        return getMongoId(uid, path);
    }

    public UUID getPgId() {
        return getPgId(uid, path);
    }

    public boolean isRoot() {
        return Objects.equals(path, ROOT);
    }

    public boolean isAreaRoot() {
        return !isRoot() && StringUtils.countMatches(path, DELIMITER) == 1;
    }

    public void checkNewPath() {
        checkNewPath(InvalidDjfsResourcePathException::new);
    }

    private String[] asStringArray() {
        return StringUtils.split(path, DELIMITER);
    }

    public void checkNewPath(Function<String, DjfsException> createException) {
        if (isRoot()) {
            return;
        }

        if (path.length() > DjfsResourcePath.MAX_PATH_LENGTH) {
            throw createException.apply("path is too long");
        }

        for (String part : asStringArray()) {
            if (part.length() > DjfsResourcePath.MAX_NAME_LENGTH) {
                throw createException.apply("name '" + part + "' is too long");
            }

            if (area != DjfsResourceArea.HIDDEN && FORBIDDEN_NAMES.containsTs(part)) {
                throw createException.apply("name '" + part + "' is forbidden");
            }
        }
    }

    public <T extends DjfsException> void checkDepth(int depthMax, Supplier<T> createException) {
        if (getDepth() > depthMax) {
            throw createException.get();
        }
    }

    public DjfsResourcePath getParent() {
        if (isRoot()) {
            // todo: exception class
            throw new RuntimeException("root path has no parent");
        }
        if (isAreaRoot()) {
            return new DjfsResourcePath(uid, area, "/");
        }

        String parentPath = StringUtils.substringBeforeLast(path, DELIMITER);
        return new DjfsResourcePath(uid, area, parentPath);
    }

    /**
     * First item is root
     */
    public ListF<DjfsResourcePath> getAllParents() {
        if (isRoot()) {
            return Cf.list();
        }

        String[] parts = asStringArray();
        ListF<DjfsResourcePath> result = Cf.arrayList(DjfsResourcePath.root(uid, area));
        for (int i = 0; i < parts.length - 1; i++) {
            String part = parts[i];
            result.add(result.last().getChildPath(part));
        }
        return result;
    }

    public int getDepth() {
        if (isRoot()) {
            return 0;
        }
        int delimitersCount = StringUtils.countMatches(path, DELIMITER);
        return path.endsWith(DELIMITER) ? delimitersCount - 1 : delimitersCount;
    }

    /**
     * Returns mongo ids of all parents starting from root
     */
    public ListF<UUID> getMongoParentIds() {
        return getAllParents().map(x -> UuidUtils.fromHex(getMongoId(uid, x.getPath())));
    }

    public DjfsResourcePath getChildPath(String name) {
        A.isFalse(StringUtils.contains(name, DELIMITER));
        if (isRoot()) {
            A.equals(area.rootFolderName, name);
            return DjfsResourcePath.areaRoot(uid, area);
        } else {
            return new DjfsResourcePath(uid, area, path + DELIMITER + name);
        }
    }

    public DjfsResourcePath getMultipleChildPath(String name) {
        A.isFalse(StringUtils.startsWith(name, DELIMITER));
        A.isFalse(StringUtils.contains(name, DOUBLE_DELIMITER));
        String[] parts = StringUtils.split(name, DELIMITER);
        DjfsResourcePath result = this;
        for (String part : parts) {
            result = result.getChildPath(part);
        }
        return result;
    }

    public boolean isParentFor(DjfsResourcePath other) {
        return Objects.equals(uid, other.uid)
                && Objects.equals(area, other.area)
                && (this.isRoot() || StringUtils.startsWith(other.path, path + DELIMITER));
    }

    public DjfsResourcePath changeArea(DjfsResourceArea area) {
        return changeParent(DjfsResourcePath.areaRoot(uid, getArea()), DjfsResourcePath.areaRoot(uid, area));
    }

    public DjfsResourcePath changeParent(DjfsResourcePath currentPrefix, DjfsResourcePath newPrefix) {
        A.isTrue(currentPrefix.isParentFor(this), "Invalid currentPrefix: " + currentPrefix);
        String newPath;
        if (currentPrefix.isRoot()) {
            newPath = newPrefix.path + path;
        } else {
            newPath = newPrefix.path + StringUtils.substringAfter(path, currentPrefix.path);
        }

        return new DjfsResourcePath(newPrefix.uid, newPrefix.area, newPath);
    }

    public DjfsResourcePath changeToPublicHash(String oldPath, String hash, DjfsUid uid, Option<String> nameFromHash) {
        final String newPath = hash + ":" + StringUtils.substringAfter(path, oldPath);

        return new DjfsResourcePath(uid, area, newPath, nameFromHash);
    }

    public DjfsResourcePath suffix(String suffix) {
        return new DjfsResourcePath(uid, area, path + suffix);
    }

    public String getName() {
        return name.getOrElse(StringUtils.substringAfterLast(path, DELIMITER));
    }

    public String getExtensionByAfterLastDot() {
        return StringUtils.substringAfterLast(getName(), EXTENSION_DELIMITER);
    }

    /**
     * Should be similar Python's os.path.splitext()
     * @return extension without preceding dot
     */
    public String getExtensionByMpfsWay() {
        String name = getName().replaceAll("^\\.+", ".");
        if (name.startsWith(EXTENSION_DELIMITER) && name.lastIndexOf(EXTENSION_DELIMITER) == 0) {
            return "";
        }
        return getExtensionByAfterLastDot();
    }

    @Override
    public DjfsUid getUid() {
        return uid;
    }

    @Override
    public String toString() {
        return uid + ":" + path;
    }

    private static Md5.Sum getId(DjfsUid uid, String path) {
        return Md5.A.digest(uid.asString() + ":" + path);
    }

    public static String getMongoId(DjfsUid uid, String path) {
        return getId(uid, path).hex();
    }

    public static UUID getPgId(DjfsUid uid, String path) {
        return UuidUtils.from(getId(uid, path).getBytes());
    }

    public static DjfsResourcePath cons(String fullpath) {
        A.notBlank(fullpath, fullpath);
        A.isTrue(StringUtils.contains(fullpath, ":"), fullpath);

        String[] parts = fullpath.split(":", 2);
        return cons(parts[0], parts[1]);
    }

    public static DjfsResourcePath cons(long uid, String path) {
        // todo: optimize: make assume static
        Assume assume = new Assume() {
            @Override
            public void fail(String message) {
                throw new InvalidDjfsResourcePathException(uid, path);
            }
        };
        assume.isTrue(DjfsUid.isValid(uid));

        return cons(DjfsUid.cons(uid), path);
    }

    public static DjfsResourcePath cons(String uid, String path) {
        Assume assume = new Assume() {
            @Override
            public void fail(String message) {
                throw new InvalidDjfsResourcePathException(uid, path);
            }
        };
        assume.isTrue(DjfsUid.isValid(uid));

        return cons(DjfsUid.cons(uid), path);
    }

    public static boolean containsUid(String rawPath) {
        if (StringUtils.isBlank(rawPath) || !StringUtils.contains(rawPath, ":")) {
            return false;
        }
        String[] parts = rawPath.split(":", 2);
        return DjfsUid.isValid(parts[0]);
    }

    public static DjfsResourcePath cons(DjfsUid uid, String path) {
        Assume assume = new Assume() {
            @Override
            public void fail(String message) {
                throw new InvalidDjfsResourcePathException(uid == null ? "<null>" : uid.asString(), path);
            }
        };
        assume.notNull(uid);
        assume.notBlank(path);
        assume.isTrue(StringUtils.startsWith(path, ROOT));
        assume.isFalse(path.contains(DOUBLE_DELIMITER));

        String areaPart = StringUtils.substringBefore(path.substring(1), DELIMITER);

        DjfsResourceArea area;
        if (uid == DjfsUid.SHARE_PRODUCTION && Objects.equals(areaPart, "share")) {
            area = DjfsResourceArea.DISK;
        } else {
            area = DjfsResourceArea.R.fromValueO(areaPart)
                    .getOrThrow(() -> new InvalidDjfsResourceAreaException(areaPart, uid, path));
        }

        return new DjfsResourcePath(uid, area, StringUtils.removeEnd(path, DELIMITER));
    }

    public static DjfsResourcePath root(String uid, DjfsResourceArea area) {
        return new DjfsResourcePath(DjfsUid.cons(uid), area, ROOT);
    }

    public static DjfsResourcePath root(Long uid, DjfsResourceArea area) {
        return new DjfsResourcePath(DjfsUid.cons(uid), area, ROOT);
    }

    public static DjfsResourcePath root(DjfsUid uid, DjfsResourceArea area) {
        return new DjfsResourcePath(uid, area, ROOT);
    }

    public static DjfsResourcePath areaRoot(String uid, DjfsResourceArea area) {
        return new DjfsResourcePath(DjfsUid.cons(uid), area, ROOT + area.rootFolderName);
    }

    public static DjfsResourcePath areaRoot(DjfsUid uid, DjfsResourceArea area) {
        return new DjfsResourcePath(uid, area, ROOT + area.rootFolderName);
    }
}
