package ru.yandex.chemodan.app.djfs.core.filesystem.model;

import java.util.Objects;
import java.util.UUID;

import lombok.Getter;
import org.joda.time.Instant;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.bolts.function.Function;
import ru.yandex.bolts.function.Function1V;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.exception.InvalidFileDjfsResourceException;
import ru.yandex.chemodan.app.djfs.core.user.DjfsUid;
import ru.yandex.chemodan.app.djfs.core.util.InstantUtils;
import ru.yandex.chemodan.app.djfs.core.util.UuidUtils;
import ru.yandex.misc.geo.Coordinates;
import ru.yandex.misc.lang.Assume;


/**
 * @author eoshch
 */
@Getter
public class FileDjfsResource extends DjfsResource {
    private final long size;
    private final String hid;
    private final String md5;
    private final String sha256;

    private final String fileStid;
    private final String digestStid;
    private final Option<String> previewStid;

    private final Instant creationTime;
    private final Instant modificationTime;
    private final Option<Instant> exifTime;

    private final Option<AntiVirusScanStatus> antiVirusScanStatus;
    private final Option<String> source;
    private final Option<String> mimetype;
    private final Option<MediaType> mediaType;
    private final Option<String> fotkiTags;
    private final Option<String> externalUrl;

    private final Option<Integer> height;
    private final Option<Integer> width;
    private final Option<Integer> angle;

    private final Option<Double> aesthetics;
    private final Option<PhotosliceAlbumType> photosliceAlbumType;

    private final Option<Coordinates> coordinates;

    /**
     * TODO заменить на enum после того как разберёмся что в этом поле должно быть:
     * PhotosliceAlbumType, GeneratedAlbumType из питона или и то и то
     */
    private final Option<ListF<String>> albumsExclusions;

    // todo: parse json
    private final Option<String> videoInfo;

    private final boolean isLivePhoto;
    private final Option<UUID> liveVideoId;

    private static final SetF<String> IGNORE_MD5 = Cf.set(
            "569a1c98696050439b5b2a1ecfa52d19",  // Мишки.jpg
            "d27d72a3059ad5ebed7a5470459d2670",  // Москва.jpg
            "a64146fee5e15b3b94c204e544426d43",  // Зима.jpg
            "ab903d9cab031eca2a8f12f37bbc9d37",  // Море.jpg
            "1392851f0668017168ee4b5a59d66e7b",  // Горы.jpg
            "f1abe3b27b410128623fd1ca00a45c29"   // Санкт-Петербург.jpg
    );

    private static final SetF<String> HEIF_MIME_TYPES = Cf.set("image/heif", "image/heic");

    private static final SetF<String> DISK_AREA_ACCEPTED_MIMETYPES = Cf.set("image/jpeg", "image/tiff");

    private static final SetF<String> VIDEO_ACCEPTED_MIMETYPES = Cf.set(
            "application/vnd.rn-realmedia",
            "application/vnd.rn-realmedia-vbr",
            "application/mp4"
    );

    private static final SetF<String> PHOTOCAMERA_ACCEPTED_MIMETYPES =
            Cf.set("image/png").plus(DISK_AREA_ACCEPTED_MIMETYPES).plus(VIDEO_ACCEPTED_MIMETYPES);

    private static final SetF<String> PHOTOSLICE_VIDEO_REJECTED_EXTENSIONS = Cf.set("mkv", "avi");

    private static final Assume A = new Assume() {
        @Override
        public void fail(String message) {
            throw new InvalidFileDjfsResourceException(message);
        }
    };

    private FileDjfsResource(Builder builder) {
        super(builder);
        A.notNull(builder.size, "size is null");
        A.notNull(builder.hid, "hid is null");
        A.notNull(builder.md5, "md5 is null");
        A.notNull(builder.sha256, "sha256 is null");
        A.notNull(builder.fileStid, "fileStid is null");
        A.notNull(builder.digestStid, "digestStid is null");
        A.notNull(builder.previewStid, "previewStid is null");
        A.notNull(builder.creationTime, "creationTime is null");
        A.notNull(builder.modificationTime, "modificationTime is null");
        A.notNull(builder.exifTime, "exifTime is null");
        A.notNull(builder.antiVirusScanStatus, "antiVirusScanStatus is null");
        A.notNull(builder.source, "source is null");
        A.notNull(builder.mimetype, "mimetype is null");
        A.notNull(builder.mediaType, "mediaType is null");
        A.notNull(builder.fotkiTags, "fotkiTags is null");
        A.notNull(builder.externalUrl, "externalUrl is null");
        A.notNull(builder.height, "height is null");
        A.notNull(builder.width, "width is null");
        A.notNull(builder.angle, "angle is null");
        A.notNull(builder.videoInfo, "videoInfo is null");
        A.notNull(builder.liveVideoId, "liveVideoId is null");
        A.notNull(builder.aesthetics, "aesthetics is null");
        A.notNull(builder.photosliceAlbumType, "photosliceAlbumType is null");
        A.notNull(builder.albumsExclusions, "albumsExclusions is null");
        A.notNull(builder.coordinates, "coordinates is null");

        this.size = builder.size;
        this.hid = builder.hid;
        this.md5 = builder.md5;
        this.sha256 = builder.sha256;
        this.fileStid = builder.fileStid;
        this.digestStid = builder.digestStid;
        this.previewStid = builder.previewStid;
        this.creationTime = builder.creationTime;
        this.modificationTime = builder.modificationTime;
        this.exifTime = builder.exifTime;
        this.antiVirusScanStatus = builder.antiVirusScanStatus;
        this.source = builder.source;
        this.mimetype = builder.mimetype;
        this.mediaType = builder.mediaType;
        this.fotkiTags = builder.fotkiTags;
        this.externalUrl = builder.externalUrl;
        this.height = builder.height;
        this.width = builder.width;
        this.angle = builder.angle;
        this.videoInfo = builder.videoInfo;
        this.isLivePhoto = builder.isLivePhoto;
        this.liveVideoId = builder.liveVideoId;
        this.aesthetics = builder.aesthetics;
        this.photosliceAlbumType = builder.photosliceAlbumType;
        this.albumsExclusions = builder.albumsExclusions;
        this.coordinates = builder.coordinates;
    }

    private Builder initializeBuilder(Builder builder) {
        return super.initializeBuilder(builder)
                .size(this.size)
                .hid(this.hid)
                .md5(this.md5)
                .sha256(this.sha256)
                .fileStid(this.fileStid)
                .digestStid(this.digestStid)
                .previewStid(this.previewStid)
                .creationTime(this.creationTime)
                .modificationTime(this.modificationTime)
                .exifTime(this.exifTime)
                .antiVirusScanStatus(this.antiVirusScanStatus)
                .source(this.source)
                .mimetype(this.mimetype)
                .mediaType(this.mediaType)
                .fotkiTags(this.fotkiTags)
                .externalUrl(this.externalUrl)
                .height(this.height)
                .width(this.width)
                .angle(this.angle)
                .videoInfo(this.videoInfo)
                .isLivePhoto(this.isLivePhoto)
                .liveVideoId(this.liveVideoId)
                .aesthetics(this.aesthetics)
                .photosliceAlbumType(this.photosliceAlbumType)
                .albumsExclusions(this.albumsExclusions)
                .coordinates(this.coordinates);
    }

    @Override
    public Builder toBuilder() {
        return initializeBuilder(new Builder());
    }

    @Override
    public Option<Instant> getCreationTimeO() {
        return Option.of(creationTime);
    }

    @Override
    public Option<Instant> getModificationTimeO() {
        return Option.of(modificationTime);
    }

    @Override
    public boolean hasPhotosliceTime() {
        return getPhotosliceTime().isPresent();
    }

    /**
     * Returns photosliceTime for Indexer purposes.
     * Should be synced with search logic
     * https://git.yandex-team.ru/gitweb/saas/personal.git?a=blob;f=src/yadisk-search-proxy/main/bundle/yadisk-search-proxy-prod.conf;h=b97211dece70790b47a18829932bb6801d458cfe;hb=HEAD#l102
     *
     * @return photoslice time
     */
    public Option<Instant> getPhotosliceTime() {
        return Option.when(isPhotosliceFile(), () -> exifTime.plus(Cf.list(creationTime, modificationTime))
                .filter(t -> t.getMillis() != 0).plus(new Instant(0)).first());
    }

    public boolean isPhotosliceFile() {
        if (!this.mimetype.isPresent()) {
            return false;
        }

        if (IGNORE_MD5.containsTs(getMd5())) {
            return false;
        }

        String mimetype = this.mimetype.get();

        if (getPath().getArea() == DjfsResourceArea.PHOTOUNLIM) {
            if (mimetype.startsWith("image/")) {
                return true;
            } else if (mimetype.startsWith("video/")) {
                return !PHOTOSLICE_VIDEO_REJECTED_EXTENSIONS.containsTs(getPath().getExtensionByAfterLastDot().toLowerCase());
            } else if (VIDEO_ACCEPTED_MIMETYPES.containsTs(mimetype)) {
                return true;
            }
        } else if (getPath().getArea() == DjfsResourceArea.DISK) {
            String path = getPath().getPath();
            if (exifTime.isPresent() && DISK_AREA_ACCEPTED_MIMETYPES.containsTs(mimetype)) {
                return true;
            } else if (HEIF_MIME_TYPES.containsTs(mimetype)) {
                return true;
            } else if (DefaultFolder.PHOTOSTREAM.representations().exists(path::startsWith)) {
                if (PHOTOCAMERA_ACCEPTED_MIMETYPES.containsTs(mimetype)) {
                    return true;
                } else if (mimetype.startsWith("video/")) {
                    return true;
                }
            }
        }

        return false;
    }

    public Option<Long> getEtime() {
        return exifTime.map(InstantUtils::toSecondsLong);
    }

    public VersioningStatus getVersioningStatus() {
        return getPath().getArea().equals(DjfsResourceArea.DISK) ?
                VersioningStatus.VERSIONABLE : VersioningStatus.DISABLED;
    }

    private static Builder initWithRandomValues(Builder builder) {
        return builder
                .hid(UuidUtils.randomToHexString())
                .md5(UuidUtils.randomToHexString())
                .sha256(UuidUtils.randomToHexString() + UuidUtils.randomToHexString())
                .fileStid(UuidUtils.randomToHexString())
                .digestStid(UuidUtils.randomToHexString())
                .antiVirusScanStatus(AntiVirusScanStatus.CLEAN);
    }

    public boolean hashesEqual(FileDjfsResource other) {
        return Objects.equals(this.md5, other.md5) && Objects.equals(this.sha256, other.sha256)
                && Objects.equals(this.hid, other.hid) && this.size == other.size;
    }

    public static FileDjfsResource cons(DjfsResourcePath path) {
        return cons(path, path.getParent().getPgId(), Instant.now(), Function1V.nop());
    }

    public static FileDjfsResource cons(DjfsResourcePath path, UUID parentId) {
        return cons(path, parentId, Instant.now(), Function1V.nop());
    }

    public static FileDjfsResource cons(FolderDjfsResource parent, String name) {
        return cons(parent.getPath().getChildPath(name), parent.getId(), Instant.now(), Function1V.nop());
    }

    public static FileDjfsResource cons(FolderDjfsResource parent, String name,
            Function1V<? super Builder> initialize)
    {
        return cons(parent.getPath().getChildPath(name), parent.getId(), Instant.now(), initialize);
    }

    public static FileDjfsResource random(DjfsUid uid, String path) {
        return random(DjfsResourcePath.cons(uid, path));
    }

    public static FileDjfsResource random(DjfsResourcePath path) {
        return cons(path, path.getParent().getPgId(), Instant.now(), FileDjfsResource::initWithRandomValues);
    }

    public static FileDjfsResource random(DjfsResourcePath path, Function1V<? super Builder> initialize) {
        return cons(path, path.getParent().getPgId(), Instant.now(), x -> initialize.apply(initWithRandomValues(x)));
    }

    public static FileDjfsResource random(FolderDjfsResource parent, String name) {
        return cons(parent, name, FileDjfsResource::initWithRandomValues);
    }

    public static FileDjfsResource random(FolderDjfsResource parent, String name,
            Function<Builder, Builder> initialize)
    {
        return cons(parent, name, x -> initialize.apply(initWithRandomValues(x)));
    }

    public static FileDjfsResource cons(DjfsResourcePath path, UUID parentId, Instant timestamp,
            Function1V<? super Builder> initialize)
    {
        Builder builder = FileDjfsResource.builder()
                .id(path.getPgId())
                .parentId(Option.of(parentId))
                .path(path)
                .isVisible(true)
                .uploadTime(Option.of(timestamp))
                .creationTime(timestamp)
                .modificationTime(timestamp)
                .version(Option.of(InstantUtils.toVersion(timestamp)))
                .fileId(DjfsFileId.random());

        initialize.apply(builder);

        return builder.build();
    }

    public static Builder builder() {
        return new Builder();
    }

    @Getter
    public static final class Builder extends DjfsResource.Builder<Builder, FileDjfsResource> {
        private long size;
        private String hid;
        private String md5;
        private String sha256;
        private String fileStid;
        private String digestStid;
        private Option<String> previewStid = Option.empty();
        private Instant creationTime;
        private Instant modificationTime;
        private Option<Instant> exifTime = Option.empty();
        private Option<AntiVirusScanStatus> antiVirusScanStatus = Option.empty();
        private Option<String> source = Option.empty();
        private Option<String> mimetype = Option.empty();
        private Option<MediaType> mediaType = Option.empty();
        private Option<String> fotkiTags = Option.empty();
        private Option<String> externalUrl = Option.empty();
        private Option<Integer> height = Option.empty();
        private Option<Integer> width = Option.empty();
        private Option<Integer> angle = Option.empty();
        private Option<String> videoInfo = Option.empty();
        private boolean isLivePhoto = false;
        private Option<UUID> liveVideoId = Option.empty();
        private Option<Double> aesthetics = Option.empty();
        private Option<PhotosliceAlbumType> photosliceAlbumType = Option.empty();
        private Option<ListF<String>> albumsExclusions = Option.empty();
        private Option<Coordinates> coordinates = Option.empty();

        private Builder() {
        }

        public Builder size(long size) {
            this.size = size;
            return this;
        }

        public Builder hid(String hid) {
            this.hid = hid;
            return this;
        }

        public Builder md5(String md5) {
            this.md5 = md5;
            return this;
        }

        public Builder sha256(String sha256) {
            this.sha256 = sha256;
            return this;
        }

        public Builder fileStid(String fileStid) {
            this.fileStid = fileStid;
            return this;
        }

        public Builder digestStid(String digestStid) {
            this.digestStid = digestStid;
            return this;
        }

        public Builder previewStid(String previewStid) {
            this.previewStid = Option.ofNullable(previewStid);
            return this;
        }

        public Builder previewStid(Option<String> previewStid) {
            this.previewStid = previewStid;
            return this;
        }

        @Override
        public Builder creationTime(Instant creationTime) {
            this.creationTime = creationTime;
            return this;
        }

        @Override
        public Builder modificationTime(Instant modificationTime) {
            this.modificationTime = modificationTime;
            return this;
        }

        public Builder exifTime(Option<Instant> exifTime) {
            this.exifTime = exifTime;
            return this;
        }

        public Builder exifTime(Instant exifTime) {
            this.exifTime = Option.of(exifTime);
            return this;
        }

        public Builder antiVirusScanStatus(Option<AntiVirusScanStatus> antiVirusScanStatus) {
            this.antiVirusScanStatus = antiVirusScanStatus;
            return this;
        }

        public Builder antiVirusScanStatus(AntiVirusScanStatus antiVirusScanStatus) {
            this.antiVirusScanStatus = Option.of(antiVirusScanStatus);
            return this;
        }

        public Builder source(Option<String> source) {
            this.source = source;
            return this;
        }

        public Builder source(String source) {
            this.source = Option.ofNullable(source);
            return this;
        }

        public Builder mimetype(Option<String> mimetype) {
            this.mimetype = mimetype;
            return this;
        }

        public Builder mimetype(String mimetype) {
            this.mimetype = Option.ofNullable(mimetype);
            return this;
        }

        public Builder mediaType(Option<MediaType> mediaType) {
            this.mediaType = mediaType;
            return this;
        }

        public Builder mediaType(MediaType mediaType) {
            this.mediaType = Option.of(mediaType);
            return this;
        }

        public Builder fotkiTags(Option<String> fotkiTags) {
            this.fotkiTags = fotkiTags;
            return this;
        }

        public Builder fotkiTags(String fotkiTags) {
            this.fotkiTags = Option.ofNullable(fotkiTags);
            return this;
        }

        public Builder externalUrl(Option<String> externalUrl) {
            this.externalUrl = externalUrl;
            return this;
        }

        public Builder externalUrl(String externalUrl) {
            this.externalUrl = Option.ofNullable(externalUrl);
            return this;
        }

        public Builder height(int height) {
            this.height = Option.of(height);
            return this;
        }

        public Builder height(Option<Integer> height) {
            this.height = height;
            return this;
        }

        public Builder width(int width) {
            this.width = Option.of(width);
            return this;
        }

        public Builder width(Option<Integer> width) {
            this.width = width;
            return this;
        }

        public Builder angle(int angle) {
            this.angle = Option.of(angle);
            return this;
        }

        public Builder angle(Option<Integer> angle) {
            this.angle = angle;
            return this;
        }

        public Builder videoInfo(Option<String> videoInfo) {
            this.videoInfo = videoInfo;
            return this;
        }

        public Builder videoInfo(String videoInfo) {
            this.videoInfo = Option.ofNullable(videoInfo);
            return this;
        }

        public Builder isLivePhoto(boolean isLivePhoto) {
            this.isLivePhoto = isLivePhoto;
            return this;
        }

        public Builder liveVideoId(Option<UUID> liveVideoId) {
            this.liveVideoId = liveVideoId;
            return this;
        }

        public Builder liveVideoId(UUID liveVideoId) {
            this.liveVideoId = Option.of(liveVideoId);
            return this;
        }

        public Builder aesthetics(double aesthetics) {
            this.aesthetics = Option.of(aesthetics);
            return this;
        }

        public Builder aesthetics(Option<Double> aesthetics) {
            this.aesthetics = aesthetics;
            return this;
        }

        public Builder photosliceAlbumType(PhotosliceAlbumType photosliceAlbumType) {
            this.photosliceAlbumType = Option.of(photosliceAlbumType);
            return this;
        }

        public Builder photosliceAlbumType(Option<PhotosliceAlbumType> photosliceAlbumType) {
            this.photosliceAlbumType = photosliceAlbumType;
            return this;
        }

        public Builder albumsExclusion(String albumsExclusion) {
            this.albumsExclusions = Option.of(albumsExclusions.getOrElse(Cf.arrayList()).plus(albumsExclusion));
            return this;
        }

        public Builder albumsExclusions(Option<ListF<String>> albumsExclusions) {
            this.albumsExclusions = albumsExclusions;
            return this;
        }

        public Builder coordinates(Option<Coordinates> coordinates) {
            this.coordinates = coordinates;
            return this;
        }

        public Builder coordinates(Coordinates coordinates) {
            this.coordinates = Option.of(coordinates);
            return this;
        }

        @Override
        public Builder getThis() {
            return this;
        }

        @Override
        public FileDjfsResource build() {
            return new FileDjfsResource(this);
        }
    }
}
