package ru.yandex.chemodan.app.djfs.core.filesystem.model;

import java.util.UUID;

import lombok.Getter;
import org.joda.time.Instant;

import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.function.Function1V;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.exception.InvalidFolderDjfsResourceException;
import ru.yandex.chemodan.app.djfs.core.user.DjfsUid;
import ru.yandex.chemodan.app.djfs.core.util.InstantUtils;
import ru.yandex.misc.lang.Assume;

/**
 * @author eoshch
 */
@Getter
public class FolderDjfsResource extends DjfsResource {
    private final Option<Instant> creationTime;
    private final Option<Instant> modificationTime;
    private final Option<FolderType> folderType;
    private final Option<Instant> lastImportTime;

    private static final Assume A = new Assume() {
        @Override
        public void fail(String message) {
            throw new InvalidFolderDjfsResourceException(message);
        }
    };

    private FolderDjfsResource(Builder builder) {
        super(builder);
        A.notNull(builder.creationTime, "creationTime is null");
        A.notNull(builder.modificationTime, "modificationTime is null");
        A.notNull(builder.folderType, "folderType is null");
        A.notNull(builder.lastImportTime, "lastImportTime is null");

        this.creationTime = builder.creationTime;
        this.modificationTime = builder.modificationTime;
        this.folderType = builder.folderType;
        this.lastImportTime = builder.lastImportTime;
    }

    private Builder initializeBuilder(Builder builder) {
        return super.initializeBuilder(builder)
                .creationTime(this.creationTime)
                .modificationTime(this.modificationTime)
                .folderType(this.folderType)
                .lastImportTime(this.lastImportTime);
    }

    public Builder toBuilder() {
        return initializeBuilder(new Builder());
    }

    @Override
    public Option<Instant> getCreationTimeO() {
        return creationTime;
    }

    @Override
    public Option<Instant> getModificationTimeO() {
        return modificationTime;
    }

    @Override
    public boolean hasPhotosliceTime() {
        return false;
    }

    public static FolderDjfsResource cons(DjfsUid uid, String path) {
        return cons(DjfsResourcePath.cons(uid, path));
    }

    public static FolderDjfsResource cons(DjfsResourcePath path) {
        return cons(path, path.getParent().getPgId(), Instant.now(), Function1V.nop());
    }

    public static FolderDjfsResource cons(DjfsResourcePath path, Function1V<? super Builder> initialize) {
        return cons(path, path.getParent().getPgId(), Instant.now(), initialize);
    }

    public static FolderDjfsResource cons(DjfsResourcePath path, Instant timestamp) {
        return cons(path, path.getParent().getPgId(), timestamp, Function1V.nop());
    }

    public static FolderDjfsResource cons(DjfsResourcePath path, UUID parentId) {
        return cons(path, parentId, Instant.now(), Function1V.nop());
    }

    public static FolderDjfsResource cons(FolderDjfsResource parent, String name) {
        return cons(parent.getPath().getChildPath(name), parent.getId(), Instant.now(), Function1V.nop());
    }

    public static FolderDjfsResource cons(FolderDjfsResource parent, String name, Function1V<? super Builder> initialize) {
        return cons(parent.getPath().getChildPath(name), parent.getId(), Instant.now(), initialize);
    }

    public static FolderDjfsResource cons(DjfsResourcePath path, UUID parentId, Instant timestamp) {
        return cons(path, parentId, timestamp, Function1V.nop());
    }

    public static FolderDjfsResource cons(DjfsResourcePath path, UUID parentId, Instant timestamp,
            Function1V<? super Builder> initialize)
    {
        Builder builder = builder()
                .id(path.getPgId())
                .parentId(Option.of(parentId))
                .path(path)
                .isVisible(true)
                .uploadTime(Option.of(timestamp))
                .creationTime(Option.of(timestamp))
                .modificationTime(Option.of(timestamp))
                .version(Option.of(InstantUtils.toVersion(timestamp)))
                .fileId(DjfsFileId.random());

        initialize.apply(builder);

        return builder.build();
    }

    public static Builder builder() {
        return new Builder();
    }

    @Getter
    public static final class Builder extends DjfsResource.Builder<Builder, FolderDjfsResource> {
        private Option<Instant> creationTime = Option.empty();
        private Option<Instant> modificationTime = Option.empty();
        private Option<FolderType> folderType = Option.empty();
        private Option<Instant> lastImportTime = Option.empty();

        private Builder() {
        }

        public Builder creationTime(Option<Instant> creationTime) {
            this.creationTime = creationTime;
            return this;
        }

        @Override
        public Builder creationTime(Instant creationTime) {
            this.creationTime = Option.of(creationTime);
            return this;
        }

        public Builder modificationTime(Option<Instant> modificationTime) {
            this.modificationTime = modificationTime;
            return this;
        }

        @Override
        public Builder modificationTime(Instant modificationTime) {
            this.modificationTime = Option.of(modificationTime);
            return this;
        }

        public Builder folderType(Option<FolderType> folderType) {
            this.folderType = folderType;
            return this;
        }

        public Builder folderType(FolderType folderType) {
            this.folderType = Option.of(folderType);
            return this;
        }

        public Builder lastImportTime(Option<Instant> lastImportTime) {
            this.lastImportTime = lastImportTime;
            return this;
        }

        public Builder lastImportTime(Instant lastImportTime) {
            this.lastImportTime = Option.of(lastImportTime);
            return this;
        }

        @Override
        public Builder getThis() {
            return this;
        }

        public FolderDjfsResource build() {
            return new FolderDjfsResource(this);
        }
    }
}
