package ru.yandex.chemodan.app.djfs.core.filesystem.model.mongo;

import java.nio.charset.StandardCharsets;

import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.impl.ArrayListF;
import ru.yandex.bolts.internal.NotImplementedException;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.AntiVirusScanStatus;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.DjfsFileId;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.DjfsResource;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.DjfsResourceArea;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.DjfsResourcePath;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.FileDjfsResource;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.FolderDjfsResource;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.FolderType;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.MediaType;
import ru.yandex.chemodan.app.djfs.core.user.DjfsUid;
import ru.yandex.chemodan.app.djfs.core.util.InstantUtils;
import ru.yandex.chemodan.app.djfs.core.util.UuidUtils;
import ru.yandex.chemodan.app.djfs.core.util.ZipUtils;
import ru.yandex.misc.lang.Validate;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

/**
 * @author eoshch
 */
public final class ResourceFactory {
    private static final Logger logger = LoggerFactory.getLogger(ResourceFactory.class);

    public static DjfsResource create(MongoResource resource) {
        Option<MongoResourceZData> zData = resource.zdata.map(ZipUtils::decompress)
                .map(MongoResourceZData.B.getParser()::parseJson);
        switch (resource.type) {
            case "file":
                FileDjfsResource.Builder fileBuilder = FileDjfsResource.builder();
                initializeFile(fileBuilder, resource, zData);
                return fileBuilder.build();
            case "dir":
                FolderDjfsResource.Builder folderBuilder = FolderDjfsResource.builder();
                initializeFolder(folderBuilder, resource, zData);
                return folderBuilder.build();
            default:
                throw new NotImplementedException();
        }
    }

    public static MongoResource toMongo(DjfsResource resource) {
        MongoResource result = new MongoResource();
        result.data = Option.of(new MongoResourceData());
        MongoResourceZData zData = new MongoResourceZData();

        if (resource instanceof FolderDjfsResource) {
            initializeMongoFolder(result, (FolderDjfsResource) resource, zData);
        } else if (resource instanceof FileDjfsResource) {
            initializeMongoFile(result, (FileDjfsResource) resource, zData);
        } else {
            throw new NotImplementedException();
        }

        result.zdata = Option.when(zData.isPresent(), zData)
                .map(x -> MongoResourceZData.B.getSerializer().serializeJson(x))
                .map(x -> ZipUtils.compress(x, 1));
        return result;
    }

    private static void initializeSetprop(MongoResourceZData zData) {
        if (!zData.setprop.isPresent()) {
            zData.setprop = Option.of(new MongoResourceSetprop());
        }
    }

    private static void initializePub(MongoResourceZData zData) {
        if (!zData.pub.isPresent()) {
            zData.pub = Option.of(new MongoResourcePub());
        }
    }

    private static void initializeMeta(MongoResourceZData zData) {
        if (!zData.meta.isPresent()) {
            zData.meta = Option.of(new MongoResourceMeta());
        }
    }

    private static void initializeDjfsResource(DjfsResource.Builder<?, ?> builder, MongoResource resource,
            Option<MongoResourceZData> zData)
    {
        builder.id(UuidUtils.fromHex(resource.id));
        builder.parentId(resource.parent.map(UuidUtils::fromHex));
        if (DjfsResourcePath.ROOT.equals(resource.key)) {
            // todo: get area from method call
            builder.path(DjfsResourcePath.root(resource.uid, DjfsResourceArea.DISK));
        } else {
            builder.path(DjfsResourcePath.cons(resource.uid, resource.key));
        }
        builder.modifyUid(resource.data.filterMap(x -> x.modify_uid).map(DjfsUid::cons));
        builder.fileId(resource.data.filterMap(x -> x.file_id).map(DjfsFileId::cons));
        builder.version(resource.version);

        builder.trashAppendTime(zData.filterMap(x -> x.setprop).filterMap(x -> x.append_time)
                .map(InstantUtils::fromSeconds));
        builder.hiddenAppendTime(resource.dtime.map(InstantUtils::fromSeconds));

        builder.trashAppendOriginalPath(resource.data.filterMap(x -> x.original_id));

        builder.isVisible(resource.data.filterMap(x -> x.visible).getOrElse(1) != 0);
        builder.isPublic(resource.data.filterMap(x -> x.publik).getOrElse(0) != 0);
        builder.isBlocked(resource.data.filterMap(x -> x.blocked).getOrElse(0) != 0);
        builder.isPublished(zData.filterMap(x -> x.setprop).filterMap(x -> x.published).getOrElse(0) != 0);

        builder.publicHash(zData.filterMap(x -> x.pub).filterMap(x -> x.public_hash));
        builder.shortUrl(zData.filterMap(x -> x.pub).filterMap(x -> x.short_url));
        builder.symlink(zData.filterMap(x -> x.pub).filterMap(x -> x.symlink));

        builder.folderUrl(zData.filterMap(x -> x.setprop).filterMap(x -> x.folder_url));
        builder.downloadCounter(resource.data.filterMap(x -> x.download_counter));
        builder.customProperties(zData.filterMap(x -> x.setprop).filterMap(x -> x.custom_properties));
    }

    private static void initializeMongoResource(MongoResource result, DjfsResource resource, MongoResourceZData zData) {
        result.id = UuidUtils.toHexString(resource.getId());
        result.parent = resource.getParentId().map(UuidUtils::toHexString);
        result.uid = resource.getPath().getUid().asString();
        result.key = resource.getPath().getPath();
        result.data.get().modify_uid = resource.getModifyUid().map(DjfsUid::asString);
        result.data.get().file_id = resource.getFileId().map(DjfsFileId::getValue);
        result.version = resource.getVersion();

        if (resource.getTrashAppendTime().isPresent()) {
            initializeSetprop(zData);
            zData.setprop.get().append_time = resource.getTrashAppendTime().map(InstantUtils::toSeconds);
        }
        result.dtime = resource.getHiddenAppendTime().map(InstantUtils::toSeconds);

        result.data.get().original_id = resource.getTrashAppendOriginalPath();

        result.data.get().visible = Option.of(resource.isVisible() ? 1 : 0);
        result.data.get().publik = Option.when(resource.isPublic(), 1);
        result.data.get().blocked = Option.when(resource.isBlocked(), 1);
        if (resource.isPublished()) {
            initializeSetprop(zData);
            zData.setprop.get().published = Option.of(1);
        }

        if (resource.getPublicHash().isPresent()) {
            initializePub(zData);
            zData.pub.get().public_hash = resource.getPublicHash();
        }
        if (resource.getShortUrl().isPresent()) {
            initializePub(zData);
            zData.pub.get().short_url = resource.getShortUrl();
        }
        if (resource.getSymlink().isPresent()) {
            initializePub(zData);
            zData.pub.get().symlink = resource.getSymlink();
        }

        if (resource.getFolderUrl().isPresent()) {
            initializeSetprop(zData);
            zData.setprop.get().folder_url = resource.getFolderUrl();
        }
        result.data.get().download_counter = resource.getDownloadCounter();
        if (resource.getCustomProperties().isPresent()) {
            initializeSetprop(zData);
            zData.setprop.get().custom_properties = resource.getCustomProperties();
        }
    }

    private static void initializeFile(FileDjfsResource.Builder builder, MongoResource resource,
            Option<MongoResourceZData> zData)
    {
        initializeDjfsResource(builder, resource, zData);
        initializeFileStids(builder, resource);

        builder.size(resource.data.filterMap(x -> x.size).getOrThrow("file without size"));
        builder.hid(resource.hid.map(x -> new String(x, StandardCharsets.UTF_8)).getOrThrow("file without hid"));

        // todo: in fact these fields need to be obtained by overwriting with values from zdata['meta'], zdata['setprop'], data consequently
        builder.md5(resource.data.filterMap(x -> x.md5).orElse(() -> zData.filterMap(x -> x.meta)
                .filterMap(x -> x.md5)).getOrThrow("file without md5"));
        builder.sha256(resource.data.filterMap(x -> x.sha256).orElse(() -> zData.filterMap(x -> x.meta)
                .filterMap(x -> x.sha256)).getOrThrow("file without sha256"));

        builder.uploadTime(resource.data.filterMap(x -> x.utime).map(InstantUtils::fromSeconds));

        builder.creationTime(resource.data.filterMap(x -> x.ctime).orElse(() -> zData.filterMap(x -> x.meta)
                .filterMap(x -> x.ctime)).map(InstantUtils::fromSeconds).getOrThrow("file without ctime"));

        builder.modificationTime(resource.data.filterMap(x -> x.mtime).orElse(() -> zData.filterMap(x -> x.meta)
                .filterMap(x -> x.mtime)).map(InstantUtils::fromSeconds).getOrThrow("file without mtime"));

        builder.exifTime(resource.data.filterMap(x -> x.etime).map(InstantUtils::fromSeconds));

        builder.antiVirusScanStatus(zData.filterMap(x -> x.meta).filterMap(x -> x.drweb)
                .filterMap(AntiVirusScanStatus.R::fromValueO));
        builder.source(zData.filterMap(x -> x.meta).filterMap(x -> x.source));
        builder.mimetype(resource.data.filterMap(x -> x.mimetype));
        builder.mediaType(resource.data.filterMap(x -> x.mt).filterMap(MediaType.R::valueOfO));
        builder.fotkiTags(zData.filterMap(x -> x.setprop).filterMap(x -> x.fotki_tags));
        builder.externalUrl(zData.filterMap(x -> x.setprop).filterMap(x -> x.external_url));
    }

    private static void initializeFileStids(FileDjfsResource.Builder builder, MongoResource resource) {
        String fileStid = null;
        String digestStid = null;
        String previewStid = null;
        Validate.some(resource.data, "file has no data");
        Validate.some(resource.data.get().stids, "file has no data.stids");
        for (MongoResourceStid stid : resource.data.get().stids.get()) {
            switch (stid.type) {
                case "file_mid":
                    if (fileStid != null) {
                        logger.warn("file has more than one stids of type file_mid. uid: " + resource.uid + " _id: " + resource.id);
                    }
                    fileStid = stid.stid;
                    break;
                case "digest_mid":
                    if (digestStid != null) {
                        logger.warn("file has more than one stids of type digest_mid. uid: " + resource.uid + " _id: " + resource.id);
                    }
                    digestStid = stid.stid;
                    break;
                case "pmid":
                    if (previewStid != null) {
                        logger.warn("file has more than one stids of type pmid. uid: " + resource.uid + " _id: " + resource.id);
                    }
                    previewStid = stid.stid;
                    break;
                default:
                    throw new IllegalArgumentException("file has unknown stid type " + stid.type);
            }
        }
        Validate.notNull(fileStid, "file without stid of type file_mid");
        Validate.notNull(digestStid, "file without stid of type digest_mid");
        builder.fileStid(fileStid);
        builder.digestStid(digestStid);
        builder.previewStid(Option.ofNullable(previewStid));
    }

    private static void initializeFolder(FolderDjfsResource.Builder builder, MongoResource resource,
            Option<MongoResourceZData> zData)
    {
        initializeDjfsResource(builder, resource, zData);

        builder.uploadTime(resource.data.filterMap(x -> x.utime).map(InstantUtils::fromSeconds));
        builder.creationTime(zData.filterMap(x -> x.meta).filterMap(x -> x.ctime).map(InstantUtils::fromSeconds));
        builder.modificationTime(resource.data.filterMap(x -> x.mtime).map(InstantUtils::fromSeconds));

        builder.folderType(zData.filterMap(x -> x.setprop).filterMap(x -> x.folder_type).map(FolderType.R::valueOf));
    }

    private static void initializeMongoFolder(MongoResource result, FolderDjfsResource folder,
            MongoResourceZData zData)
    {
        initializeMongoResource(result, folder, zData);
        result.type = "dir";

        result.data.get().utime = folder.getUploadTime().map(InstantUtils::toSeconds);
        if (folder.getCreationTime().isPresent()) {
            initializeMeta(zData);
            zData.meta.get().ctime = folder.getCreationTime().map(InstantUtils::toSeconds);
        }
        result.data.get().mtime = folder.getModificationTime().map(InstantUtils::toSeconds);

        if (folder.getFolderType().isPresent()) {
            initializeSetprop(zData);
            zData.setprop.get().folder_type = folder.getFolderType().map(x -> x.toString().toLowerCase());
        }
    }

    private static void initializeMongoFile(MongoResource result, FileDjfsResource file,
            MongoResourceZData zData)
    {
        initializeMongoResource(result, file, zData);
        initializeMeta(zData);
        result.type = "file";

        result.data.get().size = Option.of(file.getSize());
        result.hid = Option.of(file.getHid().getBytes(StandardCharsets.UTF_8));
        zData.meta.get().md5 = Option.of(file.getMd5());
        zData.meta.get().sha256 = Option.of(file.getSha256());

        ArrayListF<MongoResourceStid> stids = new ArrayListF<>();

        MongoResourceStid fileStid = new MongoResourceStid();
        fileStid.type = "file_mid";
        fileStid.stid = file.getFileStid();
        stids.add(fileStid);

        MongoResourceStid digestStid = new MongoResourceStid();
        digestStid.type = "digest_mid";
        digestStid.stid = file.getDigestStid();
        stids.add(digestStid);

        if (file.getPreviewStid().isPresent()) {
            MongoResourceStid previewStid = new MongoResourceStid();
            previewStid.type = "pmid";
            previewStid.stid = file.getPreviewStid().get();
            stids.add(previewStid);
        }

        if (file.getFotkiTags().isPresent()) {
            initializeSetprop(zData);
            zData.setprop.get().fotki_tags = file.getFotkiTags();
        }

        result.data.get().stids = Option.of(stids);

        result.data.get().utime = file.getUploadTime().map(InstantUtils::toSeconds);
        zData.meta.get().ctime = Option.of(InstantUtils.toSeconds(file.getCreationTime()));
        result.data.get().mtime = Option.of(InstantUtils.toSeconds(file.getModificationTime()));
        result.data.get().etime = file.getExifTime().map(InstantUtils::toSecondsLong);

        zData.meta.get().drweb = file.getAntiVirusScanStatus().map(AntiVirusScanStatus::value);
        zData.meta.get().source = file.getSource();
        result.data.get().mimetype = file.getMimetype();
        result.data.get().mt = file.getMediaType().map(MediaType::value);
    }
}
