package ru.yandex.chemodan.app.djfs.core.filesystem.model.mongo.parsing;

import java.util.Map;
import java.util.Objects;
import java.util.UUID;

import org.bson.BsonBinary;
import org.bson.BsonBoolean;
import org.bson.BsonDocument;
import org.bson.BsonInt32;
import org.bson.BsonInt64;
import org.bson.BsonString;
import org.bson.BsonValue;
import org.joda.time.Instant;

import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.chemodan.app.djfs.core.util.InstantUtils;
import ru.yandex.chemodan.app.djfs.core.util.UuidUtils;
import ru.yandex.misc.lang.CharsetUtils;

/**
 * @author eoshch
 */
public abstract class MongoFactoryBase {

    protected static String getString(String key, BsonDocument document) {
        if (!document.containsKey(key)) {
            throw new MissingMongoFieldException(key);
        }

        BsonValue value = document.get(key);
        if (!(value instanceof BsonString)) {
            throw new WrongMongoFieldTypeException(key, value.getClass());
        }
        return ((BsonString) value).getValue();
    }

    protected static String getString(Map.Entry<String, BsonValue> entry) {
        BsonValue value = entry.getValue();
        if (value instanceof BsonString) {
            return ((BsonString) value).getValue();
        }
        throw new WrongMongoFieldTypeException(entry.getKey(), value.getClass());
    }

    protected static int getInt(Map.Entry<String, BsonValue> entry) {
        BsonValue value = entry.getValue();
        if (value instanceof BsonInt32) {
            return ((BsonInt32) value).getValue();
        }
        throw new WrongMongoFieldTypeException(entry.getKey(), value.getClass());
    }

    protected static long getLong(Map.Entry<String, BsonValue> entry) {
        BsonValue value = entry.getValue();
        if (value instanceof BsonInt32) {
            return ((BsonInt32) value).getValue();
        }
        if (value instanceof BsonInt64) {
            return ((BsonInt64) value).getValue();
        }
        throw new WrongMongoFieldTypeException(entry.getKey(), value.getClass());
    }

    protected static String getBinaryAsHex(Map.Entry<String, BsonValue> entry) {
        BsonValue value = entry.getValue();
        if (value instanceof BsonBinary) {
            return CharsetUtils.decodeUtf8(((BsonBinary) value).getData());
        }
        throw new WrongMongoFieldTypeException(entry.getKey(), value.getClass());
    }

    protected static byte[] getBinaryAsByteArray(Map.Entry<String, BsonValue> entry) {
        BsonValue value = entry.getValue();
        if (value instanceof BsonBinary) {
            return ((BsonBinary) value).getData();
        }
        throw new WrongMongoFieldTypeException(entry.getKey(), value.getClass());
    }

    protected static UUID getUuid(Map.Entry<String, BsonValue> entry) {
        return UuidUtils.fromHex(getString(entry));
    }

    protected static Instant getInstant(Map.Entry<String, BsonValue> entry) {
        BsonValue value = entry.getValue();
        if (value instanceof BsonInt32) {
            return InstantUtils.fromSeconds(((BsonInt32) value).getValue());
        }
        if (value instanceof BsonInt64) {
            return InstantUtils.fromSeconds(((BsonInt64) value).getValue());
        }
        if (value instanceof BsonString) {
            return InstantUtils.fromSeconds(((BsonString) value).getValue());
        }
        throw new WrongMongoFieldTypeException(entry.getKey(), value.getClass());
    }

    protected static boolean getBoolean(Map.Entry<String, BsonValue> entry) {
        BsonValue value = entry.getValue();
        if (value instanceof BsonBoolean) {
            return ((BsonBoolean) value).getValue();
        }
        if (value instanceof BsonInt32) {
            return ((BsonInt32) value).getValue() != 0;
        }
        if (value instanceof BsonInt64) {
            return ((BsonInt64) value).getValue() != 0;
        }
        if (value instanceof BsonString) {
            return !Objects.equals(((BsonString) value).getValue(), "0");
        }
        throw new WrongMongoFieldTypeException(entry.getKey(), value.getClass());
    }

    protected static BsonDocument getDocument(Map.Entry<String, BsonValue> entry) {
        BsonValue value = entry.getValue();
        if (value instanceof BsonDocument) {
            return (BsonDocument) value;
        }
        throw new WrongMongoFieldTypeException(entry.getKey(), value.getClass());
    }

    protected static Tuple2<String, String> getMapEntry(Map.Entry<String, BsonValue> entry) {
        BsonValue value = entry.getValue();
        if (value instanceof BsonString) {
            return Tuple2.tuple(entry.getKey(), ((BsonString) value).getValue());
        }
        throw new WrongMongoFieldTypeException(entry.getKey(), value.getClass());
    }
}
