package ru.yandex.chemodan.app.djfs.core.filesystem.model.mongo.parsing;

import java.util.Map;

import org.bson.BsonArray;
import org.bson.BsonDocument;
import org.bson.BsonValue;

import ru.yandex.chemodan.app.djfs.core.db.mongo.MongoUtil;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.AntiVirusScanStatus;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.DjfsFileId;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.DjfsResourcePath;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.FileDjfsResource;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.MediaType;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.mongo.ResourceFactory;
import ru.yandex.chemodan.app.djfs.core.user.DjfsUid;
import ru.yandex.chemodan.app.djfs.core.util.ZipUtils;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

/**
 * @author eoshch
 */
public class MongoFileDjfsResourceParser extends MongoFactoryBase {
    private static final Logger logger = LoggerFactory.getLogger(ResourceFactory.class);

    public static FileDjfsResource create(BsonDocument document) {
        FileDjfsResource.Builder builder = FileDjfsResource.builder();
        String uid = null;
        String path = null;
        for (Map.Entry<String, BsonValue> entry : document.entrySet()) {
            String key = entry.getKey();
            switch (key) {
                case "_id":
                    builder.id(getUuid(entry));
                    break;
                case "key":
                    path = getString(entry);
                    break;
                case "uid":
                    uid = getString(entry);
                    break;
                case "type":
                    break;
                case "parent":
                    builder.parentId(getUuid(entry));
                    break;
                case "version":
                    builder.version(getLong(entry));
                    break;
                case "hid":
                    builder.hid(getBinaryAsHex(entry));
                    break;
                case "dtime":
                    builder.hiddenAppendTime(getInstant(entry));
                    break;
                case "data":
                    processData("data", builder, getDocument(entry));
                    break;
                case "zdata":
                    BsonDocument zdata = BsonDocument.parse(ZipUtils.decompressToString(getBinaryAsByteArray(entry)));
                    processZdata(builder, zdata);
                    break;
                default:
                    throw new UnexpectedMongoFieldException(key);
            }
        }
        builder.path(DjfsResourcePath.cons(uid, path));

        if (!builder.getUploadTime().isPresent()) {
            builder.uploadTime(builder.getCreationTime());
        }

        return builder.build();
    }

    private static FileDjfsResource.Builder processData(String prefix, FileDjfsResource.Builder builder,
            BsonDocument document)
    {
        for (Map.Entry<String, BsonValue> entry : document.entrySet()) {
            String key = entry.getKey();
            switch (key) {
                case "size":
                    builder.size(getLong(entry));
                    break;
                case "md5":
                    builder.md5(getString(entry));
                    break;
                case "sha256":
                    builder.sha256(getString(entry));
                    break;
                case "mimetype":
                    builder.mimetype(getString(entry));
                    break;
                case "mt":
                    builder.mediaType(MediaType.R.valueOf(getInt(entry)));
                    break;
                case "ctime":
                    builder.creationTime(getInstant(entry));
                    break;
                case "mtime":
                    builder.modificationTime(getInstant(entry));
                    break;
                case "etime":
                    builder.exifTime(getInstant(entry));
                    break;
                case "utime":
                    builder.uploadTime(getInstant(entry));
                    break;
                case "visible":
                    builder.isVisible(getBoolean(entry));
                    break;
                case "public":
                    builder.isPublic(getBoolean(entry));
                    break;
                case "blocked":
                    builder.isBlocked(getBoolean(entry));
                    break;
                case "file_id":
                    builder.fileId(DjfsFileId.cons(getString(entry)));
                    break;
                case "stids":
                    processStids(builder, entry.getValue());
                    break;
                case "source":
                    builder.source(getString(entry));
                    break;
                case "drweb":
                    int value = getInt(entry);
                    builder.antiVirusScanStatus(
                            value == 0 ? AntiVirusScanStatus.CLEAN : AntiVirusScanStatus.R.fromValue(value));
                    break;
                case "modify_uid":
                    builder.modifyUid(DjfsUid.cons(getString(entry)));
                    break;
                case "original_id":
                    builder.trashAppendOriginalPath(getString(entry));
                    break;
                case "download_counter":
                    builder.downloadCounter(getInt(entry));
                    break;
                case "height":
                    builder.height(getInt(entry));
                    break;
                case "width":
                    builder.width(getInt(entry));
                    break;
                case "angle":
                    builder.angle(getInt(entry));
                    break;
                case "is_live_photo":
                    builder.isLivePhoto(getBoolean(entry));
                    break;
                case "live_video_id":
                    builder.liveVideoId(getUuid(entry));
                    break;
                case "yarovaya_mark":
                    builder.hasYarovayaMark(getBoolean(entry));
                    break;
                case "file_id_zipped":
                    // ignored
                    break;
                case "hid_updated":
                    // only used by hardlink updater, ignored for now
                    break;
                default:
                    throw new UnexpectedMongoFieldException(prefix + "." + key);
            }
        }
        return builder;
    }

    private static FileDjfsResource.Builder processZdata(FileDjfsResource.Builder builder, BsonDocument document) {
        for (Map.Entry<String, BsonValue> entry : document.entrySet()) {
            String key = entry.getKey();
            switch (key) {
                case "meta":
                    processData("zdata.meta", builder, getDocument(entry));
                    break;
                case "setprop":
                    processSetprop("zdata.setprop", builder, getDocument(entry));
                    break;
                case "pub":
                    processPub("zdata.pub", builder, getDocument(entry));
                    break;
                default:
                    throw new UnexpectedMongoFieldException("zdata." + key);
            }
        }
        return builder;
    }

    private static FileDjfsResource.Builder processSetprop(String prefix, FileDjfsResource.Builder builder,
            BsonDocument document)
    {
        for (Map.Entry<String, BsonValue> entry : document.entrySet()) {
            String key = entry.getKey();
            switch (key) {
                case "append_time":
                    builder.trashAppendTime(getInstant(entry));
                    break;
                case "folder_url":
                    builder.folderUrl(getString(entry));
                    break;
                case "fotki_tags":
                    builder.fotkiTags(getString(entry));
                    break;
                case "published":
                    builder.isPublished(getBoolean(entry));
                    break;
                case "external_url":
                    builder.externalUrl(getString(entry));
                    break;
                case "custom_properties":
                    builder.customProperties(getString(entry));
                    break;
                case "file_id_zipped":
                    // ignored
                    break;
                case "total_results_count":
                    // ignored
                    break;
                case "revision":
                    // ignored
                    break;
                case "preview_sizes":
                    // ignored
                    break;
                case "page_blocked_items_num":
                    //ignored
                    break;
                case "video_info":
                    builder.videoInfo(MongoUtil.serializeToJson(getDocument(entry)));
                    break;
                default:
                    builder.externalProperty(getMapEntry(entry));
            }
        }
        return builder;
    }

    private static FileDjfsResource.Builder processPub(String prefix, FileDjfsResource.Builder builder,
            BsonDocument document)
    {
        for (Map.Entry<String, BsonValue> entry : document.entrySet()) {
            String key = entry.getKey();
            switch (key) {
                case "short_url":
                    builder.shortUrl(getString(entry));
                    break;
                case "symlink":
                    builder.symlink(getString(entry));
                    break;
                case "public_hash":
                    builder.publicHash(getString(entry));
                    break;
                default:
                    throw new UnexpectedMongoFieldException(prefix + "." + key);
            }
        }
        return builder;
    }

    private static FileDjfsResource.Builder processStids(FileDjfsResource.Builder builder, BsonValue value) {
        if (!(value instanceof BsonArray)) {
            throw new WrongMongoFieldTypeException("data.stids", value.getClass());
        }
        BsonArray items = (BsonArray) value;
        for (int i = 0; i < items.size(); i++) {
            BsonValue item = items.get(i);
            if (!(item instanceof BsonDocument)) {
                throw new WrongMongoFieldTypeException("data.stids." + i, value.getClass());
            }
            String type = getString("type", (BsonDocument) item);
            String stid = getString("stid", (BsonDocument) item);
            switch (type) {
                case "file_mid":
                    builder.fileStid(stid);
                    break;
                case "digest_mid":
                    builder.digestStid(stid);
                    break;
                case "pmid":
                    builder.previewStid(stid);
                    break;
                default:
                    throw new UnexpectedMongoFieldException("stid of type " + type);
            }
        }
        return builder;
    }
}
