package ru.yandex.chemodan.app.djfs.core.filesystem.model.mongo.parsing;

import org.bson.BsonArray;
import org.bson.BsonBinary;
import org.bson.BsonBoolean;
import org.bson.BsonDocument;
import org.bson.BsonInt32;
import org.bson.BsonString;

import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.chemodan.app.djfs.core.db.mongo.MongoUtil;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.FileDjfsResource;
import ru.yandex.chemodan.app.djfs.core.util.InstantUtils;
import ru.yandex.chemodan.app.djfs.core.util.UuidUtils;
import ru.yandex.chemodan.app.djfs.core.util.ZipUtils;
import ru.yandex.misc.lang.CharsetUtils;

/**
 * @author eoshch
 */
public class MongoFileDjfsResourceSerializer {
    public static BsonDocument serialize(FileDjfsResource file) {
        BsonDocument result = new BsonDocument();
        result.append("_id", new BsonString(UuidUtils.toHexString(file.getId())));
        result.append("uid", new BsonString(file.getUid().asString()));
        result.append("type", new BsonString("file"));
        result.append("key", new BsonString(file.getPath().getPath()));
        result.append("hid", new BsonBinary(CharsetUtils.encodeUtf8ToArray(file.getHid())));

        if (file.getParentId().isPresent()) {
            result.append("parent", new BsonString(UuidUtils.toHexString(file.getParentId().get())));
        }
        if (file.getVersion().isPresent()) {
            result.append("version", MongoUtil.toNumber(file.getVersion().get()));
        }
        if (file.getHiddenAppendTime().isPresent()) {
            result.append("dtime", MongoUtil.toNumber(InstantUtils.toSecondsLong(file.getHiddenAppendTime().get())));
        }

        BsonDocument data = new BsonDocument();
        if (file.getFileId().isPresent()) {
            data.append("file_id", new BsonString(file.getFileId().get().getValue()));
        }
        data.append("size", MongoUtil.toNumber(file.getSize()));
        data.append("visible", new BsonInt32(file.isVisible() ? 1 : 0));

        data.append("mtime", MongoUtil.toNumber(InstantUtils.toSecondsLong(file.getModificationTime())));
        if (file.getUploadTime().isPresent()) {
            data.append("utime", MongoUtil.toNumber(InstantUtils.toSecondsLong(file.getUploadTime().get())));
        }
        if (file.getExifTime().isPresent()) {
            data.append("etime", MongoUtil.toNumber(InstantUtils.toSecondsLong(file.getExifTime().get())));
        }

        if (file.getMimetype().isPresent()) {
            data.append("mimetype", new BsonString(file.getMimetype().get()));
        }
        if (file.getMediaType().isPresent()) {
            data.append("mt", new BsonInt32(file.getMediaType().get().value()));
        }

        if (file.getTrashAppendOriginalPath().isPresent()) {
            data.append("original_id", new BsonString(file.getTrashAppendOriginalPath().get()));
        }

        if (file.isPublic()) {
            data.append("public", new BsonInt32(1));
        }
        if (file.isBlocked()) {
            data.append("blocked", new BsonInt32(1));
        }

        if (file.getDownloadCounter().isPresent()) {
            data.append("download_counter", MongoUtil.toNumber(file.getDownloadCounter().get()));
        }
        if (file.getModifyUid().isPresent()) {
            data.append("modify_uid", new BsonString(file.getModifyUid().get().asString()));
        }

        if (file.getHeight().isPresent()) {
            data.append("height", new BsonInt32(file.getHeight().get()));
        }
        if (file.getWidth().isPresent()) {
            data.append("width", new BsonInt32(file.getWidth().get()));
        }
        if (file.getAngle().isPresent()) {
            data.append("angle", new BsonInt32(file.getAngle().get()));
        }

        if (file.isLivePhoto()) {
            data.append("is_live_photo", new BsonBoolean(true));
        }

        if (file.hasYarovayaMark()) {
            data.append("yarovaya_mark", new BsonBoolean(true));
        }

        BsonArray stids = new BsonArray();
        stids.add(new BsonDocument()
                .append("stid", new BsonString(file.getFileStid()))
                .append("type", new BsonString("file_mid")));
        stids.add(new BsonDocument()
                .append("stid", new BsonString(file.getDigestStid()))
                .append("type", new BsonString("digest_mid")));
        if (file.getPreviewStid().isPresent()) {
            stids.add(new BsonDocument()
                    .append("stid", new BsonString(file.getPreviewStid().get()))
                    .append("type", new BsonString("pmid")));
        }
        data.append("stids", stids);

        BsonDocument zdata = new BsonDocument();

        BsonDocument meta = new BsonDocument();
        meta.append("md5", new BsonString(file.getMd5()));
        meta.append("sha256", new BsonString(file.getSha256()));
        if (file.getSource().isPresent()) {
            meta.append("source", new BsonString(file.getSource().get()));
        }
        if (file.getAntiVirusScanStatus().isPresent()) {
            meta.append("drweb", new BsonInt32(file.getAntiVirusScanStatus().get().value()));
        }
        meta.append("ctime", MongoUtil.toNumber(InstantUtils.toSecondsLong(file.getCreationTime())));
        if (!meta.isEmpty()) {
            zdata.append("meta", meta);
        }

        BsonDocument setprop = new BsonDocument();
        if (file.getTrashAppendTime().isPresent()) {
            setprop.append("append_time",
                    MongoUtil.toNumber(InstantUtils.toSecondsLong(file.getTrashAppendTime().get())));
        }
        if (file.isPublished()) {
            setprop.append("published", new BsonInt32(1));
        }
        if (file.getFolderUrl().isPresent()) {
            setprop.append("folder_url", new BsonString(file.getFolderUrl().get()));
        }
        if (file.getExternalUrl().isPresent()) {
            setprop.append("external_url", new BsonString(file.getExternalUrl().get()));
        }
        if (file.getFotkiTags().isPresent()) {
            setprop.append("fotki_tags", new BsonString(file.getFotkiTags().get()));
        }
        if (file.getCustomProperties().isPresent()) {
            setprop.append("custom_properties", new BsonString(file.getCustomProperties().get()));
        }
        if (file.getVideoInfo().isPresent()) {
            setprop.append("video_info", BsonDocument.parse(file.getVideoInfo().get()));
        }
        for (Tuple2<String, String> entry : file.getExternalProperties().entries()) {
            setprop.append(entry._1, new BsonString(entry._2));
        }
        if (!setprop.isEmpty()) {
            zdata.append("setprop", setprop);
        }

        BsonDocument pub = new BsonDocument();
        if (file.getShortUrl().isPresent()) {
            pub.append("short_url", new BsonString(file.getShortUrl().get()));
        }
        if (file.getSymlink().isPresent()) {
            pub.append("symlink", new BsonString(file.getSymlink().get()));
        }
        if (file.getPublicHash().isPresent()) {
            pub.append("public_hash", new BsonString(file.getPublicHash().get()));
        }
        if (!pub.isEmpty()) {
            zdata.append("pub", pub);
        }

        if (!data.isEmpty()) {
            result.append("data", data);
        }
        if (!zdata.isEmpty()) {
            result.append("zdata", new BsonBinary(ZipUtils.compressString(MongoUtil.serializeToJson(zdata), 1)));
        }
        return result;
    }
}
