package ru.yandex.chemodan.app.djfs.core.filesystem.model.mongo.parsing;

import java.util.Map;
import java.util.Objects;

import org.bson.BsonDocument;
import org.bson.BsonValue;

import ru.yandex.chemodan.app.djfs.core.filesystem.model.DjfsFileId;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.DjfsResourceArea;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.DjfsResourcePath;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.FolderDjfsResource;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.FolderType;
import ru.yandex.chemodan.app.djfs.core.user.DjfsUid;
import ru.yandex.chemodan.app.djfs.core.util.ZipUtils;

/**
 * @author eoshch
 */
public class MongoFolderDjfsResourceParser extends MongoFactoryBase {
    public static FolderDjfsResource create(DjfsResourceArea area, BsonDocument document) {
        FolderDjfsResource.Builder builder = FolderDjfsResource.builder();
        String uid = null;
        String path = null;
        for (Map.Entry<String, BsonValue> entry : document.entrySet()) {
            String key = entry.getKey();
            switch (key) {
                case "_id":
                    builder.id(getUuid(entry));
                    break;
                case "key":
                    path = getString(entry);
                    break;
                case "uid":
                    uid = getString(entry);
                    break;
                case "type":
                    break;
                case "parent":
                    builder.parentId(getUuid(entry));
                    break;
                case "version":
                    builder.version(getLong(entry));
                    break;
                case "dtime":
                    builder.hiddenAppendTime(getInstant(entry));
                    break;
                case "data":
                    processData("data", builder, getDocument(entry));
                    break;
                case "zdata":
                    BsonDocument zdata = BsonDocument.parse(ZipUtils.decompressToString(getBinaryAsByteArray(entry)));
                    processZdata(builder, zdata);
                    break;
                case "lock":
                    // ignore
                    break;
                default:
                    throw new UnexpectedMongoFieldException(key);
            }
        }
        if (Objects.equals(path, DjfsResourcePath.ROOT)) {
            builder.path(DjfsResourcePath.root(uid, area));
        } else {
            builder.path(DjfsResourcePath.cons(uid, path));
        }
        return builder.build();
    }

    private static FolderDjfsResource.Builder processData(String prefix, FolderDjfsResource.Builder builder,
            BsonDocument document)
    {
        for (Map.Entry<String, BsonValue> entry : document.entrySet()) {
            String key = entry.getKey();
            switch (key) {
                case "ctime":
                    builder.creationTime(getInstant(entry));
                    break;
                case "mtime":
                    builder.modificationTime(getInstant(entry));
                    break;
                case "utime":
                    builder.uploadTime(getInstant(entry));
                    break;
                case "visible":
                    builder.isVisible(getBoolean(entry));
                    break;
                case "public":
                    builder.isPublic(getBoolean(entry));
                    break;
                case "blocked":
                    builder.isBlocked(getBoolean(entry));
                    break;
                case "file_id":
                    builder.fileId(DjfsFileId.cons(getString(entry)));
                    break;
                case "modify_uid":
                    builder.modifyUid(DjfsUid.cons(getString(entry)));
                    break;
                case "original_id":
                    builder.trashAppendOriginalPath(getString(entry));
                    break;
                case "download_counter":
                    builder.downloadCounter(getInt(entry));
                    break;
                case "yarovaya_mark":
                    builder.hasYarovayaMark(getBoolean(entry));
                    break;
                case "file_id_zipped":
                    // ignored
                    break;
                case "meta":
                    if (entry.getValue().isArray() && entry.getValue().asArray().isEmpty()) {
                        // ignored
                    } else {
                        throw new UnexpectedMongoFieldException(prefix + "." + key);
                    }
                    break;
                default:
                    throw new UnexpectedMongoFieldException(prefix + "." + key);
            }
        }
        return builder;
    }

    private static FolderDjfsResource.Builder processZdata(FolderDjfsResource.Builder builder, BsonDocument document) {
        for (Map.Entry<String, BsonValue> entry : document.entrySet()) {
            String key = entry.getKey();
            switch (key) {
                case "meta":
                    processData("zdata.meta", builder, getDocument(entry));
                    break;
                case "setprop":
                    processSetprop("zdata.setprop", builder, getDocument(entry));
                    break;
                case "pub":
                    processPub("zdata.pub", builder, getDocument(entry));
                    break;
                default:
                    throw new UnexpectedMongoFieldException("zdata." + key);
            }
        }
        return builder;
    }

    private static FolderDjfsResource.Builder processSetprop(String prefix, FolderDjfsResource.Builder builder,
            BsonDocument document)
    {
        for (Map.Entry<String, BsonValue> entry : document.entrySet()) {
            String key = entry.getKey();
            switch (key) {
                case "append_time":
                    builder.trashAppendTime(getInstant(entry));
                    break;
                case "folder_url":
                    builder.folderUrl(getString(entry));
                    break;
                case "published":
                    builder.isPublished(getBoolean(entry));
                    break;
                case "folder_type":
                    builder.folderType(FolderType.R.valueOf(getString(entry)));
                    break;
                case "custom_properties":
                    builder.customProperties(getString(entry));
                    break;
                case "last_import_time":
                    builder.lastImportTime(getInstant(entry));
                case "file_id_zipped":
                    // ignored
                    break;
                case "total_results_count":
                    // ignored
                    break;
                case "revision":
                    // ignored
                    break;
                case "folder_id":
                    if (entry.getValue().isDocument())
                        // ignored
                        break;
                    builder.externalProperty(getMapEntry(entry));
                case "lenta_block_id":
                    if (entry.getValue().isDocument())
                        // ignored
                        break;
                    builder.externalProperty(getMapEntry(entry));
                default:
                    builder.externalProperty(getMapEntry(entry));
            }
        }
        return builder;
    }

    private static FolderDjfsResource.Builder processPub(String prefix, FolderDjfsResource.Builder builder,
            BsonDocument document)
    {
        for (Map.Entry<String, BsonValue> entry : document.entrySet()) {
            String key = entry.getKey();
            switch (key) {
                case "short_url":
                    builder.shortUrl(getString(entry));
                    break;
                case "symlink":
                    builder.symlink(getString(entry));
                    break;
                case "public_hash":
                    builder.publicHash(getString(entry));
                    break;
                default:
                    throw new UnexpectedMongoFieldException(prefix + "." + key);
            }
        }
        return builder;
    }

}
