package ru.yandex.chemodan.app.djfs.core.filesystem.model.mongo.parsing;

import org.bson.BsonBinary;
import org.bson.BsonBoolean;
import org.bson.BsonDocument;
import org.bson.BsonInt32;
import org.bson.BsonString;

import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.chemodan.app.djfs.core.db.mongo.MongoUtil;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.FolderDjfsResource;
import ru.yandex.chemodan.app.djfs.core.util.InstantUtils;
import ru.yandex.chemodan.app.djfs.core.util.UuidUtils;
import ru.yandex.chemodan.app.djfs.core.util.ZipUtils;

/**
 * @author eoshch
 */
public class MongoFolderDjfsResourceSerializer {
    public static BsonDocument serialize(FolderDjfsResource folder) {
        BsonDocument result = new BsonDocument();
        result.append("_id", new BsonString(UuidUtils.toHexString(folder.getId())));
        result.append("uid", new BsonString(folder.getUid().asString()));
        result.append("type", new BsonString("dir"));
        result.append("key", new BsonString(folder.getPath().getPath()));

        if (folder.getParentId().isPresent()) {
            result.append("parent", new BsonString(UuidUtils.toHexString(folder.getParentId().get())));
        }
        if (folder.getVersion().isPresent()) {
            result.append("version", MongoUtil.toNumber(folder.getVersion().get()));
        }
        if (folder.getHiddenAppendTime().isPresent()) {
            result.append("dtime", MongoUtil.toNumber(InstantUtils.toSecondsLong(folder.getHiddenAppendTime().get())));
        }

        BsonDocument data = new BsonDocument();
        if (folder.getFileId().isPresent()) {
            data.append("file_id", new BsonString(folder.getFileId().get().getValue()));
        }
        data.append("visible", new BsonInt32(folder.isVisible() ? 1 : 0));

        if (folder.getModificationTime().isPresent()) {
            data.append("mtime", MongoUtil.toNumber(InstantUtils.toSecondsLong(folder.getModificationTime().get())));
        }
        if (folder.getUploadTime().isPresent()) {
            data.append("utime", MongoUtil.toNumber(InstantUtils.toSecondsLong(folder.getUploadTime().get())));
        }

        if (folder.getTrashAppendOriginalPath().isPresent()) {
            data.append("original_id", new BsonString(folder.getTrashAppendOriginalPath().get()));
        }

        if (folder.isPublic()) {
            data.append("public", new BsonInt32(1));
        }
        if (folder.isBlocked()) {
            data.append("blocked", new BsonInt32(1));
        }

        if (folder.getDownloadCounter().isPresent()) {
            data.append("download_counter", MongoUtil.toNumber(folder.getDownloadCounter().get()));
        }
        if (folder.getModifyUid().isPresent()) {
            data.append("modify_uid", new BsonString(folder.getModifyUid().get().asString()));
        }

        if (folder.hasYarovayaMark()) {
            data.append("yarovaya_mark", new BsonBoolean(true));
        }

        BsonDocument zdata = new BsonDocument();

        BsonDocument meta = new BsonDocument();
        if (folder.getCreationTime().isPresent()) {
            meta.append("ctime", MongoUtil.toNumber(InstantUtils.toSecondsLong(folder.getCreationTime().get())));
        }
        if (!meta.isEmpty()) {
            zdata.append("meta", meta);
        }

        BsonDocument setprop = new BsonDocument();
        if (folder.getTrashAppendTime().isPresent()) {
            setprop.append("append_time", MongoUtil.toNumber(InstantUtils.toSecondsLong(
                    folder.getTrashAppendTime().get())));
        }
        if (folder.isPublished()) {
            setprop.append("published", new BsonInt32(1));
        }
        if (folder.getFolderUrl().isPresent()) {
            setprop.append("folder_url", new BsonString(folder.getFolderUrl().get()));
        }
        if (folder.getFolderType().isPresent()) {
            setprop.append("folder_type", new BsonString(folder.getFolderType().get().name().toLowerCase()));
        }
        if (folder.getCustomProperties().isPresent()) {
            setprop.append("custom_properties", new BsonString(folder.getCustomProperties().get()));
        }
        if (folder.getLastImportTime().isPresent()) {
            setprop.append("last_import_time", MongoUtil.toNumber(InstantUtils.toSecondsLong(
                    folder.getLastImportTime().get())));
        }
        for (Tuple2<String, String> entry : folder.getExternalProperties().entries()) {
            setprop.append(entry._1, new BsonString(entry._2));
        }
        if (!setprop.isEmpty()) {
            zdata.append("setprop", setprop);
        }

        BsonDocument pub = new BsonDocument();
        if (folder.getShortUrl().isPresent()) {
            pub.append("short_url", new BsonString(folder.getShortUrl().get()));
        }
        if (folder.getSymlink().isPresent()) {
            pub.append("symlink", new BsonString(folder.getSymlink().get()));
        }
        if (folder.getPublicHash().isPresent()) {
            pub.append("public_hash", new BsonString(folder.getPublicHash().get()));
        }
        if (!pub.isEmpty()) {
            zdata.append("pub", pub);
        }

        if (!data.isEmpty()) {
            result.append("data", data);
        }
        if (!zdata.isEmpty()) {
            result.append("zdata", new BsonBinary(ZipUtils.compressString(MongoUtil.serializeToJson(zdata), 1)));
        }
        return result;
    }
}
