package ru.yandex.chemodan.app.djfs.core.filesystem.operation.copy;

import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;

import com.google.common.base.Stopwatch;
import org.joda.time.Duration;

import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.chemodan.app.djfs.core.SafeCloseable;
import ru.yandex.chemodan.app.djfs.core.filesystem.CopyActivity;
import ru.yandex.chemodan.app.djfs.core.filesystem.CopyIntent;
import ru.yandex.chemodan.app.djfs.core.filesystem.DjfsPrincipal;
import ru.yandex.chemodan.app.djfs.core.filesystem.DjfsResourceDao;
import ru.yandex.chemodan.app.djfs.core.filesystem.Filesystem;
import ru.yandex.chemodan.app.djfs.core.filesystem.FilesystemAccessController;
import ru.yandex.chemodan.app.djfs.core.filesystem.QuotaManager;
import ru.yandex.chemodan.app.djfs.core.filesystem.exception.NoFreeSpaceException;
import ru.yandex.chemodan.app.djfs.core.filesystem.exception.NoParentFolderException;
import ru.yandex.chemodan.app.djfs.core.filesystem.exception.ParentIsFileException;
import ru.yandex.chemodan.app.djfs.core.filesystem.exception.ResourceExistsException;
import ru.yandex.chemodan.app.djfs.core.filesystem.iteration.ResourceIterator;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.AntiVirusScanStatus;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.DjfsResource;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.DjfsResourcePath;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.FileDjfsResource;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.FolderDjfsResource;
import ru.yandex.chemodan.app.djfs.core.history.EventHistoryLogger;
import ru.yandex.chemodan.app.djfs.core.lock.LockManager;
import ru.yandex.chemodan.app.djfs.core.operations.MpfsOperationHandler;
import ru.yandex.chemodan.app.djfs.core.operations.MpfsOperationHandlerContext;
import ru.yandex.chemodan.app.djfs.core.operations.Operation;
import ru.yandex.chemodan.app.djfs.core.share.ShareInfo;
import ru.yandex.chemodan.app.djfs.core.share.ShareInfoManager;
import ru.yandex.chemodan.app.djfs.core.tasks.DelayDjfsCeleryOnetimeTaskException;
import ru.yandex.chemodan.app.djfs.core.user.DjfsUid;
import ru.yandex.chemodan.app.djfs.core.user.UserDao;
import ru.yandex.chemodan.app.djfs.core.web.ConnectionIdHolder;
import ru.yandex.commune.bazinga.impl.TaskId;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

/**
 * @author eoshch
 */
public class CopyOperationHandler extends MpfsOperationHandler {
    private static final Logger logger = LoggerFactory.getLogger(CopyOperationHandler.class);

    private final DjfsResourceDao djfsResourceDao;
    private final Filesystem filesystem;
    private final LockManager lockManager;
    private final ShareInfoManager shareInfoManager;
    private final UserDao userDao;
    private final ResourceIterator.Factory resourceIteratorFactory;
    private final EventHistoryLogger eventHistoryLogger;
    private final QuotaManager quotaManager;
    private final FilesystemAccessController acl;

    private final Duration lockUpdateInterval;
    private final Duration internalStateUpdateInterval;

    public CopyOperationHandler(MpfsOperationHandlerContext mpfsOperationHandlerContext,
            CopyOperationProperties properties, DjfsResourceDao djfsResourceDao,
            Filesystem filesystem, LockManager lockManager, ShareInfoManager shareInfoManager, UserDao userDao,
            ResourceIterator.Factory resourceIteratorFactory, EventHistoryLogger eventHistoryLogger,
            QuotaManager quotaManager, FilesystemAccessController filesystemAccessController)
    {
        super(mpfsOperationHandlerContext);
        this.djfsResourceDao = djfsResourceDao;
        this.filesystem = filesystem;
        this.lockManager = lockManager;
        this.shareInfoManager = shareInfoManager;
        this.userDao = userDao;
        this.resourceIteratorFactory = resourceIteratorFactory;
        this.eventHistoryLogger = eventHistoryLogger;
        this.quotaManager = quotaManager;
        this.acl = filesystemAccessController;

        this.lockUpdateInterval = properties.getFilesystemLockUpdateInterval();
        this.internalStateUpdateInterval = properties.getInternalStateUpdateInterval();
    }

    @Override
    protected TaskId celeryTaskId() {
        return CopyOperation.TASK_ID;
    }

    @Override
    public Duration timeout() {
        return Duration.standardDays(1);
    }

    @Override
    public Status handle(Operation operation, AtomicBoolean terminated) {
        DjfsUid uid = operation.getUid();
        DjfsPrincipal principal = DjfsPrincipal.cons(uid);
        String oid = operation.getId();

        CopyOperationData data = operation.getData(CopyOperationData.B);

        try (SafeCloseable ignored = ConnectionIdHolder.set(data.getConnectionId())) {
            DjfsResourcePath sourcePath = DjfsResourcePath.cons(data.getSource());
            DjfsResourcePath destinationPath = DjfsResourcePath.cons(data.getDestination());

            Option<ShareInfo> sourceShareInfo = shareInfoManager.get(sourcePath);
            Option<ShareInfo> destinationShareInfo = shareInfoManager.get(destinationPath);

            DjfsResourcePath databaseSourcePath = sourceShareInfo
                    .filterMap(x -> x.participantPathToOwnerPath(sourcePath)).getOrElse(sourcePath);
            DjfsResourcePath databaseDestinationPath = destinationShareInfo
                    .filterMap(x -> x.participantPathToOwnerPath(destinationPath)).getOrElse(destinationPath);

            CopyIntent intent = new CopyIntent(principal, databaseSourcePath, databaseDestinationPath,
                    sourceShareInfo, destinationShareInfo);

            if (!acl.canReadResource(principal, databaseSourcePath, () -> sourceShareInfo)) {
                logger.info("no read access to " + databaseSourcePath.getFullPath());
                return Status.FAIL;
            }
            if (!acl.canWriteResource(principal, databaseDestinationPath, destinationShareInfo)) {
                logger.info("no write access to " + databaseDestinationPath.getFullPath());
                return Status.FAIL;
            }
            if (!databaseSourcePath.getArea().isActivityPermitted(intent)) {
                logger.info("operation not permitted");
                return Status.FAIL;
            }
            if (!databaseDestinationPath.getArea().isActivityPermitted(intent)) {
                logger.info("operation not permitted");
                return Status.FAIL;
            }

            if (databaseSourcePath.equals(databaseDestinationPath)) {
                logger.info("destination equals source");
                data = data.toBuilder().error(Option.of(new Operation.ErrorData(409, 37, "CopySame"))).build();
                operationDao.setData(uid, oid, data, CopyOperationData.B, operation);
                return Status.FAIL;
            }

            if (databaseSourcePath.isParentFor(databaseDestinationPath)) {
                logger.info("destination is child of source");
                data = data.toBuilder().error(Option.of(new Operation.ErrorData(409, 37, "CopySame"))).build();
                operationDao.setData(uid, oid, data, CopyOperationData.B, operation);
                return Status.FAIL;
            }

            if (lockManager.isLocked(uid)) {
                logger.info("user locked");
                return Status.FAIL;
            }

            if (isLocked(operation, data, databaseSourcePath)) {
                return Status.FAIL;
            }

            if (isLocked(operation, data, databaseDestinationPath)) {
                return Status.FAIL;
            }

            if (!tryAcquireOrRenewLock(operation, data, databaseSourcePath)) {
                return Status.FAIL;
            }

            try {
                if (!tryAcquireOrRenewLock(operation, data, databaseDestinationPath)) {
                    return Status.FAIL;
                }

                try {
                    Option<DjfsResource> sourceResource = djfsResourceDao.find(databaseSourcePath);
                    if (!sourceResource.isPresent()) {
                        logger.info("nothing to copy");
                        return Status.DONE;
                    }

                    if (sourceResource.get() instanceof FileDjfsResource) {
                        Option<Operation.ErrorData> error = copy(principal, sourceResource.get(),
                                databaseSourcePath, databaseDestinationPath);
                        if (error.isPresent()) {
                            data = data.toBuilder().error(error).build();
                            operationDao.setData(uid, oid, data, CopyOperationData.B, operation);
                            return Status.FAIL;
                        }
                        Option<DjfsResource> copy = djfsResourceDao.find2(databaseDestinationPath);
                        if (copy.isPresent()) {
                            eventHistoryLogger.log(new CopyActivity(principal, sourceResource.get(), copy.get(),
                                    sourceShareInfo, destinationShareInfo));
                        }
                        return Status.DONE;
                    } else {
                        Option<Operation.ErrorData> error = copy(principal, sourceResource.get(),
                                databaseSourcePath, databaseDestinationPath);
                        if (error.isPresent()) {
                            data = data.toBuilder().error(error).build();
                            operationDao.setData(uid, oid, data, CopyOperationData.B, operation);
                            return Status.FAIL;
                        }
                    }

                    ListF<ShareInfo> sharedSubfolders = shareInfoManager.getParticipantSharedSubfolders(sourcePath)
                            .sortedBy(x -> x.getGroupId());

                    if (!data.getCurrentSharedFolderId().isPresent()) {
                        Option<Status> result = copy(uid, principal, operation, data, databaseSourcePath,
                                databaseDestinationPath, databaseSourcePath, databaseDestinationPath, terminated);
                        if (result.isPresent()) {
                            return result.get();
                        }
                    } else {
                        String groupId = data.getCurrentSharedFolderId().get();
                        Option<ShareInfo> shareInfo = shareInfoManager.get(groupId);
                        if (shareInfo.isPresent()) {
                            Option<Status> result = copy(uid, principal, operation, data,
                                    databaseSourcePath, databaseDestinationPath, shareInfo.get(), databaseSourcePath,
                                    databaseDestinationPath, terminated);
                            if (result.isPresent()) {
                                return result.get();
                            }
                            sharedSubfolders = sharedSubfolders.filter(x -> x.getGroupId().compareTo(groupId) > 0);
                        }
                    }

                    for (ShareInfo shareInfo : sharedSubfolders) {
                        data = data.toBuilder().currentSharedFolderId(Option.of(shareInfo.getGroupId()))
                                .state(Option.empty()).build();
                        operationDao.setData(uid, oid, data, CopyOperationData.B, operation);
                        Option<Status> result = copy(uid, principal, operation, data,
                                databaseSourcePath, databaseDestinationPath, shareInfo, databaseSourcePath,
                                databaseDestinationPath, terminated);
                        if (result.isPresent()) {
                            return result.get();
                        }
                    }

                    Option<DjfsResource> copy = djfsResourceDao.find2(databaseDestinationPath);
                    if (copy.isPresent()) {
                        eventHistoryLogger.log(new CopyActivity(principal, sourceResource.get(), copy.get(),
                                sourceShareInfo, destinationShareInfo));
                    }
                    return Status.DONE;

                } finally {
                    lockManager.unlock(oid, databaseDestinationPath);
                    quotaManager.notifyIfLowSpace(uid, destinationShareInfo);
                }
            } finally {
                lockManager.unlock(oid, databaseSourcePath);
            }
        }
    }

    private Option<Status> copy(DjfsUid uid, DjfsPrincipal principal, Operation operation,
            CopyOperationData data, DjfsResourcePath sourcePath, DjfsResourcePath destinationPath,
            ShareInfo shareInfo, DjfsResourcePath sourceLockPath, DjfsResourcePath destinationLockPath,
            AtomicBoolean terminated)
    {
        Option<DjfsResourcePath> ownerRootPath = shareInfo.getRootPath(shareInfo.getOwnerUid());
        if (!ownerRootPath.isPresent()) {
            return Option.empty();
        }
        Option<DjfsResourcePath> uidRootPath = shareInfo.getRootPath(uid);
        if (!uidRootPath.isPresent()) {
            return Option.empty();
        }

        if (uidRootPath.get().equals(sourcePath)) {
            return copy(uid, principal, operation, data, ownerRootPath.get(),
                    destinationPath, sourceLockPath, destinationLockPath,
                    terminated);
        }

        return copy(uid, principal, operation, data, ownerRootPath.get(),
                uidRootPath.get().changeParent(sourcePath, destinationPath), sourceLockPath, destinationLockPath,
                terminated);
    }

    private Option<Status> copy(DjfsUid uid, DjfsPrincipal principal, Operation operation,
            CopyOperationData data, DjfsResourcePath sourcePath, DjfsResourcePath destinationPath,
            DjfsResourcePath sourceLockPath, DjfsResourcePath destinationLockPath, AtomicBoolean terminated)
    {
        String oid = operation.getId();

        ResourceIterator iterator = resourceIteratorFactory.create(sourcePath,
                ResourceIterator.TraversalType.TOP_DOWN);

        ResourceIterator.ResourceIteratorState state;

        if (!data.getState().isPresent()) {
            state = iterator.initialize();
            data = data.toBuilder().state(Option.of(state)).build();
            operationDao.setData(uid, oid, data, CopyOperationData.B, operation);
        } else {
            state = data.getState().get();
        }

        Stopwatch lockStopwatch = Stopwatch.createStarted();
        Stopwatch stateStopwatch = Stopwatch.createStarted();
        while (iterator.hasNext(state)) {
            if (terminated.get()) {
                throw new DelayDjfsCeleryOnetimeTaskException();
            }

            if (lockStopwatch.elapsed(TimeUnit.MILLISECONDS) > lockUpdateInterval.getMillis()) {
                if (!tryAcquireOrRenewLock(operation, data, sourceLockPath)) {
                    return Option.of(Status.FAIL);
                }
                if (!tryAcquireOrRenewLock(operation, data, destinationLockPath)) {
                    return Option.of(Status.FAIL);
                }
                lockStopwatch.reset();
                lockStopwatch.start();
            }

            if (stateStopwatch.elapsed(TimeUnit.MILLISECONDS) > internalStateUpdateInterval.getMillis()) {
                operationDao.setData(uid, oid, data, CopyOperationData.B, operation);
            }

            Tuple2<ResourceIterator.ResourceIteratorState, ListF<DjfsResource>> next =
                    iterator.next(state, 1);
            state = next._1;

            for (DjfsResource resource : next._2) {
                DjfsResourcePath resourcePath = resource.getPath();
                DjfsResourcePath destination = resourcePath.changeParent(sourcePath, destinationPath);
                Option<Operation.ErrorData> error = copy(principal, resource, resourcePath, destination);
                if (error.isPresent()) {
                    data = data.toBuilder().error(error).build();
                    operationDao.setData(uid, oid, data, CopyOperationData.B, operation);
                    return Option.of(Status.FAIL);
                }
            }

            data = data.toBuilder().state(Option.of(state)).build();
        }
        return Option.empty();
    }

    private Option<Operation.ErrorData> copy(DjfsPrincipal principal, DjfsResource resource,
            DjfsResourcePath source, DjfsResourcePath destination)
    {
        // do not copy public infected file to another user
        // todo: move this check to appropriate place. copy operation should not have business logic
        if (resource instanceof FileDjfsResource
                && ((FileDjfsResource) resource).getAntiVirusScanStatus()
                .map(x -> x == AntiVirusScanStatus.INFECTED).getOrElse(false)
                && resource.isFullyPublic()
                && !resource.getUid().equals(destination.getUid()))
        {
            return Option.empty();
        }

        try {
            filesystem.copySingleResource(principal, source, destination);
        } catch (ResourceExistsException e) {
            Option<DjfsResource> existingO = djfsResourceDao.find(destination);
            if (!existingO.isPresent()) {
                // where did it go? retry
                throw e;
            }
            DjfsResource existing = existingO.get();
            if (resource.getClass() != existing.getClass()) {
                logger.info("CopyOperation.Handler handled exception", e);
                logger.info("can't copy " + resource.getClass() + "over + " + existing.getClass()
                        + ", failing operation");
                return Option.of(new Operation.ErrorData(500, 1, e.getMessage()));
            } else if (resource instanceof FileDjfsResource) {
                if (((FileDjfsResource) resource).hashesEqual((FileDjfsResource) existing)) {
                    return Option.empty();
                }
                logger.info("CopyOperation.Handler handled exception", e);
                logger.info(resource.getPath().getFullPath() + " and " + existing.getPath().getFullPath()
                        + " files have different hashes, failing");
                return Option.of(new Operation.ErrorData(500, 1, e.getMessage()));
            } else if (resource instanceof FolderDjfsResource) {
                // we do not care if this is copied folder or created by someone else
                return Option.empty();
            } else {
                // should be unreachable
                throw e;
            }
        } catch (NoFreeSpaceException e) {
            logger.info("CopyOperation.Handler handled exception", e);
            logger.info("failing");
            // todo: better check to see if this is a shared folder
            if (source.getUid().equals(destination.getUid())) {
                return Option.of(new Operation.ErrorData(507, 59, "NoFreeSpaceCopyToDisk"));
            } else {
                return Option.of(new Operation.ErrorData(507, 234, "OwnerHasNoFreeSpace"));
            }
        } catch (NoParentFolderException | ParentIsFileException e) {
            logger.info("CopyOperation.Handler handled exception", e);
            logger.info("failing");
            return Option.of(new Operation.ErrorData(500, 1, e.getMessage()));
        }
        return Option.empty();
    }

    private boolean isLocked(Operation operation, CopyOperationData data, DjfsResourcePath path) {
        String oid = operation.getId();
        if (lockManager.isLockedFor(oid, path)) {
            logger.info(path.getFullPath() + " is locked");
            data = data.toBuilder().error(Option.of(new Operation.ErrorData(423, 105, "ResourceLocked"))).build();
            operationDao.setData(operation.getUid(), oid, data, CopyOperationData.B, operation);
            return true;
        }
        return false;
    }

    private boolean tryAcquireOrRenewLock(Operation operation, CopyOperationData data, DjfsResourcePath path) {
        String oid = operation.getId();
        if (!lockManager.tryAcquireOrRenewLock(oid, oid, "copy_resource", path)) {
            logger.info(path.getFullPath() + " is locked");
            data = data.toBuilder().error(Option.of(new Operation.ErrorData(423, 105, "ResourceLocked"))).build();
            operationDao.setData(operation.getUid(), oid, data, CopyOperationData.B, operation);
            return false;
        }
        return true;
    }
}
