package ru.yandex.chemodan.app.djfs.core.filesystem.operation.postprocess;

import java.util.concurrent.ThreadLocalRandom;

import com.mongodb.ReadPreference;
import lombok.Value;
import org.joda.time.Duration;
import org.joda.time.Instant;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.chemodan.app.djfs.core.db.pg.TransactionUtils;
import ru.yandex.chemodan.app.djfs.core.filesystem.DjfsPrincipal;
import ru.yandex.chemodan.app.djfs.core.filesystem.DjfsResourceDao;
import ru.yandex.chemodan.app.djfs.core.filesystem.Filesystem;
import ru.yandex.chemodan.app.djfs.core.filesystem.QuotaManager;
import ru.yandex.chemodan.app.djfs.core.filesystem.iteration.ResourceIterator;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.DjfsResource;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.DjfsResourceArea;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.DjfsResourcePath;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.FileDjfsResource;
import ru.yandex.chemodan.app.djfs.core.tasks.DjfsTaskQueueName;
import ru.yandex.chemodan.app.djfs.core.user.DjfsUid;
import ru.yandex.chemodan.bazinga.YcridOnetimeTaskSupport;
import ru.yandex.chemodan.bazinga.YcridTaskParameters;
import ru.yandex.commune.bazinga.scheduler.ExecutionContext;
import ru.yandex.commune.bazinga.scheduler.TaskQueueName;
import ru.yandex.commune.bazinga.scheduler.schedule.CompoundReschedulePolicy;
import ru.yandex.commune.bazinga.scheduler.schedule.RescheduleConstant;
import ru.yandex.commune.bazinga.scheduler.schedule.ReschedulePolicy;
import ru.yandex.misc.bender.annotation.BenderBindAllFields;
import ru.yandex.misc.bender.annotation.BenderPart;
import ru.yandex.misc.bender.custom.AnyPojoWrapper;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;


public class QuickMovePostProcessTask extends YcridOnetimeTaskSupport<QuickMovePostProcessTask.Parameters> {
    private static final Logger logger = LoggerFactory.getLogger(QuickMovePostProcessTask.class);

    private final Filesystem filesystem;
    private final DjfsResourceDao djfsResourceDao;
    private final ResourceIterator.Factory resourceIteratorFactory;
    private final QuickMovePostProcessTaskProperties properties;
    private final QuotaManager quotaManager;
    private final TransactionUtils transactionUtils;
    private final QuickMovePostProcessTaskSubmitter quickMovePostProcessTaskSubmitter;

    private final MapF<DjfsResourceArea, AreaPostProcessor> areaPostProcessors;

    public QuickMovePostProcessTask(
            QuickMovePostProcessTaskProperties properties, ResourceIterator.Factory resourceIteratorFactory,
            Filesystem filesystem, DjfsResourceDao djfsResourceDao,
            QuotaManager quotaManager, TransactionUtils transactionUtils,
            QuickMovePostProcessTaskSubmitter quickMovePostProcessTaskSubmitter,
            TrashAreaPostProcessor trashAreaPostProcessor)
    {
        super(Parameters.class);

        this.resourceIteratorFactory = resourceIteratorFactory;
        this.filesystem = filesystem;
        this.djfsResourceDao = djfsResourceDao;
        this.properties = properties;
        this.quotaManager = quotaManager;
        this.transactionUtils = transactionUtils;
        this.quickMovePostProcessTaskSubmitter = quickMovePostProcessTaskSubmitter;

        this.areaPostProcessors = Cf.map(
                DjfsResourceArea.TRASH, trashAreaPostProcessor
        );
    }

    public QuickMovePostProcessTask(DjfsResourcePath path) {
        super(new Parameters(path.getUid().asString(), path.getPath(), path.getArea().value()));

        this.resourceIteratorFactory = null;
        this.filesystem = null;
        this.djfsResourceDao = null;
        this.properties = null;
        this.quotaManager = null;
        this.transactionUtils = null;
        this.quickMovePostProcessTaskSubmitter = null;
        this.areaPostProcessors = null;
    }

    static public String formatIndexKey(DjfsUid uid, DjfsResourceArea area) {
        return formatIndexKey(area.value(), uid.asString());
    }

    static private String formatIndexKey(String uid, String area) {
        return area + "_" + uid;
    }

    private Instant getRescheduleDate(int runningTasksCount) {
        long maxDelay = properties.getPerTaskDelay().multipliedBy(runningTasksCount).getStandardSeconds();
        long delay = ThreadLocalRandom.current().nextLong(maxDelay);

        return Instant.now()
                .plus(properties.getReschedulePeriod())
                .plus(Duration.standardSeconds(delay));
    }

    @Override
    public void doExecute(Parameters parameters, ExecutionContext context) {
        DjfsUid uid = DjfsUid.cons(parameters.uid);
        DjfsResourcePath targetResourcePath = DjfsResourcePath.cons(uid, parameters.getPath());
        DjfsResourceArea targetArea = targetResourcePath.getArea();

        int runningTasks = quickMovePostProcessTaskSubmitter.countRunningTasks(id(), uid, targetArea);
        if (runningTasks > 1) {
            quickMovePostProcessTaskSubmitter.schedule(targetResourcePath, getRescheduleDate(runningTasks));
            return;
        }

        if (!filesystem.find(DjfsPrincipal.cons(targetResourcePath.getUid()), targetResourcePath, Option.of(ReadPreference.primary())).isPresent()) {
            return;
        }

        ResourceIterator iterator = resourceIteratorFactory.create(targetResourcePath,
                ResourceIterator.TraversalType.TOP_DOWN);
        ResourceIterator.ResourceIteratorState state;

        Option<AnyPojoWrapper> executionInfo = context.getExecutionInfo().map(AnyPojoWrapper.consF());
        if (executionInfo.isPresent() && executionInfo.get().getPojo().isPresent()) {
            QuickMovePostProcessTaskData taskData = (QuickMovePostProcessTaskData)executionInfo.get().getPojo().get();
            if (taskData.getState().isPresent()) {
                state = taskData.getState().get();
            } else {
                state = iterator.initialize();
            }
        } else {
            state = iterator.initialize();
        }

        AreaPostProcessor areaPostProcessor = areaPostProcessors.getOrThrow(targetArea);

        while (iterator.hasNext(state)) {
            final ResourceIterator.ResourceIteratorState capturedState = state;

            Tuple2<ListF<DjfsResource>, ResourceIterator.ResourceIteratorState> resourcesWithState =
                    transactionUtils.executeInNewOrCurrentTransaction(uid, () -> {
                        Tuple2<ResourceIterator.ResourceIteratorState, ListF<DjfsResource>> next =
                                iterator.next(capturedState, properties.getInternalStateUpdateFilesCount(), true);

                        ListF<DjfsResource> unprocessedItems = getUnprocessedItems(next._2, targetArea);
                        if (unprocessedItems.isEmpty()) {
                            return Tuple2.tuple(unprocessedItems, next._1);
                        }

                        areaPostProcessor.processInsideTransaction(unprocessedItems);

                        long size = calculateSize(unprocessedItems);
                        quotaManager.incrementAreaCounter(uid, targetArea, size);

                        setItemsProcessed(unprocessedItems, targetArea);
                        context.saveExecutionInfoImmediately(capturedState);

                        return Tuple2.tuple(unprocessedItems, next._1);
                    });

            state = resourcesWithState.get2();
            areaPostProcessor.processAfterTransaction(resourcesWithState.get1());
        }
    }

    private ListF<DjfsResource> getUnprocessedItems(ListF<DjfsResource> resources, DjfsResourceArea targetArea) {
        return resources
                .filter(x -> x.getPath().getArea() == targetArea)
                .filter(x -> !x.getArea().isPresent() || x.getArea().get() != targetArea);
    }

    private void setItemsProcessed(ListF<DjfsResource> resources, DjfsResourceArea targetArea) {
        if (resources.isEmpty()) {
            return;
        }

        SetF<DjfsUid> uids = resources.map(DjfsResource::getUid).unique();
        if (uids.size() != 1) {
            throw new RuntimeException("Unexpected uids in quick move post process task");
        }

        djfsResourceDao.setArea(uids.single(), resources.map(DjfsResource::getId), targetArea);
    }

    private long calculateSize(ListF<DjfsResource> resources) {
        return resources
                .filterByType(FileDjfsResource.class)
                .map(FileDjfsResource::getSize)
                .reduceRightO(Long::sum).getOrElse(0L);
    }

    @Override
    public TaskQueueName queueName() {
        return DjfsTaskQueueName.FILESYSTEM_POST_PROCESS_TASKS;
    }

    @Override
    public int priority() {
        return 0;
    }

    @Override
    public Duration timeout() {
        return Duration.standardDays(1);
    }

    @Override
    public ReschedulePolicy reschedulePolicy() {
        return new CompoundReschedulePolicy(
                new RescheduleConstant(Duration.standardSeconds(10), 3),
                new RescheduleConstant(Duration.standardMinutes(5), 999)
        );
    }

    @Value
    @BenderBindAllFields
    static class Parameters extends YcridTaskParameters {
        @BenderPart(name = "index_key", strictName = true)
        private final String indexKey;

        private final String uid;
        private final String path;

        Parameters(String uid, String path, String area) {
            this.uid = uid;
            this.path = path;
            this.indexKey = formatIndexKey(uid, area);
        }
    }
}
