package ru.yandex.chemodan.app.djfs.core.filesystem.operation.postprocess;

import lombok.RequiredArgsConstructor;
import org.joda.time.Instant;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.chemodan.app.djfs.core.album.FileOperationPostProcessHandler;
import ru.yandex.chemodan.app.djfs.core.db.EntityAlreadyExistsException;
import ru.yandex.chemodan.app.djfs.core.filesystem.DjfsResourceDao;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.DjfsResource;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.FileDjfsResource;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.MediaType;
import ru.yandex.chemodan.app.djfs.core.globalgallery.DeletionLogDao;
import ru.yandex.chemodan.app.djfs.core.globalgallery.DeletionLogEntry;
import ru.yandex.chemodan.app.djfs.core.util.CeleryJobUtils;
import ru.yandex.chemodan.app.djfs.core.util.UuidUtils;
import ru.yandex.chemodan.queller.celery.job.CeleryJob;
import ru.yandex.chemodan.queller.worker.CeleryTaskManager;
import ru.yandex.commune.json.JsonArray;
import ru.yandex.commune.json.JsonString;
import ru.yandex.commune.json.JsonValue;


@RequiredArgsConstructor
public class TrashAreaPostProcessor implements AreaPostProcessor {

    private final DeletionLogDao deletionLogDao;
    private final DjfsResourceDao djfsResourceDao;

    private final CeleryTaskManager celeryTaskManager;

    @Override
    public void processInsideTransaction(ListF<DjfsResource> resources) {
        setPublishedAndRemovePublic(resources);
    }

    @Override
    public void processAfterTransaction(ListF<DjfsResource> resources) {
        addDeletionLogEntries(resources);
        addFileOperationPostProcessTasks(resources);
    }

    private void addFileOperationPostProcessTasks(ListF<DjfsResource> resources) {
        JsonArray resourceIds = new JsonArray(resources
                .filterMap(DjfsResource::getResourceId).map(resourceId -> JsonString.valueOf(resourceId.getValue())));
        MapF<String, JsonValue> kwargs = Cf.map("resource_ids", resourceIds);
        CeleryJob celeryJob = CeleryJobUtils.create(FileOperationPostProcessHandler.TASK_ID, kwargs);
        celeryTaskManager.submit(celeryJob);
    }

    private void addDeletionLogEntries(ListF<DjfsResource> resources) {
        ListF<FileDjfsResource> files = resources.filterByType(FileDjfsResource.class);
        final SetF<MediaType> typesToLog = Cf.set(MediaType.IMAGE, MediaType.VIDEO);

        ListF<FileDjfsResource> filesToLog = files
                .filter(x -> x.getMediaType().isPresent())
                .filter(x -> typesToLog.exists(y -> x.getMediaType().get() == y));

        if (filesToLog.isEmpty()) {
            return;
        }

        for (FileDjfsResource file : filesToLog) {
            DeletionLogEntry.DeletionLogEntryBuilder entryBuilder = DeletionLogEntry.builder()
                    .uid(file.getUid())
                    .fileId(file.getFileId().get())
                    .storageId(UuidUtils.fromHex(file.getHid()))
                    .isLivePhoto(file.isLivePhoto());

            boolean isInserted = false;
            DeletionLogEntry entry = entryBuilder.build();

            while (!isInserted) {
                try {
                    deletionLogDao.insert(entry);
                    isInserted = true;
                } catch (EntityAlreadyExistsException e) {
                    entry = entryBuilder.revision(Instant.now()).build();  // because we have primary key (uid, revision)
                }
            }
        }
    }

    private void setPublishedAndRemovePublic(ListF<DjfsResource> trashResources) {
        for (DjfsResource resource : trashResources) {
            djfsResourceDao.setPublishedAndRemovePublic(resource.getUid(), resource.getId());
        }
    }
}
