package ru.yandex.chemodan.app.djfs.core.filesystem.operation.trashappend;


import java.util.concurrent.atomic.AtomicBoolean;

import org.joda.time.Duration;

import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.djfs.core.DjfsException;
import ru.yandex.chemodan.app.djfs.core.SafeCloseable;
import ru.yandex.chemodan.app.djfs.core.client.OperationCallbackData;
import ru.yandex.chemodan.app.djfs.core.filesystem.DjfsPrincipal;
import ru.yandex.chemodan.app.djfs.core.filesystem.Filesystem;
import ru.yandex.chemodan.app.djfs.core.filesystem.OperationCallbackHandler;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.DjfsResource;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.DjfsResourcePath;
import ru.yandex.chemodan.app.djfs.core.filesystem.operation.move.MoveCallbacks;
import ru.yandex.chemodan.app.djfs.core.history.EventHistoryLogger;
import ru.yandex.chemodan.app.djfs.core.legacy.exception.LegacyMpfsException;
import ru.yandex.chemodan.app.djfs.core.legacy.exception.LegacyMpfsExceptionOrigin;
import ru.yandex.chemodan.app.djfs.core.legacy.web.LegacyMpfsExceptionHandler;
import ru.yandex.chemodan.app.djfs.core.operations.MpfsOperationHandler;
import ru.yandex.chemodan.app.djfs.core.operations.MpfsOperationHandlerContext;
import ru.yandex.chemodan.app.djfs.core.operations.Operation;
import ru.yandex.chemodan.app.djfs.core.user.DjfsUid;
import ru.yandex.chemodan.app.djfs.core.web.ConnectionIdHolder;
import ru.yandex.commune.bazinga.impl.TaskId;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;


public class TrashAppendOperationHandler  extends MpfsOperationHandler {
    private static final Logger logger = LoggerFactory.getLogger(TrashAppendOperationHandler.class);

    private final Filesystem filesystem;
    private final EventHistoryLogger eventHistoryLogger;
    private final OperationCallbackHandler operationCallbackHandler;
    private final LegacyMpfsExceptionHandler legacyMpfsExceptionHandler;

    public TrashAppendOperationHandler(MpfsOperationHandlerContext mpfsOperationHandlerContext, Filesystem filesystem,
            EventHistoryLogger eventHistoryLogger, OperationCallbackHandler operationCallbackHandler,
            LegacyMpfsExceptionHandler legacyMpfsExceptionHandler)
    {
        super(mpfsOperationHandlerContext);

        this.filesystem = filesystem;
        this.eventHistoryLogger = eventHistoryLogger;
        this.operationCallbackHandler = operationCallbackHandler;
        this.legacyMpfsExceptionHandler = legacyMpfsExceptionHandler;
    }

    @Override
    public Status handle(Operation operation, AtomicBoolean terminated) {
        DjfsUid uid = operation.getUid();
        TrashAppendOperationData data = operation.getData(TrashAppendOperationData.B);

        DjfsResourcePath path = DjfsResourcePath.cons(data.getPath());
        String callback = data.getCallback();
        long userDiskVersion = data.getAtVersion();

        try (SafeCloseable ignored = ConnectionIdHolder.set(data.getConnectionId())) {
            Option<DjfsResource> removedResource = Option.empty();
            Option<Operation.ErrorData> error = Option.empty();

            try {
                DjfsResource trashResource = filesystem.trashAppendResource(DjfsPrincipal.cons(uid), path,
                        MoveCallbacks.defaultWithLogging(eventHistoryLogger));
                removedResource = Option.of(trashResource);
            } catch (DjfsException e) {
                Option<LegacyMpfsException> translatedExceptionO =
                        legacyMpfsExceptionHandler.translate(e, uid, LegacyMpfsExceptionOrigin.MOVE);
                if (translatedExceptionO.isPresent()) {
                    LegacyMpfsException legacyMpfsException = translatedExceptionO.get();
                    error = Option.of(new Operation.ErrorData(
                            legacyMpfsException.getHttpStatusCode().getStatusCode(), legacyMpfsException.getCode(),
                            legacyMpfsException.getPythonClassName()));
                } else {
                    error = Option.of(new Operation.ErrorData(500, 1, e.getMessage()));
                }
            } catch (Exception e) {
                error = Option.of(new Operation.ErrorData(500, 1, e.getMessage()));
            }

            if (error.isPresent()) {
                data = data.toBuilder().error(error).build();

                operationDao.setData(uid, operation.getId(), data, TrashAppendOperationData.B, operation);
                submitCallbackTask(callback, path, userDiskVersion, false);

                return Status.FAIL;
            }

            TrashAppendOperationData newData = data.toBuilder()
                    .affectedResource(removedResource.map(DjfsResource::getPath).map(DjfsResourcePath::getPath))
                    .build();
            operationDao.setData(uid, operation.getId(), newData, TrashAppendOperationData.B, operation);

            submitCallbackTask(callback, path, userDiskVersion, true);

            // TODO: recreate yateam folder if it was moved

            return Status.DONE;
        }
    }

    private void submitCallbackTask(String callbackUrl, DjfsResourcePath path, long userDiskVersion, boolean success) {
        if (callbackUrl.isEmpty()) {
            return;
        }

        String state = success ? Operation.State.COMPLETED.name() : Operation.State.FAILED.name();
        String status = success ? "DONE" : "FAILED";

        OperationCallbackData callbackData = OperationCallbackData.builder()
                .state(state)
                .status(status)
                .atVersion(userDiskVersion)
                .params(OperationCallbackData.Params.builder()
                        .path(Option.of(path.getFullPath()))
                        .build())
                .type("trash")
                .build();

        operationCallbackHandler.submitCallbackTask(callbackUrl, callbackData);
    }

    @Override
    protected TaskId celeryTaskId() {
        return TrashAppendOperation.TASK_ID;
    }

    @Override
    public Duration timeout() {
        return Duration.standardMinutes(1);
    }
}
