package ru.yandex.chemodan.app.djfs.core.globalgallery;

import java.util.Map;
import java.util.Objects;

import org.postgresql.util.PSQLException;
import org.postgresql.util.ServerErrorMessage;
import org.springframework.dao.DataIntegrityViolationException;
import org.springframework.jdbc.core.RowMapper;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.chemodan.app.djfs.core.db.EntityAlreadyExistsException;
import ru.yandex.chemodan.app.djfs.core.db.pg.PgShardedDao;
import ru.yandex.chemodan.app.djfs.core.db.pg.PgShardedDaoContext;
import ru.yandex.chemodan.app.djfs.core.db.pg.ResultSetUtils;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.DjfsFileId;
import ru.yandex.chemodan.app.djfs.core.user.DjfsUid;
import ru.yandex.chemodan.app.djfs.core.util.InstantUtils;
import ru.yandex.chemodan.app.djfs.core.util.UuidUtils;
import ru.yandex.misc.codec.Hex;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;


public class PgDeletionLogDao extends PgShardedDao implements DeletionLogDao {
    private static final Logger logger = LoggerFactory.getLogger(PgDeletionLogDao.class);

    private static final RowMapper<DeletionLogEntry> M = (rs, rowNum) -> DeletionLogEntry.builder()
            .uid(DjfsUid.cons(rs.getLong("uid")))
            .fileId(DjfsFileId.cons(Hex.encode(rs.getBytes("file_id"))))
            .storageId(UuidUtils.from(rs.getString("storage_id")))
            .revision(ResultSetUtils.getInstant(rs, "deletion_log_revision"))
            .isLivePhoto(rs.getBoolean("is_live_photo"))
            .build();

    public PgDeletionLogDao(PgShardedDaoContext dependencies) {
        super(dependencies);
    }

    @Override
    public void insert(DeletionLogEntry entry) {
        String sql = collectStats(entry)
                + " INSERT INTO disk.deletion_log (uid, file_id, storage_id, deletion_log_revision, is_live_photo) "
                + " VALUES (:uid, :file_id, :storage_id, :deletion_log_revision, :is_live_photo)";

        Map<String, Object> parameters = Cf.toMap(Tuple2List.fromPairs(
                "uid", entry.getUid(),
                "file_id", Hex.decode(entry.getFileId().getValue()),
                "storage_id", entry.getStorageId(),
                "deletion_log_revision", entry.getRevision(),
                "is_live_photo", entry.isLivePhoto()
        ));

        try {
            jdbcTemplate(entry).update(sql, parameters);
        } catch (DataIntegrityViolationException e) {
            Throwable cause = e.getCause();
            if (cause instanceof PSQLException) {
                ServerErrorMessage error = ((PSQLException) cause).getServerErrorMessage();
                if (Objects.equals(error.getConstraint(), "pk_deletion_log")) {
                    long revision = InstantUtils.toVersion(entry.getRevision());
                    logger.warn("PgDeletionLogDao.insert(DeletionLogEntry) handled exception for uid "
                            + entry.getUid().asString() + " deletion_log_revision " + revision + " : ", e);
                    throw new EntityAlreadyExistsException("deletion_log_revision=" + revision + " uid="
                            + entry.getUid().asString(), e);
                }
            }
            throw e;
        }
    }

    @Override
    public ListF<DeletionLogEntry> findAll(DjfsUid uid) {
        String sql = collectStats(uid) +
                " SELECT * FROM disk.deletion_log WHERE uid=:uid ORDER BY deletion_log_revision DESC";
        return jdbcTemplate(uid).query(sql, M, Cf.map("uid", uid));
    }
}
